## PDC AUTH

PDC Authentication Manager

## Installation
Installation: To install from PyPI use:

`$ python -m pip install pdc-auth`

After installation, import with something like:

```python
import pdc_auth as pdca
```

## Configure Endpoint
Before using authenticator configure the endpoint first. Enpoint by default has url "http://localhost:4000" and path "/v2/login". If method `get_host() `called will return "http://localhost:4000/v2/login". Configure to get a custom host.

```python
from pdc_auth.endpoint import EndpointConfig, configure_endpoint


url = "www.myendpoint.com"
path = "/v1/auth"
configure_endpoint(url, path)

endpoint = EndpointConfig()
endpoint.get_host() # www.myendpoint.com/v1/auth
```

## Create Config File
By default the configuration file path is "data/config.json". Path can be customized as desired by adding parameters to the authenticator later. However the configuration file must be `json` and include fields like below.

```json
{
    "lisensi": {
        "email": "myemail@custom.com",
        "pwd": "secret"
    }
}
```

## Authenticator
Authenticator is used to check the login to the endpoint according to the previous configuration. With the appropriate email and password listed in the configuration. When the `login()` function is called checking for an error in the login returns an error. If the login is successful it will return the value `True`.

```python
import pdc_auth.authenticator as Authenticator
from pdc_auth.exceptions.login_provider_exc import LoginProviderFailedException

config_fname="config/lisensi.json"
authenticator = Authenticator(config_fname=config_fname) # config_fname by default is: "data/config.json"

try:
    authenticator.login() 
except LoginProviderFailedException as e:
    pass
```

## Custom Login Provider
Customizing the provider on the login authenticator.
```python
import pdc_auth.authenticator as Authenticator
from pdc_auth.exceptions.login_provider_exc import LoginProviderFailedException
from pdc_auth.login_provider import LoginProvider

provider = LoginProvider()
authenticator = Authenticator(provider=provider)

try:
    authenticator.login() 
except LoginProviderFailedException as e:
    pass
```

Some of the data that can be customized on the provider are as follows:

1. __Custom Bot__
```python
from pdc_auth.login_provider import LoginProvider

bot_id = 10
version = '3.0.0'
provider = LoginProvider(bot_id=bot_id, version=version)
```
or
```python
from pdc_auth.login_provider import LoginProvider

bot_id = 10
version = '3.0.0'
latest_version = '3.0.18'
provider = LoginProvider()
provider.update_bot(bot_id=bot_id, version=version, latest_version=latest_version)
```

2. __Custom Headers__
```python
from pdc_auth.login_provider import LoginProvider

custom_headers = { "X-Secret-Key": "secret" }
provider = LoginProvider()
provider.update_headers(custom_headers)
```