"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = void 0;
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    /**
     * The full ARN of the secret in AWS Secrets Manager, which is the ARN including the Secrets Manager-supplied 6-character suffix.
     *
     * This is equal to `secretArn` in most cases, but is undefined when a full ARN is not available (e.g., secrets imported by name).
     */
    get secretFullArn() { return this.secretArn; }
    /**
     * Grants reading the secret value to some role.
     */
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    /**
     * Grants writing and updating the secret value to some role.
     */
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            scope: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    /**
     * Retrieve the value of the stored secret as a `SecretValue`.
     */
    get secretValue() {
        return this.secretValueFromJson('');
    }
    /**
     * Interpret the secret as a JSON object and return a field's value from it as a `SecretValue`.
     */
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    /**
     * Adds a rotation schedule to the secret.
     */
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this secret.
     *
     * If this secret was created in this stack, a resource policy will be
     * automatically created upon the first call to `addToResourcePolicy`. If
     * the secret is imported, then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     */
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Denies the `DeleteSecret` action to all principals within the current account.
     */
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies.
     *
     * Typically, this is just the secret ARN.
     * However, secrets imported by name require a different format.
     */
    get arnForPolicies() { return this.secretArn; }
    /**
     * Attach a target to this secret.
     *
     * @param target The target to attach.
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        /**
         *
         */
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        this.secretName = parseSecretName(this, this.secretArn);
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principal);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    /**
     * @deprecated use `fromSecretCompleteArn` or `fromSecretPartialArn`
     */
    static fromSecretArn(scope, id, secretArn) {
        const attrs = arnIsComplete(secretArn) ? { secretCompleteArn: secretArn } : { secretPartialArn: secretArn };
        return Secret.fromSecretAttributes(scope, id, attrs);
    }
    /**
     * Imports a secret by complete ARN.
     *
     * The complete ARN is the ARN with the Secrets Manager-supplied suffix.
     */
    static fromSecretCompleteArn(scope, id, secretCompleteArn) {
        return Secret.fromSecretAttributes(scope, id, { secretCompleteArn });
    }
    /**
     * Imports a secret by partial ARN.
     *
     * The partial ARN is the ARN without the Secrets Manager-supplied suffix.
     */
    static fromSecretPartialArn(scope, id, secretPartialArn) {
        return Secret.fromSecretAttributes(scope, id, { secretPartialArn });
    }
    /**
     * (deprecated) Imports a secret by secret name;
     *
     * the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     *
     * @deprecated use `fromSecretNameV2`
     */
    static fromSecretName(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    sep: ':',
                });
            }
        }(scope, id);
    }
    /**
     * Imports a secret by secret name.
     *
     * A secret with this name must exist in the same account & region.
     * Replaces the deprecated `fromSecretName`.
     */
    static fromSecretNameV2(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretName = secretName;
                this.secretArn = this.partialArn;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return this.partialArn + '-??????';
            }
            // Creates a "partial" ARN from the secret name. The "full" ARN would include the SecretsManager-provided suffix.
            get partialArn() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: secretName,
                    sep: ':',
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        let secretArn;
        let secretArnIsPartial;
        if (attrs.secretArn) {
            if (attrs.secretCompleteArn || attrs.secretPartialArn) {
                throw new Error('cannot use `secretArn` with `secretCompleteArn` or `secretPartialArn`');
            }
            secretArn = attrs.secretArn;
            secretArnIsPartial = false;
        }
        else {
            if ((attrs.secretCompleteArn && attrs.secretPartialArn) ||
                (!attrs.secretCompleteArn && !attrs.secretPartialArn)) {
                throw new Error('must use only one of `secretCompleteArn` or `secretPartialArn`');
            }
            if (attrs.secretCompleteArn && !arnIsComplete(attrs.secretCompleteArn)) {
                throw new Error('`secretCompleteArn` does not appear to be complete; missing 6-character suffix');
            }
            [secretArn, secretArnIsPartial] = attrs.secretCompleteArn ? [attrs.secretCompleteArn, false] : [attrs.secretPartialArn, true];
        }
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = secretArn;
                this.secretName = parseSecretName(scope, secretArn);
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return secretArnIsPartial ? undefined : secretArn; }
        }(scope, id);
    }
    /**
     * (deprecated) Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         *
         */
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        this.secretName = props.secret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    /**
     *
     */
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).parseArn(secretArn, ':').resourceName;
    if (resourceName) {
        // Can't operate on the token to remove the SecretsManager suffix, so just return the full secret name
        if (core_1.Token.isUnresolved(resourceName)) {
            return resourceName;
        }
        // Secret resource names are in the format `${secretName}-${6-character SecretsManager suffix}`
        // If there is no hyphen (or 6-character suffix) assume no suffix was provided, and return the whole name.
        const lastHyphenIndex = resourceName.lastIndexOf('-');
        const hasSecretsSuffix = lastHyphenIndex !== -1 && resourceName.substr(lastHyphenIndex + 1).length === 6;
        return hasSecretsSuffix ? resourceName.substr(0, lastHyphenIndex) : resourceName;
    }
    throw new Error('invalid ARN format; no secret name provided');
}
/** Performs a best guess if an ARN is complete, based on if it ends with a 6-character suffix. */
function arnIsComplete(secretArn) {
    return core_1.Token.isUnresolved(secretArn) || /-[a-z0-9]{6}$/i.test(secretArn);
}
//# sourceMappingURL=data:application/json;base64,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