"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? true;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-22.AbstractPod", version: "2.0.0-rc.110" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_b = JSII_RTTI_SYMBOL_1;
LabelSelector[_b] = { fqn: "cdk8s-plus-22.LabelSelector", version: "2.0.0-rc.110" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-22.Pod", version: "2.0.0-rc.110" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_d = JSII_RTTI_SYMBOL_1;
PodDns[_d] = { fqn: "cdk8s-plus-22.PodDns", version: "2.0.0-rc.110" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_e = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_e] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "2.0.0-rc.110" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, 'DoesNotExist', undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, 'Gt', values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, 'Lt', values);
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_f = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_f] = { fqn: "cdk8s-plus-22.NodeLabelQuery", version: "2.0.0-rc.110" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_g = JSII_RTTI_SYMBOL_1;
LabelExpression[_g] = { fqn: "cdk8s-plus-22.LabelExpression", version: "2.0.0-rc.110" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery('Equal', key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery('Exists', key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery('Exists');
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_h = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_h] = { fqn: "cdk8s-plus-22.NodeTaintQuery", version: "2.0.0-rc.110" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_j = JSII_RTTI_SYMBOL_1;
Pods[_j] = { fqn: "cdk8s-plus-22.Pods", version: "2.0.0-rc.110" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_k = JSII_RTTI_SYMBOL_1;
LabeledNode[_k] = { fqn: "cdk8s-plus-22.LabeledNode", version: "2.0.0-rc.110" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_l = JSII_RTTI_SYMBOL_1;
TaintedNode[_l] = { fqn: "cdk8s-plus-22.TaintedNode", version: "2.0.0-rc.110" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_m = JSII_RTTI_SYMBOL_1;
NamedNode[_m] = { fqn: "cdk8s-plus-22.NamedNode", version: "2.0.0-rc.110" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_o = JSII_RTTI_SYMBOL_1;
Node[_o] = { fqn: "cdk8s-plus-22.Node", version: "2.0.0-rc.110" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_p = JSII_RTTI_SYMBOL_1;
Topology[_p] = { fqn: "cdk8s-plus-22.Topology", version: "2.0.0-rc.110" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push({
                key: query.key,
                value: query.value,
                effect: query.effect,
                operator: query.operator,
                tolerationSeconds: query.evictAfter?.toSeconds(),
            });
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => ({ key: s.key, operator: s.operator, values: s.values })) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_q = JSII_RTTI_SYMBOL_1;
PodScheduling[_q] = { fqn: "cdk8s-plus-22.PodScheduling", version: "2.0.0-rc.110" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n.number));
    }
}
exports.PodConnections = PodConnections;
_r = JSII_RTTI_SYMBOL_1;
PodConnections[_r] = { fqn: "cdk8s-plus-22.PodConnections", version: "2.0.0-rc.110" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicG9kLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3BvZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLGlDQUFzRjtBQUN0RiwyQ0FBbUQ7QUFDbkQsK0JBQStCO0FBQy9CLHlDQUF5QztBQUN6QyxxQ0FBcUM7QUFFckMsa0RBQWtEO0FBSWxELG1DQUFvRDtBQUdwRCxNQUFzQixXQUFZLFNBQVEsSUFBSSxDQUFDLFFBQVE7SUFnQnJELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBMEIsRUFBRTtRQUNwRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBUkYsZ0JBQVcsR0FBMEIsRUFBRSxDQUFDO1FBQ3hDLG9CQUFlLEdBQTBCLEVBQUUsQ0FBQztRQUM1QyxpQkFBWSxHQUFnQixFQUFFLENBQUM7UUFDL0IsYUFBUSxHQUErQixJQUFJLEdBQUcsRUFBRSxDQUFDO1FBT2hFLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUN6QyxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7UUFDM0MsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUNyRSxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNqQyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO1FBQ25ELElBQUksQ0FBQyw0QkFBNEIsR0FBRyxLQUFLLENBQUMsNEJBQTRCLElBQUksSUFBSSxDQUFDO1FBRS9FLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNwQixLQUFLLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNyRDtRQUVELElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRTtZQUNqQixLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMvQztRQUVELElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUN4QixLQUFLLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQzdEO1FBRUQsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ3JCLEtBQUssQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3REO0lBRUgsQ0FBQztJQUVELElBQVcsVUFBVTtRQUNuQixPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVELElBQVcsT0FBTztRQUNoQixPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRCxJQUFXLFdBQVc7UUFDcEIsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7T0FFRztJQUNJLG1CQUFtQjtRQUN4QixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDaEUsSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNmLHVGQUF1RjtZQUN2RixNQUFNLElBQUksS0FBSyxDQUFDLDJDQUEyQyxHQUFHLENBQUMsYUFBYSxtQkFBbUIsQ0FBQyxDQUFDO1NBQ2xHO1FBQ0QsT0FBTztZQUNMLGFBQWEsRUFBRSxhQUFhLENBQUMsRUFBRSxDQUFDLEVBQUUsTUFBTSxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsVUFBVSxFQUFFLEVBQUUsQ0FBQztZQUNoRixVQUFVLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO2dCQUNwQyxLQUFLLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQzthQUNqQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ2QsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLENBQUM7SUFDckQsQ0FBQztJQUVEOztPQUVHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFTSxZQUFZLENBQUMsSUFBOEI7UUFDaEQsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVNLGdCQUFnQixDQUFDLElBQThCO1FBRXBELDBHQUEwRztRQUMxRyxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO1NBQ3BFO1FBRUQsSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELENBQUMsQ0FBQztTQUNuRTtRQUVELElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNoQixNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxDQUFDLENBQUM7U0FDbEU7UUFFRCxNQUFNLElBQUksR0FBRyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUM7WUFDbkMsR0FBRyxJQUFJO1lBQ1AsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLElBQUksUUFBUSxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sRUFBRTtTQUN6RCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFTSxZQUFZLENBQUMsU0FBb0I7UUFDdEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVNLFNBQVMsQ0FBQyxHQUFrQjtRQUNqQyxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbkQsSUFBSSxjQUFjLEVBQUU7WUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsR0FBRyxDQUFDLElBQUksaUJBQWlCLENBQUMsQ0FBQztTQUNoRTtRQUNELElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksc0JBQXNCO1FBRTNCLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxJQUFJLENBQUMsSUFBSSxDQUFDLDRCQUE0QixFQUFFO1lBQzlELE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxJQUFJLENBQUMsSUFBSSxpREFBaUQ7a0JBQ3pFLGdHQUFnRyxDQUFDLENBQUM7U0FDdkc7UUFFRCxxRUFBcUU7UUFDckUsa0JBQWtCO1FBQ2xCLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLGNBQWMsRUFBRSxJQUFJLElBQUksU0FBUyxDQUFDO1FBRWxFLE9BQU87WUFDTCxJQUFJLEVBQUUsZ0JBQWdCO1lBQ3RCLElBQUksRUFBRSxrQkFBa0I7WUFDeEIsUUFBUSxFQUFFLEVBQUU7U0FDYixDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0ksVUFBVTtRQUVmLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ2hDLE1BQU0sSUFBSSxLQUFLLENBQUMsd0NBQXdDLENBQUMsQ0FBQztTQUMzRDtRQUVELE1BQU0sT0FBTyxHQUErQixJQUFJLEdBQUcsRUFBRSxDQUFDO1FBQ3RELE1BQU0sVUFBVSxHQUFvQixFQUFFLENBQUM7UUFDdkMsTUFBTSxjQUFjLEdBQW9CLEVBQUUsQ0FBQztRQUUzQyxLQUFLLE1BQU0sSUFBSSxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDbEMsb0RBQW9EO1lBQ3BELHVEQUF1RDtZQUN2RCxLQUFLLE1BQU0sS0FBSyxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7Z0JBQy9CLFNBQVMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7YUFDekI7WUFDRCxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQ2pDO1FBRUQsS0FBSyxNQUFNLElBQUksSUFBSSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3RDLG9EQUFvRDtZQUNwRCx1REFBdUQ7WUFDdkQsS0FBSyxNQUFNLEtBQUssSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO2dCQUMvQixTQUFTLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2FBQ3pCO1lBQ0QsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUNyQztRQUVELEtBQUssTUFBTSxHQUFHLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUM5QixTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDaEI7UUFFRCxTQUFTLFNBQVMsQ0FBQyxHQUFrQjtZQUNuQyxNQUFNLGNBQWMsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM3QywyRUFBMkU7WUFDM0UsMERBQTBEO1lBQzFELElBQUksY0FBYyxJQUFJLGNBQWMsS0FBSyxHQUFHLEVBQUU7Z0JBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO2FBQ2hIO1lBQ0QsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQzdCLENBQUM7UUFFRCxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBRS9CLE9BQU87WUFDTCxhQUFhLEVBQUUsSUFBSSxDQUFDLGFBQWE7WUFDakMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxJQUFJO1lBQzdDLFVBQVUsRUFBRSxVQUFVO1lBQ3RCLGVBQWUsRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQ2pFLGNBQWMsRUFBRSx3QkFBZ0IsQ0FBQyxjQUFjLENBQUM7WUFDaEQsV0FBVyxFQUFFLHdCQUFnQixDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7WUFDL0MsT0FBTyxFQUFFLHdCQUFnQixDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFDN0UsU0FBUyxFQUFFLEdBQUcsQ0FBQyxNQUFNO1lBQ3JCLFNBQVMsRUFBRSx3QkFBZ0IsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDO1lBQ3ZDLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUTtZQUN0QixTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7WUFDeEIsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLGNBQWM7WUFDckMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2hHLDRCQUE0QixFQUFFLElBQUksQ0FBQyw0QkFBNEI7U0FDaEUsQ0FBQztJQUVKLENBQUM7O0FBMU5ILGtDQTROQzs7O0FBNk1EOztHQUVHO0FBQ0gsTUFBYSxhQUFhO0lBTXhCLFlBQ21CLFdBQThCLEVBQzlCLE1BQWlDO1FBRGpDLGdCQUFXLEdBQVgsV0FBVyxDQUFtQjtRQUM5QixXQUFNLEdBQU4sTUFBTSxDQUEyQjtJQUFHLENBQUM7SUFOakQsTUFBTSxDQUFDLEVBQUUsQ0FBQyxVQUFnQyxFQUFFO1FBQ2pELE9BQU8sSUFBSSxhQUFhLENBQUMsT0FBTyxDQUFDLFdBQVcsSUFBSSxFQUFFLEVBQUUsT0FBTyxDQUFDLE1BQU0sSUFBSSxFQUFFLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBTU0sT0FBTztRQUNaLE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTztRQUNaLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ2xCLE9BQU8sRUFBRSxDQUFDO1NBQ1g7UUFDRCxPQUFPO1lBQ0wsZ0JBQWdCLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxHQUFHLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQyxRQUFRLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDdkgsV0FBVyxFQUFFLHdCQUFnQixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7U0FDM0MsQ0FBQztJQUNKLENBQUM7O0FBekJILHNDQTBCQzs7O0FBNkJEOzs7R0FHRztBQUNILE1BQWEsR0FBSSxTQUFRLFdBQVc7SUFrQmxDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBa0IsRUFBRTtRQUM1RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQU5WLGlCQUFZLEdBQUcsTUFBTSxDQUFDO1FBUXBDLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDakQsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLElBQUksRUFBRSxZQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO1NBQ2xELENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxhQUFhLEVBQUUsYUFBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBRXBFLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDMUMsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQsSUFBVyxXQUFXO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUN2QixDQUFDO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUM3QyxPQUFPO1lBQ0wsR0FBRyxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ3BCLFFBQVEsRUFBRSxVQUFVLENBQUMsUUFBUTtZQUM3QixRQUFRLEVBQUUsVUFBVSxDQUFDLFFBQVE7WUFDN0IsV0FBVyxFQUFFLFVBQVUsQ0FBQyxXQUFXO1NBQ3BDLENBQUM7SUFDSixDQUFDOztBQS9DSCxrQkFpREM7OztBQS9DQzs7O0dBR0c7QUFDb0IsaUJBQWEsR0FBRyx3QkFBd0IsQ0FBQztBQWdIbEU7O0dBRUc7QUFDSCxNQUFhLE1BQU07SUEwQmpCLFlBQVksUUFBcUIsRUFBRTtRQUNqQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sSUFBSSxTQUFTLENBQUMsYUFBYSxDQUFDO1FBQ3RELElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUM7UUFDcEQsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLEVBQUUsQ0FBQztRQUM1QyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksRUFBRSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUM7SUFDdEMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLFFBQVE7UUFDakIsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsT0FBTztRQUNoQixPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksYUFBYSxDQUFDLEdBQUcsV0FBcUI7UUFDM0MsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxXQUFXLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxTQUFTLENBQUMsR0FBRyxRQUFrQjtRQUNwQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7T0FFRztJQUNJLFNBQVMsQ0FBQyxHQUFHLE9BQW9CO1FBQ3RDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsT0FBTyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTztRQUVaLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxTQUFTLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNuRSxNQUFNLElBQUksS0FBSyxDQUFDLHFFQUFxRSxDQUFDLENBQUM7U0FDeEY7UUFFRCxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUMvQixNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxDQUFDLENBQUM7U0FDakU7UUFFRCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxDQUFDLENBQUM7U0FDcEU7UUFFRCxPQUFPO1lBQ0wsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ3ZCLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztZQUN6QixjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7WUFDbkMsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO1lBQ25CLE1BQU0sRUFBRTtnQkFDTixXQUFXLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQztnQkFDL0MsUUFBUSxFQUFFLHdCQUFnQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7Z0JBQ3pDLE9BQU8sRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO2FBQ3hDO1NBQ0YsQ0FBQztJQUNKLENBQUM7O0FBMUdILHdCQTRHQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGtCQUFrQjtJQVU3QixZQUFZLFFBQWlDLEVBQUU7UUFGOUIsYUFBUSxHQUFhLEVBQUUsQ0FBQztRQUd2QyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLElBQUksS0FBSyxDQUFDO1FBQ2xELElBQUksQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUMsbUJBQW1CLElBQUksbUJBQW1CLENBQUMsTUFBTSxDQUFDO1FBQ25GLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztRQUN2QixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUM7UUFDekIsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBRTdCLEtBQUssTUFBTSxNQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sSUFBSSxFQUFFLEVBQUU7WUFDeEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDNUI7SUFFSCxDQUFDO0lBRUQsSUFBVyxPQUFPO1FBQ2hCLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUM1QixDQUFDO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osT0FBTztZQUNMLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSztZQUN0QixTQUFTLEVBQUUsSUFBSSxDQUFDLElBQUk7WUFDcEIsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPO1lBQ3JCLFlBQVksRUFBRSxJQUFJLENBQUMsYUFBYTtZQUNoQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsbUJBQW1CO1lBQzdDLE9BQU8sRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1NBQ3pDLENBQUM7SUFDSixDQUFDOztBQXZDSCxnREF5Q0M7OztBQUVEOztHQUVHO0FBQ0gsSUFBWSxhQWVYO0FBZkQsV0FBWSxhQUFhO0lBQ3ZCOztPQUVHO0lBQ0gsa0NBQWlCLENBQUE7SUFFakI7O09BRUc7SUFDSCx5Q0FBd0IsQ0FBQTtJQUV4Qjs7T0FFRztJQUNILGdDQUFlLENBQUE7QUFDakIsQ0FBQyxFQWZXLGFBQWEsR0FBYixxQkFBYSxLQUFiLHFCQUFhLFFBZXhCO0FBRUQsSUFBWSxtQkFhWDtBQWJELFdBQVksbUJBQW1CO0lBRTdCOzs7O09BSUc7SUFDSCwwREFBbUMsQ0FBQTtJQUVuQzs7T0FFRztJQUNILHdDQUFpQixDQUFBO0FBQ25CLENBQUMsRUFiVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQWE5QjtBQW9CRDs7R0FFRztBQUNILElBQVksU0E2Qlg7QUE3QkQsV0FBWSxTQUFTO0lBRW5COzs7OztPQUtHO0lBQ0gsMkNBQThCLENBQUE7SUFFOUI7OztPQUdHO0lBQ0gsb0VBQXVELENBQUE7SUFFdkQ7OztPQUdHO0lBQ0gsZ0NBQW1CLENBQUE7SUFFbkI7Ozs7T0FJRztJQUNILDBCQUFhLENBQUE7QUFFZixDQUFDLEVBN0JXLFNBQVMsR0FBVCxpQkFBUyxLQUFULGlCQUFTLFFBNkJwQjtBQWtCRDs7R0FFRztBQUNILE1BQWEsY0FBYztJQW1EekIsWUFDa0IsR0FBVyxFQUNYLFFBQWdCLEVBQ2hCLE1BQWlCO1FBRmpCLFFBQUcsR0FBSCxHQUFHLENBQVE7UUFDWCxhQUFRLEdBQVIsUUFBUSxDQUFRO1FBQ2hCLFdBQU0sR0FBTixNQUFNLENBQVc7SUFDbkMsQ0FBQztJQXJERDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxFQUFFLENBQUMsR0FBVyxFQUFFLEtBQWE7UUFDekMsT0FBTyxjQUFjLENBQUMsRUFBRSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxHQUFXLEVBQUUsTUFBZ0I7UUFDNUMsT0FBTyxJQUFJLGNBQWMsQ0FBQyxHQUFHLEVBQUUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBVyxFQUFFLE1BQWdCO1FBQy9DLE9BQU8sSUFBSSxjQUFjLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQVc7UUFDOUIsT0FBTyxJQUFJLGNBQWMsQ0FBQyxHQUFHLEVBQUUsUUFBUSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBVztRQUNwQyxPQUFPLElBQUksY0FBYyxDQUFDLEdBQUcsRUFBRSxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxHQUFXLEVBQUUsTUFBZ0I7UUFDNUMsT0FBTyxJQUFJLGNBQWMsQ0FBQyxHQUFHLEVBQUUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxFQUFFLENBQUMsR0FBVyxFQUFFLE1BQWdCO1FBQzVDLE9BQU8sSUFBSSxjQUFjLENBQUMsR0FBRyxFQUFFLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztJQUMvQyxDQUFDOztBQWpESCx3Q0F3REM7OztBQUVEOztHQUVHO0FBQ0gsTUFBYSxlQUFlO0lBOEIxQixZQUNrQixHQUFXLEVBQ1gsUUFBZ0IsRUFDaEIsTUFBaUI7UUFGakIsUUFBRyxHQUFILEdBQUcsQ0FBUTtRQUNYLGFBQVEsR0FBUixRQUFRLENBQVE7UUFDaEIsV0FBTSxHQUFOLE1BQU0sQ0FBVztJQUNuQyxDQUFDO0lBaENEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxHQUFXLEVBQUUsTUFBZ0I7UUFDNUMsT0FBTyxJQUFJLGVBQWUsQ0FBQyxHQUFHLEVBQUUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBVyxFQUFFLE1BQWdCO1FBQy9DLE9BQU8sSUFBSSxlQUFlLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQVc7UUFDOUIsT0FBTyxJQUFJLGVBQWUsQ0FBQyxHQUFHLEVBQUUsUUFBUSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBVztRQUNwQyxPQUFPLElBQUksZUFBZSxDQUFDLEdBQUcsRUFBRSxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDN0QsQ0FBQzs7QUE1QkgsMENBb0NDOzs7QUFFRDs7R0FFRztBQUNILElBQVksV0F1Qlg7QUF2QkQsV0FBWSxXQUFXO0lBQ3JCOzs7T0FHRztJQUNILHlDQUEwQixDQUFBO0lBRTFCOzs7O09BSUc7SUFDSCxzREFBdUMsQ0FBQTtJQUV2Qzs7Ozs7OztPQU9HO0lBQ0gsdUNBQXdCLENBQUE7QUFDMUIsQ0FBQyxFQXZCVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQXVCdEI7QUFzQkQ7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUF1QnpCLFlBQ2tCLFFBQWdCLEVBQ2hCLEdBQVksRUFDWixLQUFjLEVBQ2QsTUFBZSxFQUNmLFVBQXFCO1FBSnJCLGFBQVEsR0FBUixRQUFRLENBQVE7UUFDaEIsUUFBRyxHQUFILEdBQUcsQ0FBUztRQUNaLFVBQUssR0FBTCxLQUFLLENBQVM7UUFDZCxXQUFNLEdBQU4sTUFBTSxDQUFTO1FBQ2YsZUFBVSxHQUFWLFVBQVUsQ0FBVztRQUVyQyxJQUFJLFVBQVUsSUFBSSxNQUFNLEtBQUssV0FBVyxDQUFDLFVBQVUsRUFBRTtZQUNuRCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDM0U7SUFDSCxDQUFDO0lBL0JEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxHQUFXLEVBQUUsS0FBYSxFQUFFLFVBQWlDLEVBQUU7UUFDOUUsT0FBTyxJQUFJLGNBQWMsQ0FBQyxPQUFPLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxPQUFPLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNyRixDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQVcsRUFBRSxVQUFpQyxFQUFFO1FBQ25FLE9BQU8sSUFBSSxjQUFjLENBQUMsUUFBUSxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUUsT0FBTyxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDMUYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQUc7UUFDZixPQUFPLElBQUksY0FBYyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7O0FBckJILHdDQWtDQzs7O0FBNkNEOztHQUVHO0FBQ0gsTUFBYSxJQUFLLFNBQVEsc0JBQVM7SUFnQmpDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQ3JCLFdBQStCLEVBQy9CLE1BQWtDLEVBQ2xDLFVBQXlDO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFIQSxnQkFBVyxHQUFYLFdBQVcsQ0FBb0I7UUFDL0IsV0FBTSxHQUFOLE1BQU0sQ0FBNEI7UUFDbEMsZUFBVSxHQUFWLFVBQVUsQ0FBK0I7SUFFNUQsQ0FBQztJQW5CRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsT0FBMEI7UUFDM0UsT0FBTyxJQUFJLElBQUksQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLE9BQU8sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDdEYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxVQUEwQixFQUFFO1FBQzFFLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFTRDs7T0FFRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPO1lBQ0wsYUFBYSxFQUFFLGFBQWEsQ0FBQyxFQUFFLENBQUMsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ3ZGLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFLHlCQUF5QixFQUFFO1NBQ3pELENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsbUJBQW1CLEVBQUUsRUFBRSxDQUFDO0lBQ3JELENBQUM7SUFFRDs7T0FFRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDOztBQTdDSCxvQkErQ0M7OztBQUVEOztHQUVHO0FBQ0gsTUFBYSxXQUFXO0lBQ3RCLFlBQW1DLGFBQStCO1FBQS9CLGtCQUFhLEdBQWIsYUFBYSxDQUFrQjtJQUFHLENBQUM7SUFBQSxDQUFDOztBQUR6RSxrQ0FFQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLFdBQVc7SUFDdEIsWUFBbUMsYUFBK0I7UUFBL0Isa0JBQWEsR0FBYixhQUFhLENBQWtCO0lBQUcsQ0FBQztJQUFBLENBQUM7O0FBRHpFLGtDQUVDOzs7QUFFRDs7R0FFRztBQUNILE1BQWEsU0FBUztJQUNwQixZQUFtQyxJQUFZO1FBQVosU0FBSSxHQUFKLElBQUksQ0FBUTtJQUFHLENBQUM7SUFBQSxDQUFDOztBQUR0RCw4QkFFQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLElBQUk7SUFFZjs7T0FFRztJQUNJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxhQUErQjtRQUN0RCxPQUFPLElBQUksV0FBVyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBZ0I7UUFDbEMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsYUFBK0I7UUFDdEQsT0FBTyxJQUFJLFdBQVcsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUN4QyxDQUFDOztBQXJCSCxvQkF1QkM7OztBQUVEOztHQUVHO0FBQ0gsTUFBYSxRQUFRO0lBMENuQixZQUFvQyxHQUFXO1FBQVgsUUFBRyxHQUFILEdBQUcsQ0FBUTtJQUFHLENBQUM7SUFQbkQ7O09BRUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLEdBQVc7UUFDOUIsT0FBTyxJQUFJLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUMzQixDQUFDO0lBRWtELENBQUM7O0FBMUN0RCw0QkEyQ0M7OztBQXpDQzs7OztHQUlHO0FBQ29CLGlCQUFRLEdBQUcsSUFBSSxRQUFRLENBQUMsd0JBQXdCLENBQUMsQ0FBQztBQUV6RTs7Ozs7Ozs7O0dBU0c7QUFDb0IsYUFBSSxHQUFHLElBQUksUUFBUSxDQUFDLDZCQUE2QixDQUFDLENBQUM7QUFFMUU7Ozs7Ozs7Ozs7O0dBV0c7QUFDb0IsZUFBTSxHQUFHLElBQUksUUFBUSxDQUFDLCtCQUErQixDQUFDLENBQUM7QUE4RGhGOztHQUVHO0FBQ0gsTUFBYSxhQUFhO0lBV3hCLFlBQStCLFFBQXFCO1FBQXJCLGFBQVEsR0FBUixRQUFRLENBQWE7UUFUNUMsMkJBQXNCLEdBQWtDLEVBQUUsQ0FBQztRQUMzRCwwQkFBcUIsR0FBMkIsRUFBRSxDQUFDO1FBQ25ELDBCQUFxQixHQUFrQyxFQUFFLENBQUM7UUFDMUQseUJBQW9CLEdBQTBCLEVBQUUsQ0FBQztRQUNqRCw4QkFBeUIsR0FBa0MsRUFBRSxDQUFDO1FBQzlELDZCQUF3QixHQUEwQixFQUFFLENBQUM7UUFDckQsaUJBQVksR0FBcUIsRUFBRSxDQUFDO0lBR1csQ0FBQztJQUV4RDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxNQUFNLENBQUMsSUFBZTtRQUUzQixJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbEIsZ0RBQWdEO1lBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsaUJBQWlCLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLElBQUksWUFBWSxJQUFJLENBQUMsSUFBSSxvQ0FBb0MsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7U0FDM0k7YUFBTTtZQUNMLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztTQUM1QjtJQUNILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQW1CRztJQUNJLFFBQVEsQ0FBQyxJQUFpQjtRQUMvQixLQUFLLE1BQU0sS0FBSyxJQUFJLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFFdEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7Z0JBQ3JCLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztnQkFDZCxLQUFLLEVBQUUsS0FBSyxDQUFDLEtBQUs7Z0JBQ2xCLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtnQkFDcEIsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUN4QixpQkFBaUIsRUFBRSxLQUFLLENBQUMsVUFBVSxFQUFFLFNBQVMsRUFBRTthQUNqRCxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksT0FBTyxDQUFDLElBQWlCLEVBQUUsVUFBdUMsRUFBRTtRQUV6RSxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFL0MsSUFBSSxPQUFPLENBQUMsTUFBTSxFQUFFO1lBQ2xCLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3BDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUNoRjthQUFNO1lBQ0wsSUFBSSxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN2QztJQUNILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLFFBQVEsQ0FBQyxRQUFzQixFQUFFLFVBQXdDLEVBQUU7UUFFaEYsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLFFBQVEsSUFBSSxRQUFRLENBQUMsUUFBUSxDQUFDO1FBQ3ZELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFNUQsSUFBSSxPQUFPLENBQUMsTUFBTSxFQUFFO1lBQ2xCLElBQUksQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ3BDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztTQUNwRjthQUFNO1lBQ0wsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN0QztJQUNILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxRQUFRLENBQUMsUUFBc0IsRUFBRSxVQUF3QyxFQUFFO1FBRWhGLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxRQUFRLElBQUksUUFBUSxDQUFDLFFBQVEsQ0FBQztRQUN2RCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRTVELElBQUksT0FBTyxDQUFDLE1BQU0sRUFBRTtZQUNsQixJQUFJLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNwQyxJQUFJLENBQUMseUJBQXlCLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLEVBQUUsZUFBZSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7U0FDeEY7YUFBTTtZQUNMLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDMUM7SUFFSCxDQUFDO0lBRU8scUJBQXFCLENBQUMsUUFBa0IsRUFBRSxRQUFzQjtRQUN0RSxNQUFNLE1BQU0sR0FBRyxRQUFRLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUM5QyxPQUFPO1lBQ0wsV0FBVyxFQUFFLFFBQVEsQ0FBQyxHQUFHO1lBQ3pCLGFBQWEsRUFBRSxNQUFNLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRTtZQUM3QyxpQkFBaUIsRUFBRSxNQUFNLENBQUMsVUFBVSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUU7WUFDOUQsVUFBVSxFQUFFLE1BQU0sQ0FBQyxVQUFVLEVBQUUsS0FBSztTQUNyQyxDQUFDO0lBQ0osQ0FBQztJQUVPLHNCQUFzQixDQUFDLElBQWlCO1FBQzlDLE9BQU8sRUFBRSxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLEdBQUcsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDLFFBQVMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDO0lBQ3RILENBQUM7SUFFTyxjQUFjLENBQUMsTUFBYztRQUNuQyxJQUFJLE1BQU0sR0FBRyxDQUFDLElBQUksTUFBTSxHQUFHLEdBQUcsRUFBRTtZQUM5QixnR0FBZ0c7WUFDaEcsTUFBTSxJQUFJLEtBQUssQ0FBQyw0QkFBNEIsTUFBTSwwQkFBMEIsQ0FBQyxDQUFDO1NBQy9FO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTztRQUVaLE1BQU0sVUFBVSxHQUFHLENBQUMsR0FBRyxNQUFvQixFQUFFLEVBQUU7WUFDN0MsT0FBTyxNQUFNLENBQUMsSUFBSSxFQUFFLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztRQUNsQyxDQUFDLENBQUM7UUFFRixNQUFNLGVBQWUsR0FBRyxVQUFVLENBQUMsSUFBSSxDQUFDLHNCQUFzQixFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQzVGLE1BQU0sY0FBYyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMscUJBQXFCLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDekYsTUFBTSxpQkFBaUIsR0FBRyxVQUFVLENBQUMsSUFBSSxDQUFDLHlCQUF5QixFQUFFLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ3BHLE1BQU0sV0FBVyxHQUFHLGVBQWUsSUFBSSxjQUFjLElBQUksaUJBQWlCLENBQUM7UUFFM0UsT0FBTztZQUNMLFFBQVEsRUFBRSxXQUFXLENBQUMsQ0FBQyxDQUFDO2dCQUN0QixZQUFZLEVBQUUsZUFBZSxDQUFDLENBQUMsQ0FBQztvQkFDOUIsK0NBQStDLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDO29CQUM5Riw4Q0FBOEMsRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7d0JBQ3RGLGlCQUFpQixFQUFFLElBQUksQ0FBQyxxQkFBcUI7cUJBQzlDLENBQUMsQ0FBQyxDQUFDLFNBQVM7aUJBQ2QsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDYixXQUFXLEVBQUUsY0FBYyxDQUFDLENBQUMsQ0FBQztvQkFDNUIsK0NBQStDLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDO29CQUM3Riw4Q0FBOEMsRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUM7aUJBQzVGLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2IsZUFBZSxFQUFFLGlCQUFpQixDQUFDLENBQUMsQ0FBQztvQkFDbkMsK0NBQStDLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDO29CQUNqRyw4Q0FBOEMsRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsd0JBQXdCLENBQUM7aUJBQ2hHLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDZCxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2IsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTO1lBQ3hCLFdBQVcsRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDO1NBQ2pELENBQUM7SUFDSixDQUFDOztBQXBOSCxzQ0FxTkM7OztBQUVEOzs7R0FHRztBQUNILElBQVksdUJBWVg7QUFaRCxXQUFZLHVCQUF1QjtJQUVqQzs7T0FFRztJQUNILHNDQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILHdDQUFhLENBQUE7QUFFZixDQUFDLEVBWlcsdUJBQXVCLEdBQXZCLCtCQUF1QixLQUF2QiwrQkFBdUIsUUFZbEM7QUE0Q0Q7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFFekIsWUFBK0IsUUFBcUI7UUFBckIsYUFBUSxHQUFSLFFBQVEsQ0FBYTtJQUFHLENBQUM7SUFFeEQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksT0FBTyxDQUFDLElBQXNDLEVBQUUsVUFBd0MsRUFBRTtRQUMvRixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUNwRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksU0FBUyxDQUFDLElBQXNDLEVBQUUsVUFBMEMsRUFBRTtRQUNuRyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxHQUFHLE9BQU8sRUFBRSxDQUFDLENBQUM7SUFDOUYsQ0FBQztJQUVPLEtBQUssQ0FBQyxTQUErQixFQUFFLElBQXNDLEVBQUUsVUFBeUUsRUFBRTtRQUVoSyxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMseUJBQXlCLEVBQUUsQ0FBQztRQUNoRCxhQUFhLENBQUMsa0JBQWtCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFekMsTUFBTSxXQUFXLEdBQUcsZUFBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWxDLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssdUJBQXVCLENBQUMsR0FBRyxFQUFFO1lBRTNFLE1BQU0sR0FBRyxHQUFHLElBQUksYUFBYSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFFBQVEsU0FBUyxHQUFHLFdBQVcsRUFBRSxFQUFFO2dCQUM1RixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7Z0JBQ3ZCLHlEQUF5RDtnQkFDekQsdUJBQXVCO2dCQUN2QixRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFO2FBQzFELENBQUMsQ0FBQztZQUVILFFBQVEsU0FBUyxFQUFFO2dCQUNqQixLQUFLLFFBQVE7b0JBQ1gsR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO29CQUN2QyxNQUFNO2dCQUNSLEtBQUssU0FBUztvQkFDWixHQUFHLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDM0M7U0FFRjtRQUVELElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssdUJBQXVCLENBQUMsSUFBSSxFQUFFO1lBRTVFLElBQUksTUFBTSxDQUFDLE9BQU8sRUFBRTtnQkFDbEIsZ0VBQWdFO2dCQUNoRSxPQUFPO2FBQ1I7WUFFRCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDekMsSUFBSSxDQUFDLFdBQVcsRUFBRTtnQkFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLGdDQUFnQyxDQUFDLENBQUM7YUFDeEc7WUFFRCxNQUFNLGlCQUFpQixHQUFHLFNBQVMsS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDO1lBRXhFLE1BQU0saUJBQWlCLEdBQUcsV0FBVyxDQUFDLG1CQUFtQixFQUFFLENBQUM7WUFDNUQsSUFBSSxVQUFrQyxDQUFDO1lBRXZDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLEVBQUU7Z0JBRWpDLHdFQUF3RTtnQkFDeEUsVUFBVSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUM7YUFFakQ7aUJBQU07Z0JBRUwscURBQXFEO2dCQUNyRCw2REFBNkQ7Z0JBQzdELElBQUksaUJBQWlCLENBQUMsVUFBVSxDQUFDLGFBQWEsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLEVBQUU7b0JBQ3ZHLE1BQU0sSUFBSSxLQUFLLENBQUMsdUJBQXVCLGlCQUFpQixxQkFBcUIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLFVBQVUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLDhDQUE4QyxDQUFDLENBQUM7aUJBQ3hLO2dCQUVELDBDQUEwQztnQkFDMUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUU7b0JBQ3ZDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUJBQXVCLGlCQUFpQixxQkFBcUIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLFVBQVUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLHNDQUFzQyxDQUFDLENBQUM7aUJBQ2hLO2dCQUVELFVBQVUsR0FBRyxpQkFBaUIsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO2FBQ2pEO1lBRUQsS0FBSyxNQUFNLElBQUksSUFBSSxVQUFVLEVBQUU7Z0JBQzdCLFFBQVEsU0FBUyxFQUFFO29CQUNqQixLQUFLLFFBQVE7d0JBQ1gsSUFBSSxhQUFhLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsZUFBZSxJQUFJLEdBQUcsV0FBVyxFQUFFLEVBQUU7NEJBQ2xGLFFBQVEsRUFBRSxXQUFXOzRCQUNyQixRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFOzRCQUM3QixPQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsQ0FBQyxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxPQUFPLENBQUMsS0FBSyxFQUFFLENBQUMsRUFBRTt5QkFDcEUsQ0FBQyxDQUFDO3dCQUNILE1BQU07b0JBQ1IsS0FBSyxTQUFTO3dCQUNaLElBQUksYUFBYSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLGNBQWMsSUFBSSxHQUFHLFdBQVcsRUFBRSxFQUFFOzRCQUNqRixRQUFRLEVBQUUsV0FBVzs0QkFDckIsUUFBUSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRTs0QkFDN0IsTUFBTSxFQUFFLEVBQUUsS0FBSyxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLEtBQUssRUFBRSxDQUFDLEVBQUU7eUJBQ25FLENBQUMsQ0FBQzt3QkFDSCxNQUFNO29CQUNSO3dCQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsMEJBQTBCLFNBQVMsRUFBRSxDQUFDLENBQUM7aUJBQzFEO2FBQ0Y7U0FFRjtJQUNILENBQUM7SUFFTyxZQUFZLENBQUMsUUFBMkM7UUFDOUQsT0FBTyxTQUFTLENBQUMscUJBQXFCLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUMzRyxDQUFDOztBQTFJSCx3Q0EySUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBcGlPYmplY3QsIEFwaU9iamVjdE1ldGFkYXRhRGVmaW5pdGlvbiwgRHVyYXRpb24sIExhenksIE5hbWVzIH0gZnJvbSAnY2RrOHMnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBiYXNlIGZyb20gJy4vYmFzZSc7XG5pbXBvcnQgKiBhcyBjb250YWluZXIgZnJvbSAnLi9jb250YWluZXInO1xuaW1wb3J0ICogYXMgazhzIGZyb20gJy4vaW1wb3J0cy9rOHMnO1xuaW1wb3J0ICogYXMgbmFtZXNwYWNlIGZyb20gJy4vbmFtZXNwYWNlJztcbmltcG9ydCAqIGFzIG5ldHdvcmtwb2xpY3kgZnJvbSAnLi9uZXR3b3JrLXBvbGljeSc7XG5pbXBvcnQgKiBhcyByYiBmcm9tICcuL3JvbGUtYmluZGluZyc7XG5pbXBvcnQgKiBhcyBzZWNyZXQgZnJvbSAnLi9zZWNyZXQnO1xuaW1wb3J0ICogYXMgc2VydmljZWFjY291bnQgZnJvbSAnLi9zZXJ2aWNlLWFjY291bnQnO1xuaW1wb3J0IHsgdW5kZWZpbmVkSWZFbXB0eSwgYWRkcmVzcyB9IGZyb20gJy4vdXRpbHMnO1xuaW1wb3J0ICogYXMgdm9sdW1lIGZyb20gJy4vdm9sdW1lJztcblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEFic3RyYWN0UG9kIGV4dGVuZHMgYmFzZS5SZXNvdXJjZSBpbXBsZW1lbnRzIElQb2RTZWxlY3RvciwgbmV0d29ya3BvbGljeS5JTmV0d29ya1BvbGljeVBlZXIsIHJiLklTdWJqZWN0IHtcblxuICBwdWJsaWMgcmVhZG9ubHkgcmVzdGFydFBvbGljeT86IFJlc3RhcnRQb2xpY3k7XG4gIHB1YmxpYyByZWFkb25seSBzZXJ2aWNlQWNjb3VudD86IHNlcnZpY2VhY2NvdW50LklTZXJ2aWNlQWNjb3VudDtcbiAgcHVibGljIHJlYWRvbmx5IHNlY3VyaXR5Q29udGV4dDogUG9kU2VjdXJpdHlDb250ZXh0O1xuICBwdWJsaWMgcmVhZG9ubHkgZG5zOiBQb2REbnM7XG4gIHB1YmxpYyByZWFkb25seSBkb2NrZXJSZWdpc3RyeUF1dGg/OiBzZWNyZXQuRG9ja2VyQ29uZmlnU2VjcmV0O1xuICBwdWJsaWMgcmVhZG9ubHkgYXV0b21vdW50U2VydmljZUFjY291bnRUb2tlbjogYm9vbGVhbjtcblxuICBwcml2YXRlIHJlYWRvbmx5IF9jb250YWluZXJzOiBjb250YWluZXIuQ29udGFpbmVyW10gPSBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBfaW5pdENvbnRhaW5lcnM6IGNvbnRhaW5lci5Db250YWluZXJbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9ob3N0QWxpYXNlczogSG9zdEFsaWFzW10gPSBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBfdm9sdW1lczogTWFwPHN0cmluZywgdm9sdW1lLlZvbHVtZT4gPSBuZXcgTWFwKCk7XG5cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHBvZE1ldGFkYXRhOiBBcGlPYmplY3RNZXRhZGF0YURlZmluaXRpb247XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFic3RyYWN0UG9kUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnJlc3RhcnRQb2xpY3kgPSBwcm9wcy5yZXN0YXJ0UG9saWN5O1xuICAgIHRoaXMuc2VydmljZUFjY291bnQgPSBwcm9wcy5zZXJ2aWNlQWNjb3VudDtcbiAgICB0aGlzLnNlY3VyaXR5Q29udGV4dCA9IG5ldyBQb2RTZWN1cml0eUNvbnRleHQocHJvcHMuc2VjdXJpdHlDb250ZXh0KTtcbiAgICB0aGlzLmRucyA9IG5ldyBQb2REbnMocHJvcHMuZG5zKTtcbiAgICB0aGlzLmRvY2tlclJlZ2lzdHJ5QXV0aCA9IHByb3BzLmRvY2tlclJlZ2lzdHJ5QXV0aDtcbiAgICB0aGlzLmF1dG9tb3VudFNlcnZpY2VBY2NvdW50VG9rZW4gPSBwcm9wcy5hdXRvbW91bnRTZXJ2aWNlQWNjb3VudFRva2VuID8/IHRydWU7XG5cbiAgICBpZiAocHJvcHMuY29udGFpbmVycykge1xuICAgICAgcHJvcHMuY29udGFpbmVycy5mb3JFYWNoKGMgPT4gdGhpcy5hZGRDb250YWluZXIoYykpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy52b2x1bWVzKSB7XG4gICAgICBwcm9wcy52b2x1bWVzLmZvckVhY2godiA9PiB0aGlzLmFkZFZvbHVtZSh2KSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmluaXRDb250YWluZXJzKSB7XG4gICAgICBwcm9wcy5pbml0Q29udGFpbmVycy5mb3JFYWNoKGMgPT4gdGhpcy5hZGRJbml0Q29udGFpbmVyKGMpKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuaG9zdEFsaWFzZXMpIHtcbiAgICAgIHByb3BzLmhvc3RBbGlhc2VzLmZvckVhY2goYyA9PiB0aGlzLmFkZEhvc3RBbGlhcyhjKSk7XG4gICAgfVxuXG4gIH1cblxuICBwdWJsaWMgZ2V0IGNvbnRhaW5lcnMoKTogY29udGFpbmVyLkNvbnRhaW5lcltdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX2NvbnRhaW5lcnNdO1xuICB9XG5cbiAgcHVibGljIGdldCBpbml0Q29udGFpbmVycygpOiBjb250YWluZXIuQ29udGFpbmVyW10ge1xuICAgIHJldHVybiBbLi4udGhpcy5faW5pdENvbnRhaW5lcnNdO1xuICB9XG5cbiAgcHVibGljIGdldCB2b2x1bWVzKCk6IHZvbHVtZS5Wb2x1bWVbXSB7XG4gICAgcmV0dXJuIEFycmF5LmZyb20odGhpcy5fdm9sdW1lcy52YWx1ZXMoKSk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGhvc3RBbGlhc2VzKCk6IEhvc3RBbGlhc1tdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX2hvc3RBbGlhc2VzXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAc2VlIElQb2RTZWxlY3Rvci50b1BvZFNlbGVjdG9yQ29uZmlnKClcbiAgICovXG4gIHB1YmxpYyB0b1BvZFNlbGVjdG9yQ29uZmlnKCk6IFBvZFNlbGVjdG9yQ29uZmlnIHtcbiAgICBjb25zdCBwb2RBZGRyZXNzID0gdGhpcy5wb2RNZXRhZGF0YS5nZXRMYWJlbChQb2QuQUREUkVTU19MQUJFTCk7XG4gICAgaWYgKCFwb2RBZGRyZXNzKSB7XG4gICAgICAvLyBzaG91bGRuJ3QgaGFwcGVuIGJlY2F1c2Ugd2UgYWRkIHRoaXMgbGFiZWwgYXV0b21hdGljYWxseSBpbiBib3RoIHBvZHMgYW5kIHdvcmtsb2Fkcy5cbiAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIGNyZWF0ZSBhIGxhYmVsIHNlbGVjdG9yIHNpbmNlICR7UG9kLkFERFJFU1NfTEFCRUx9IGxhYmVsIGlzIG1pc3NpbmdgKTtcbiAgICB9XG4gICAgcmV0dXJuIHtcbiAgICAgIGxhYmVsU2VsZWN0b3I6IExhYmVsU2VsZWN0b3Iub2YoeyBsYWJlbHM6IHsgW1BvZC5BRERSRVNTX0xBQkVMXTogcG9kQWRkcmVzcyB9IH0pLFxuICAgICAgbmFtZXNwYWNlczogdGhpcy5tZXRhZGF0YS5uYW1lc3BhY2UgPyB7XG4gICAgICAgIG5hbWVzOiBbdGhpcy5tZXRhZGF0YS5uYW1lc3BhY2VdLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEBzZWUgSU5ldHdvcmtQb2xpY3lQZWVyLnRvTmV0d29ya1BvbGljeVBlZXJDb25maWcoKVxuICAgKi9cbiAgcHVibGljIHRvTmV0d29ya1BvbGljeVBlZXJDb25maWcoKTogbmV0d29ya3BvbGljeS5OZXR3b3JrUG9saWN5UGVlckNvbmZpZyB7XG4gICAgcmV0dXJuIHsgcG9kU2VsZWN0b3I6IHRoaXMudG9Qb2RTZWxlY3RvckNvbmZpZygpIH07XG4gIH1cblxuICAvKipcbiAgICogQHNlZSBJTmV0d29ya1BvbGljeVBlZXIudG9Qb2RTZWxlY3RvcigpXG4gICAqL1xuICBwdWJsaWMgdG9Qb2RTZWxlY3RvcigpOiBJUG9kU2VsZWN0b3IgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgcHVibGljIGFkZENvbnRhaW5lcihjb250OiBjb250YWluZXIuQ29udGFpbmVyUHJvcHMpOiBjb250YWluZXIuQ29udGFpbmVyIHtcbiAgICBjb25zdCBpbXBsID0gbmV3IGNvbnRhaW5lci5Db250YWluZXIoY29udCk7XG4gICAgdGhpcy5fY29udGFpbmVycy5wdXNoKGltcGwpO1xuICAgIHJldHVybiBpbXBsO1xuICB9XG5cbiAgcHVibGljIGFkZEluaXRDb250YWluZXIoY29udDogY29udGFpbmVyLkNvbnRhaW5lclByb3BzKTogY29udGFpbmVyLkNvbnRhaW5lciB7XG5cbiAgICAvLyBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy93b3JrbG9hZHMvcG9kcy9pbml0LWNvbnRhaW5lcnMvI2RpZmZlcmVuY2VzLWZyb20tcmVndWxhci1jb250YWluZXJzXG4gICAgaWYgKGNvbnQucmVhZGluZXNzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0luaXQgY29udGFpbmVycyBtdXN0IG5vdCBoYXZlIGEgcmVhZGluZXNzIHByb2JlJyk7XG4gICAgfVxuXG4gICAgaWYgKGNvbnQubGl2ZW5lc3MpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignSW5pdCBjb250YWluZXJzIG11c3Qgbm90IGhhdmUgYSBsaXZlbmVzcyBwcm9iZScpO1xuICAgIH1cblxuICAgIGlmIChjb250LnN0YXJ0dXApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignSW5pdCBjb250YWluZXJzIG11c3Qgbm90IGhhdmUgYSBzdGFydHVwIHByb2JlJyk7XG4gICAgfVxuXG4gICAgY29uc3QgaW1wbCA9IG5ldyBjb250YWluZXIuQ29udGFpbmVyKHtcbiAgICAgIC4uLmNvbnQsXG4gICAgICBuYW1lOiBjb250Lm5hbWUgPz8gYGluaXQtJHt0aGlzLl9pbml0Q29udGFpbmVycy5sZW5ndGh9YCxcbiAgICB9KTtcblxuICAgIHRoaXMuX2luaXRDb250YWluZXJzLnB1c2goaW1wbCk7XG4gICAgcmV0dXJuIGltcGw7XG4gIH1cblxuICBwdWJsaWMgYWRkSG9zdEFsaWFzKGhvc3RBbGlhczogSG9zdEFsaWFzKTogdm9pZCB7XG4gICAgdGhpcy5faG9zdEFsaWFzZXMucHVzaChob3N0QWxpYXMpO1xuICB9XG5cbiAgcHVibGljIGFkZFZvbHVtZSh2b2w6IHZvbHVtZS5Wb2x1bWUpOiB2b2lkIHtcbiAgICBjb25zdCBleGlzdGluZ1ZvbHVtZSA9IHRoaXMuX3ZvbHVtZXMuZ2V0KHZvbC5uYW1lKTtcbiAgICBpZiAoZXhpc3RpbmdWb2x1bWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVm9sdW1lIHdpdGggbmFtZSAke3ZvbC5uYW1lfSBhbHJlYWR5IGV4aXN0c2ApO1xuICAgIH1cbiAgICB0aGlzLl92b2x1bWVzLnNldCh2b2wubmFtZSwgdm9sKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAc2VlIElTdWJlY3QudG9TdWJqZWN0Q29uZmlndXJhdGlvbigpXG4gICAqL1xuICBwdWJsaWMgdG9TdWJqZWN0Q29uZmlndXJhdGlvbigpOiByYi5TdWJqZWN0Q29uZmlndXJhdGlvbiB7XG5cbiAgICBpZiAoIXRoaXMuc2VydmljZUFjY291bnQgJiYgIXRoaXMuYXV0b21vdW50U2VydmljZUFjY291bnRUb2tlbikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGAke3RoaXMubmFtZX0gY2Fubm90IGJlIGNvbnZlcnRlZCB0byBhIHJvbGUgYmluZGluZyBzdWJqZWN0OmBcbiAgICAgICAgKyAnIFlvdSBtdXN0IGVpdGhlciBhc3NpZ24gYSBzZXJ2aWNlIGFjY291bnQgdG8gaXQsIG9yIHVzZSBcXCdhdXRvbW91bnRTZXJ2aWNlQWNjb3VudFRva2VuOiB0cnVlXFwnJyk7XG4gICAgfVxuXG4gICAgLy8gJ2RlZmF1bHQnIGlzIGFzc3VtZWQgdG8gYmUgdGhlIG5hbWUgb2YgdGhlIGRlZmF1bHQgc2VydmljZSBhY2NvdW50XG4gICAgLy8gaW4gdGhlIGNsdXN0ZXIuXG4gICAgY29uc3Qgc2VydmljZUFjY291bnROYW1lID0gdGhpcy5zZXJ2aWNlQWNjb3VudD8ubmFtZSA/PyAnZGVmYXVsdCc7XG5cbiAgICByZXR1cm4ge1xuICAgICAga2luZDogJ1NlcnZpY2VBY2NvdW50JyxcbiAgICAgIG5hbWU6IHNlcnZpY2VBY2NvdW50TmFtZSxcbiAgICAgIGFwaUdyb3VwOiAnJyxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b1BvZFNwZWMoKTogazhzLlBvZFNwZWMge1xuXG4gICAgaWYgKHRoaXMuY29udGFpbmVycy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignUG9kU3BlYyBtdXN0IGhhdmUgYXQgbGVhc3QgMSBjb250YWluZXInKTtcbiAgICB9XG5cbiAgICBjb25zdCB2b2x1bWVzOiBNYXA8c3RyaW5nLCB2b2x1bWUuVm9sdW1lPiA9IG5ldyBNYXAoKTtcbiAgICBjb25zdCBjb250YWluZXJzOiBrOHMuQ29udGFpbmVyW10gPSBbXTtcbiAgICBjb25zdCBpbml0Q29udGFpbmVyczogazhzLkNvbnRhaW5lcltdID0gW107XG5cbiAgICBmb3IgKGNvbnN0IGNvbnQgb2YgdGhpcy5jb250YWluZXJzKSB7XG4gICAgICAvLyBhdXRvbWF0aWNhbGx5IGFkZCB2b2x1bWUgZnJvbSB0aGUgY29udGFpbmVyIG1vdW50XG4gICAgICAvLyB0byB0aGlzIHBvZCBzbyB0aGF0cyBpdHMgYXZhaWxhYmxlIHRvIHRoZSBjb250YWluZXIuXG4gICAgICBmb3IgKGNvbnN0IG1vdW50IG9mIGNvbnQubW91bnRzKSB7XG4gICAgICAgIGFkZFZvbHVtZShtb3VudC52b2x1bWUpO1xuICAgICAgfVxuICAgICAgY29udGFpbmVycy5wdXNoKGNvbnQuX3RvS3ViZSgpKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IGNvbnQgb2YgdGhpcy5pbml0Q29udGFpbmVycykge1xuICAgICAgLy8gYXV0b21hdGljYWxseSBhZGQgdm9sdW1lIGZyb20gdGhlIGNvbnRhaW5lciBtb3VudFxuICAgICAgLy8gdG8gdGhpcyBwb2Qgc28gdGhhdHMgaXRzIGF2YWlsYWJsZSB0byB0aGUgY29udGFpbmVyLlxuICAgICAgZm9yIChjb25zdCBtb3VudCBvZiBjb250Lm1vdW50cykge1xuICAgICAgICBhZGRWb2x1bWUobW91bnQudm9sdW1lKTtcbiAgICAgIH1cbiAgICAgIGluaXRDb250YWluZXJzLnB1c2goY29udC5fdG9LdWJlKCkpO1xuICAgIH1cblxuICAgIGZvciAoY29uc3Qgdm9sIG9mIHRoaXMudm9sdW1lcykge1xuICAgICAgYWRkVm9sdW1lKHZvbCk7XG4gICAgfVxuXG4gICAgZnVuY3Rpb24gYWRkVm9sdW1lKHZvbDogdm9sdW1lLlZvbHVtZSkge1xuICAgICAgY29uc3QgZXhpc3RpbmdWb2x1bWUgPSB2b2x1bWVzLmdldCh2b2wubmFtZSk7XG4gICAgICAvLyBpdHMgb2sgdG8gY2FsbCB0aGlzIGZ1bmN0aW9uIHR3aWNlIG9uIHRoZSBzYW1lIHZvbHVtZSwgYnV0IGl0cyBub3Qgb2sgdG9cbiAgICAgIC8vIGNhbGwgaXQgdHdpY2Ugb24gYSBkaWZmZXJlbnQgdm9sdW1lIHdpdGggdGhlIHNhbWUgbmFtZS5cbiAgICAgIGlmIChleGlzdGluZ1ZvbHVtZSAmJiBleGlzdGluZ1ZvbHVtZSAhPT0gdm9sKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBtb3VudCBjb25maWd1cmF0aW9uLiBBdCBsZWFzdCB0d28gZGlmZmVyZW50IHZvbHVtZXMgaGF2ZSB0aGUgc2FtZSBuYW1lOiAke3ZvbC5uYW1lfWApO1xuICAgICAgfVxuICAgICAgdm9sdW1lcy5zZXQodm9sLm5hbWUsIHZvbCk7XG4gICAgfVxuXG4gICAgY29uc3QgZG5zID0gdGhpcy5kbnMuX3RvS3ViZSgpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIHJlc3RhcnRQb2xpY3k6IHRoaXMucmVzdGFydFBvbGljeSxcbiAgICAgIHNlcnZpY2VBY2NvdW50TmFtZTogdGhpcy5zZXJ2aWNlQWNjb3VudD8ubmFtZSxcbiAgICAgIGNvbnRhaW5lcnM6IGNvbnRhaW5lcnMsXG4gICAgICBzZWN1cml0eUNvbnRleHQ6IHVuZGVmaW5lZElmRW1wdHkodGhpcy5zZWN1cml0eUNvbnRleHQuX3RvS3ViZSgpKSxcbiAgICAgIGluaXRDb250YWluZXJzOiB1bmRlZmluZWRJZkVtcHR5KGluaXRDb250YWluZXJzKSxcbiAgICAgIGhvc3RBbGlhc2VzOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMuaG9zdEFsaWFzZXMpLFxuICAgICAgdm9sdW1lczogdW5kZWZpbmVkSWZFbXB0eShBcnJheS5mcm9tKHZvbHVtZXMudmFsdWVzKCkpLm1hcCh2ID0+IHYuX3RvS3ViZSgpKSksXG4gICAgICBkbnNQb2xpY3k6IGRucy5wb2xpY3ksXG4gICAgICBkbnNDb25maWc6IHVuZGVmaW5lZElmRW1wdHkoZG5zLmNvbmZpZyksXG4gICAgICBob3N0bmFtZTogZG5zLmhvc3RuYW1lLFxuICAgICAgc3ViZG9tYWluOiBkbnMuc3ViZG9tYWluLFxuICAgICAgc2V0SG9zdG5hbWVBc0ZxZG46IGRucy5ob3N0bmFtZUFzRlFETixcbiAgICAgIGltYWdlUHVsbFNlY3JldHM6IHRoaXMuZG9ja2VyUmVnaXN0cnlBdXRoID8gW3sgbmFtZTogdGhpcy5kb2NrZXJSZWdpc3RyeUF1dGgubmFtZSB9XSA6IHVuZGVmaW5lZCxcbiAgICAgIGF1dG9tb3VudFNlcnZpY2VBY2NvdW50VG9rZW46IHRoaXMuYXV0b21vdW50U2VydmljZUFjY291bnRUb2tlbixcbiAgICB9O1xuXG4gIH1cblxufVxuXG4vKipcbiAqIFN5c2N0bCBkZWZpbmVzIGEga2VybmVsIHBhcmFtZXRlciB0byBiZSBzZXRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTeXNjdGwge1xuICAvKipcbiAgICogTmFtZSBvZiBhIHByb3BlcnR5IHRvIHNldFxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBWYWx1ZSBvZiBhIHByb3BlcnR5IHRvIHNldFxuICAgKi9cbiAgcmVhZG9ubHkgdmFsdWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgUG9kU2VjdXJpdHlDb250ZXh0YFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFBvZFNlY3VyaXR5Q29udGV4dFByb3BzIHtcblxuICAvKipcbiAgICogTW9kaWZ5IHRoZSBvd25lcnNoaXAgYW5kIHBlcm1pc3Npb25zIG9mIHBvZCB2b2x1bWVzIHRvIHRoaXMgR0lELlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFZvbHVtZSBvd25lcnNoaXAgaXMgbm90IGNoYW5nZWQuXG4gICAqL1xuICByZWFkb25seSBmc0dyb3VwPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGJlaGF2aW9yIG9mIGNoYW5naW5nIG93bmVyc2hpcCBhbmQgcGVybWlzc2lvbiBvZiB0aGUgdm9sdW1lIGJlZm9yZSBiZWluZyBleHBvc2VkIGluc2lkZSBQb2QuXG4gICAqIFRoaXMgZmllbGQgd2lsbCBvbmx5IGFwcGx5IHRvIHZvbHVtZSB0eXBlcyB3aGljaCBzdXBwb3J0IGZzR3JvdXAgYmFzZWQgb3duZXJzaGlwKGFuZCBwZXJtaXNzaW9ucykuXG4gICAqIEl0IHdpbGwgaGF2ZSBubyBlZmZlY3Qgb24gZXBoZW1lcmFsIHZvbHVtZSB0eXBlcyBzdWNoIGFzOiBzZWNyZXQsIGNvbmZpZ21hcHMgYW5kIGVtcHR5ZGlyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBGc0dyb3VwQ2hhbmdlUG9saWN5LkFMV0FZU1xuICAgKi9cbiAgcmVhZG9ubHkgZnNHcm91cENoYW5nZVBvbGljeT86IEZzR3JvdXBDaGFuZ2VQb2xpY3k7XG5cbiAgLyoqXG4gICAqIFRoZSBVSUQgdG8gcnVuIHRoZSBlbnRyeXBvaW50IG9mIHRoZSBjb250YWluZXIgcHJvY2Vzcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBVc2VyIHNwZWNpZmllZCBpbiBpbWFnZSBtZXRhZGF0YVxuICAgKi9cbiAgcmVhZG9ubHkgdXNlcj86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIEdJRCB0byBydW4gdGhlIGVudHJ5cG9pbnQgb2YgdGhlIGNvbnRhaW5lciBwcm9jZXNzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEdyb3VwIGNvbmZpZ3VyZWQgYnkgY29udGFpbmVyIHJ1bnRpbWVcbiAgICovXG4gIHJlYWRvbmx5IGdyb3VwPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgdGhhdCB0aGUgY29udGFpbmVyIG11c3QgcnVuIGFzIGEgbm9uLXJvb3QgdXNlci5cbiAgICogSWYgdHJ1ZSwgdGhlIEt1YmVsZXQgd2lsbCB2YWxpZGF0ZSB0aGUgaW1hZ2UgYXQgcnVudGltZSB0byBlbnN1cmUgdGhhdCBpdCBkb2VzXG4gICAqIG5vdCBydW4gYXMgVUlEIDAgKHJvb3QpIGFuZCBmYWlsIHRvIHN0YXJ0IHRoZSBjb250YWluZXIgaWYgaXQgZG9lcy5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGVuc3VyZU5vblJvb3Q/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBTeXNjdGxzIGhvbGQgYSBsaXN0IG9mIG5hbWVzcGFjZWQgc3lzY3RscyB1c2VkIGZvciB0aGUgcG9kLlxuICAgKiBQb2RzIHdpdGggdW5zdXBwb3J0ZWQgc3lzY3RscyAoYnkgdGhlIGNvbnRhaW5lciBydW50aW1lKSBtaWdodCBmYWlsIHRvIGxhdW5jaC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBzeXNjdGxzXG4gICAqL1xuICByZWFkb25seSBzeXNjdGxzPzogU3lzY3RsW107XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYEFic3RyYWN0UG9kYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBYnN0cmFjdFBvZFByb3BzIGV4dGVuZHMgYmFzZS5SZXNvdXJjZVByb3BzIHtcblxuICAvKipcbiAgICogTGlzdCBvZiBjb250YWluZXJzIGJlbG9uZ2luZyB0byB0aGUgcG9kLiBDb250YWluZXJzIGNhbm5vdCBjdXJyZW50bHkgYmVcbiAgICogYWRkZWQgb3IgcmVtb3ZlZC4gVGhlcmUgbXVzdCBiZSBhdCBsZWFzdCBvbmUgY29udGFpbmVyIGluIGEgUG9kLlxuICAgKlxuICAgKiBZb3UgY2FuIGFkZCBhZGRpdGlvbm5hbCBjb250YWluZXJzIHVzaW5nIGBwb2RTcGVjLmFkZENvbnRhaW5lcigpYFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGNvbnRhaW5lcnMuIE5vdGUgdGhhdCBhIHBvZCBzcGVjIG11c3QgaW5jbHVkZSBhdCBsZWFzdCBvbmUgY29udGFpbmVyLlxuICAgKi9cbiAgcmVhZG9ubHkgY29udGFpbmVycz86IGNvbnRhaW5lci5Db250YWluZXJQcm9wc1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGluaXRpYWxpemF0aW9uIGNvbnRhaW5lcnMgYmVsb25naW5nIHRvIHRoZSBwb2QuXG4gICAqIEluaXQgY29udGFpbmVycyBhcmUgZXhlY3V0ZWQgaW4gb3JkZXIgcHJpb3IgdG8gY29udGFpbmVycyBiZWluZyBzdGFydGVkLlxuICAgKiBJZiBhbnkgaW5pdCBjb250YWluZXIgZmFpbHMsIHRoZSBwb2QgaXMgY29uc2lkZXJlZCB0byBoYXZlIGZhaWxlZCBhbmQgaXMgaGFuZGxlZCBhY2NvcmRpbmcgdG8gaXRzIHJlc3RhcnRQb2xpY3kuXG4gICAqIFRoZSBuYW1lIGZvciBhbiBpbml0IGNvbnRhaW5lciBvciBub3JtYWwgY29udGFpbmVyIG11c3QgYmUgdW5pcXVlIGFtb25nIGFsbCBjb250YWluZXJzLlxuICAgKiBJbml0IGNvbnRhaW5lcnMgbWF5IG5vdCBoYXZlIExpZmVjeWNsZSBhY3Rpb25zLCBSZWFkaW5lc3MgcHJvYmVzLCBMaXZlbmVzcyBwcm9iZXMsIG9yIFN0YXJ0dXAgcHJvYmVzLlxuICAgKiBUaGUgcmVzb3VyY2VSZXF1aXJlbWVudHMgb2YgYW4gaW5pdCBjb250YWluZXIgYXJlIHRha2VuIGludG8gYWNjb3VudCBkdXJpbmcgc2NoZWR1bGluZyBieSBmaW5kaW5nIHRoZSBoaWdoZXN0IHJlcXVlc3QvbGltaXRcbiAgICogZm9yIGVhY2ggcmVzb3VyY2UgdHlwZSwgYW5kIHRoZW4gdXNpbmcgdGhlIG1heCBvZiBvZiB0aGF0IHZhbHVlIG9yIHRoZSBzdW0gb2YgdGhlIG5vcm1hbCBjb250YWluZXJzLlxuICAgKiBMaW1pdHMgYXJlIGFwcGxpZWQgdG8gaW5pdCBjb250YWluZXJzIGluIGEgc2ltaWxhciBmYXNoaW9uLlxuICAgKlxuICAgKiBJbml0IGNvbnRhaW5lcnMgY2Fubm90IGN1cnJlbnRseSBiZSBhZGRlZCAscmVtb3ZlZCBvciB1cGRhdGVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3dvcmtsb2Fkcy9wb2RzL2luaXQtY29udGFpbmVycy9cbiAgICogQGRlZmF1bHQgLSBObyBpbml0IGNvbnRhaW5lcnMuXG4gICAqL1xuICByZWFkb25seSBpbml0Q29udGFpbmVycz86IGNvbnRhaW5lci5Db250YWluZXJQcm9wc1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIHZvbHVtZXMgdGhhdCBjYW4gYmUgbW91bnRlZCBieSBjb250YWluZXJzIGJlbG9uZ2luZyB0byB0aGUgcG9kLlxuICAgKlxuICAgKiBZb3UgY2FuIGFsc28gYWRkIHZvbHVtZXMgbGF0ZXIgdXNpbmcgYHBvZFNwZWMuYWRkVm9sdW1lKClgXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gdm9sdW1lcy5cbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZXM/OiB2b2x1bWUuVm9sdW1lW107XG5cbiAgLyoqXG4gICAqIFJlc3RhcnQgcG9saWN5IGZvciBhbGwgY29udGFpbmVycyB3aXRoaW4gdGhlIHBvZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy93b3JrbG9hZHMvcG9kcy9wb2QtbGlmZWN5Y2xlLyNyZXN0YXJ0LXBvbGljeVxuICAgKlxuICAgKiBAZGVmYXVsdCBSZXN0YXJ0UG9saWN5LkFMV0FZU1xuICAgKi9cbiAgcmVhZG9ubHkgcmVzdGFydFBvbGljeT86IFJlc3RhcnRQb2xpY3k7XG5cbiAgLyoqXG4gICAqIEEgc2VydmljZSBhY2NvdW50IHByb3ZpZGVzIGFuIGlkZW50aXR5IGZvciBwcm9jZXNzZXMgdGhhdCBydW4gaW4gYSBQb2QuXG4gICAqXG4gICAqIFdoZW4geW91IChhIGh1bWFuKSBhY2Nlc3MgdGhlIGNsdXN0ZXIgKGZvciBleGFtcGxlLCB1c2luZyBrdWJlY3RsKSwgeW91IGFyZVxuICAgKiBhdXRoZW50aWNhdGVkIGJ5IHRoZSBhcGlzZXJ2ZXIgYXMgYSBwYXJ0aWN1bGFyIFVzZXIgQWNjb3VudCAoY3VycmVudGx5IHRoaXNcbiAgICogaXMgdXN1YWxseSBhZG1pbiwgdW5sZXNzIHlvdXIgY2x1c3RlciBhZG1pbmlzdHJhdG9yIGhhcyBjdXN0b21pemVkIHlvdXJcbiAgICogY2x1c3RlcikuIFByb2Nlc3NlcyBpbiBjb250YWluZXJzIGluc2lkZSBwb2RzIGNhbiBhbHNvIGNvbnRhY3QgdGhlXG4gICAqIGFwaXNlcnZlci4gV2hlbiB0aGV5IGRvLCB0aGV5IGFyZSBhdXRoZW50aWNhdGVkIGFzIGEgcGFydGljdWxhciBTZXJ2aWNlXG4gICAqIEFjY291bnQgKGZvciBleGFtcGxlLCBkZWZhdWx0KS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy90YXNrcy9jb25maWd1cmUtcG9kLWNvbnRhaW5lci9jb25maWd1cmUtc2VydmljZS1hY2NvdW50L1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHNlcnZpY2UgYWNjb3VudC5cbiAgICovXG4gIHJlYWRvbmx5IHNlcnZpY2VBY2NvdW50Pzogc2VydmljZWFjY291bnQuSVNlcnZpY2VBY2NvdW50O1xuXG4gIC8qKlxuICAgKiBTZWN1cml0eUNvbnRleHQgaG9sZHMgcG9kLWxldmVsIHNlY3VyaXR5IGF0dHJpYnV0ZXMgYW5kIGNvbW1vbiBjb250YWluZXIgc2V0dGluZ3MuXG4gICAqXG4gICAqIEBkZWZhdWx0XG4gICAqXG4gICAqICAgZnNHcm91cENoYW5nZVBvbGljeTogRnNHcm91cENoYW5nZVBvbGljeS5Gc0dyb3VwQ2hhbmdlUG9saWN5LkFMV0FZU1xuICAgKiAgIGVuc3VyZU5vblJvb3Q6IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUNvbnRleHQ/OiBQb2RTZWN1cml0eUNvbnRleHRQcm9wcztcblxuICAvKipcbiAgICogSG9zdEFsaWFzIGhvbGRzIHRoZSBtYXBwaW5nIGJldHdlZW4gSVAgYW5kIGhvc3RuYW1lcyB0aGF0IHdpbGwgYmUgaW5qZWN0ZWQgYXMgYW4gZW50cnkgaW4gdGhlIHBvZCdzIGhvc3RzIGZpbGUuXG4gICAqXG4gICAqIEBzY2hlbWEgaW8uazhzLmFwaS5jb3JlLnYxLkhvc3RBbGlhc1xuICAgKi9cbiAgcmVhZG9ubHkgaG9zdEFsaWFzZXM/OiBIb3N0QWxpYXNbXTtcblxuICAvKipcbiAgICogRE5TIHNldHRpbmdzIGZvciB0aGUgcG9kLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3NlcnZpY2VzLW5ldHdvcmtpbmcvZG5zLXBvZC1zZXJ2aWNlL1xuICAgKlxuICAgKiBAZGVmYXVsdFxuICAgKlxuICAgKiAgcG9saWN5OiBEbnNQb2xpY3kuQ0xVU1RFUl9GSVJTVFxuICAgKiAgaG9zdG5hbWVBc0ZRRE46IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBkbnM/OiBQb2REbnNQcm9wcztcblxuICAvKipcbiAgICogQSBzZWNyZXQgY29udGFpbmluZyBkb2NrZXIgY3JlZGVudGlhbHMgZm9yIGF1dGhlbnRpY2F0aW5nIHRvIGEgcmVnaXN0cnkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYXV0aC4gSW1hZ2VzIGFyZSBhc3N1bWVkIHRvIGJlIHB1YmxpY2x5IGF2YWlsYWJsZS5cbiAgICovXG4gIHJlYWRvbmx5IGRvY2tlclJlZ2lzdHJ5QXV0aD86IHNlY3JldC5Eb2NrZXJDb25maWdTZWNyZXQ7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIGEgc2VydmljZSBhY2NvdW50IHRva2VuIHNob3VsZCBiZSBhdXRvbWF0aWNhbGx5IG1vdW50ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy90YXNrcy9jb25maWd1cmUtcG9kLWNvbnRhaW5lci9jb25maWd1cmUtc2VydmljZS1hY2NvdW50LyN1c2UtdGhlLWRlZmF1bHQtc2VydmljZS1hY2NvdW50LXRvLWFjY2Vzcy10aGUtYXBpLXNlcnZlclxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b21vdW50U2VydmljZUFjY291bnRUb2tlbj86IGJvb2xlYW47XG5cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgUG9kYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb2RQcm9wcyBleHRlbmRzIEFic3RyYWN0UG9kUHJvcHMge31cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgTGFiZWxTZWxlY3Rvci5vZmAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFiZWxTZWxlY3Rvck9wdGlvbnMge1xuXG4gIC8qKlxuICAgKiBTdHJpY3QgbGFiZWwgbWF0Y2hlcnMuXG4gICAqL1xuICByZWFkb25seSBsYWJlbHM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBFeHByZXNzaW9uIGJhc2VkIGxhYmVsIG1hdGNoZXJzLlxuICAgKi9cbiAgcmVhZG9ubHkgZXhwcmVzc2lvbnM/OiBMYWJlbEV4cHJlc3Npb25bXTtcbn1cblxuLyoqXG4gKiBNYXRjaCBhIHJlc291cmNlIGJ5IGxhYmVscy5cbiAqL1xuZXhwb3J0IGNsYXNzIExhYmVsU2VsZWN0b3Ige1xuXG4gIHB1YmxpYyBzdGF0aWMgb2Yob3B0aW9uczogTGFiZWxTZWxlY3Rvck9wdGlvbnMgPSB7fSkge1xuICAgIHJldHVybiBuZXcgTGFiZWxTZWxlY3RvcihvcHRpb25zLmV4cHJlc3Npb25zID8/IFtdLCBvcHRpb25zLmxhYmVscyA/PyB7fSk7XG4gIH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKFxuICAgIHByaXZhdGUgcmVhZG9ubHkgZXhwcmVzc2lvbnM6IExhYmVsRXhwcmVzc2lvbltdLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgbGFiZWxzOiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9KSB7fVxuXG4gIHB1YmxpYyBpc0VtcHR5KCkge1xuICAgIHJldHVybiB0aGlzLmV4cHJlc3Npb25zLmxlbmd0aCA9PT0gMCAmJiBPYmplY3Qua2V5cyh0aGlzLmxhYmVscykubGVuZ3RoID09PSAwO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLkxhYmVsU2VsZWN0b3Ige1xuICAgIGlmICh0aGlzLmlzRW1wdHkoKSkge1xuICAgICAgcmV0dXJuIHt9O1xuICAgIH1cbiAgICByZXR1cm4ge1xuICAgICAgbWF0Y2hFeHByZXNzaW9uczogdW5kZWZpbmVkSWZFbXB0eSh0aGlzLmV4cHJlc3Npb25zLm1hcChxID0+ICh7IGtleTogcS5rZXksIG9wZXJhdG9yOiBxLm9wZXJhdG9yLCB2YWx1ZXM6IHEudmFsdWVzIH0pKSksXG4gICAgICBtYXRjaExhYmVsczogdW5kZWZpbmVkSWZFbXB0eSh0aGlzLmxhYmVscyksXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIHNlbGVjdGluZyBwb2RzLCBvcHRpb25hbGx5IGluIHBhcnRpY3VsYXIgbmFtZXNwYWNlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb2RTZWxlY3RvckNvbmZpZyB7XG5cbiAgLyoqXG4gICAqIEEgc2VsZWN0b3IgdG8gc2VsZWN0IHBvZHMgYnkgbGFiZWxzLlxuICAgKi9cbiAgcmVhZG9ubHkgbGFiZWxTZWxlY3RvcjogTGFiZWxTZWxlY3RvcjtcblxuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBmb3Igc2VsZWN0aW5nIHdoaWNoIG5hbWVwc2FjZXMgYXJlIHRoZSBwb2RzIGFsbG93ZWQgdG8gYmUgaW4uXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VzPzogbmFtZXNwYWNlLk5hbWVzcGFjZVNlbGVjdG9yQ29uZmlnO1xuXG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBvYmplY3QgdGhhdCBjYW4gc2VsZWN0IHBvZHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVBvZFNlbGVjdG9yIGV4dGVuZHMgSUNvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIGNvbmZpZ3VyYXRpb24gb2YgdGhpcyBzZWxlY3Rvci5cbiAgICovXG4gIHRvUG9kU2VsZWN0b3JDb25maWcoKTogUG9kU2VsZWN0b3JDb25maWc7XG59XG5cbi8qKlxuICogUG9kIGlzIGEgY29sbGVjdGlvbiBvZiBjb250YWluZXJzIHRoYXQgY2FuIHJ1biBvbiBhIGhvc3QuIFRoaXMgcmVzb3VyY2UgaXNcbiAqIGNyZWF0ZWQgYnkgY2xpZW50cyBhbmQgc2NoZWR1bGVkIG9udG8gaG9zdHMuXG4gKi9cbmV4cG9ydCBjbGFzcyBQb2QgZXh0ZW5kcyBBYnN0cmFjdFBvZCB7XG5cbiAgLyoqXG4gICAqIFRoaXMgbGFiZWwgaXMgYXV0b2FtdGljYWxseSBhZGRlZCBieSBjZGs4cyB0byBhbnkgcG9kLiBJdCBwcm92aWRlc1xuICAgKiBhIHVuaXF1ZSBhbmQgc3RhYmxlIGlkZW50aWZpZXIgZm9yIHRoZSBwb2QuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEFERFJFU1NfTEFCRUwgPSAnY2RrOHMuaW8vbWV0YWRhdGEuYWRkcic7XG5cbiAgLyoqXG4gICAqIEBzZWUgYmFzZS5SZXNvdXJjZS5hcGlPYmplY3RcbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlPYmplY3Q6IEFwaU9iamVjdDtcblxuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VUeXBlID0gJ3BvZHMnO1xuXG4gIHB1YmxpYyByZWFkb25seSBzY2hlZHVsaW5nOiBQb2RTY2hlZHVsaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IFBvZENvbm5lY3Rpb25zO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBQb2RQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLmFwaU9iamVjdCA9IG5ldyBrOHMuS3ViZVBvZCh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBtZXRhZGF0YTogcHJvcHMubWV0YWRhdGEsXG4gICAgICBzcGVjOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuX3RvS3ViZSgpIH0pLFxuICAgIH0pO1xuXG4gICAgdGhpcy5tZXRhZGF0YS5hZGRMYWJlbChQb2QuQUREUkVTU19MQUJFTCwgTmFtZXMudG9MYWJlbFZhbHVlKHRoaXMpKTtcblxuICAgIHRoaXMuc2NoZWR1bGluZyA9IG5ldyBQb2RTY2hlZHVsaW5nKHRoaXMpO1xuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgUG9kQ29ubmVjdGlvbnModGhpcyk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHBvZE1ldGFkYXRhKCk6IEFwaU9iamVjdE1ldGFkYXRhRGVmaW5pdGlvbiB7XG4gICAgcmV0dXJuIHRoaXMubWV0YWRhdGE7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiBrOHMuUG9kU3BlYyB7XG4gICAgY29uc3Qgc2NoZWR1bGluZyA9IHRoaXMuc2NoZWR1bGluZy5fdG9LdWJlKCk7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLnRoaXMuX3RvUG9kU3BlYygpLFxuICAgICAgYWZmaW5pdHk6IHNjaGVkdWxpbmcuYWZmaW5pdHksXG4gICAgICBub2RlTmFtZTogc2NoZWR1bGluZy5ub2RlTmFtZSxcbiAgICAgIHRvbGVyYXRpb25zOiBzY2hlZHVsaW5nLnRvbGVyYXRpb25zLFxuICAgIH07XG4gIH1cblxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBQb2REbnNgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFBvZERuc1Byb3BzIHtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBob3N0bmFtZSBvZiB0aGUgUG9kLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFNldCB0byBhIHN5c3RlbS1kZWZpbmVkIHZhbHVlLlxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdG5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIElmIHNwZWNpZmllZCwgdGhlIGZ1bGx5IHF1YWxpZmllZCBQb2QgaG9zdG5hbWUgd2lsbCBiZSBcIjxob3N0bmFtZT4uPHN1YmRvbWFpbj4uPHBvZCBuYW1lc3BhY2U+LnN2Yy48Y2x1c3RlciBkb21haW4+XCIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gc3ViZG9tYWluLlxuICAgKi9cbiAgcmVhZG9ubHkgc3ViZG9tYWluPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJZiB0cnVlIHRoZSBwb2QncyBob3N0bmFtZSB3aWxsIGJlIGNvbmZpZ3VyZWQgYXMgdGhlIHBvZCdzIEZRRE4sIHJhdGhlciB0aGFuIHRoZSBsZWFmIG5hbWUgKHRoZSBkZWZhdWx0KS5cbiAgICogSW4gTGludXggY29udGFpbmVycywgdGhpcyBtZWFucyBzZXR0aW5nIHRoZSBGUUROIGluIHRoZSBob3N0bmFtZSBmaWVsZCBvZiB0aGUga2VybmVsICh0aGUgbm9kZW5hbWUgZmllbGQgb2Ygc3RydWN0IHV0c25hbWUpLlxuICAgKiBJbiBXaW5kb3dzIGNvbnRhaW5lcnMsIHRoaXMgbWVhbnMgc2V0dGluZyB0aGUgcmVnaXN0cnkgdmFsdWUgb2YgaG9zdG5hbWUgZm9yIHRoZSByZWdpc3RyeVxuICAgKiBrZXkgSEtFWV9MT0NBTF9NQUNISU5FXFxTWVNURU1cXEN1cnJlbnRDb250cm9sU2V0XFxTZXJ2aWNlc1xcVGNwaXBcXFBhcmFtZXRlcnMgdG8gRlFETi5cbiAgICogSWYgYSBwb2QgZG9lcyBub3QgaGF2ZSBGUUROLCB0aGlzIGhhcyBubyBlZmZlY3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBob3N0bmFtZUFzRlFETj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFNldCBETlMgcG9saWN5IGZvciB0aGUgcG9kLlxuICAgKlxuICAgKiBJZiBwb2xpY3kgaXMgc2V0IHRvIGBOb25lYCwgb3RoZXIgY29uZmlndXJhdGlvbiBtdXN0IGJlIHN1cHBsaWVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEbnNQb2xpY3kuQ0xVU1RFUl9GSVJTVFxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5PzogRG5zUG9saWN5O1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgSVAgYWRkcmVzc2VzIHRoYXQgd2lsbCBiZSB1c2VkIGFzIEROUyBzZXJ2ZXJzIGZvciB0aGUgUG9kLiBUaGVyZSBjYW4gYmUgYXQgbW9zdCAzIElQIGFkZHJlc3NlcyBzcGVjaWZpZWQuXG4gICAqIFdoZW4gdGhlIHBvbGljeSBpcyBzZXQgdG8gXCJOT05FXCIsIHRoZSBsaXN0IG11c3QgY29udGFpbiBhdCBsZWFzdCBvbmUgSVAgYWRkcmVzcyxcbiAgICogb3RoZXJ3aXNlIHRoaXMgcHJvcGVydHkgaXMgb3B0aW9uYWwuXG4gICAqIFRoZSBzZXJ2ZXJzIGxpc3RlZCB3aWxsIGJlIGNvbWJpbmVkIHRvIHRoZSBiYXNlIG5hbWVzZXJ2ZXJzIGdlbmVyYXRlZCBmcm9tXG4gICAqIHRoZSBzcGVjaWZpZWQgRE5TIHBvbGljeSB3aXRoIGR1cGxpY2F0ZSBhZGRyZXNzZXMgcmVtb3ZlZC5cbiAgICovXG4gIHJlYWRvbmx5IG5hbWVzZXJ2ZXJzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBETlMgc2VhcmNoIGRvbWFpbnMgZm9yIGhvc3RuYW1lIGxvb2t1cCBpbiB0aGUgUG9kLlxuICAgKiBXaGVuIHNwZWNpZmllZCwgdGhlIHByb3ZpZGVkIGxpc3Qgd2lsbCBiZSBtZXJnZWQgaW50byB0aGUgYmFzZVxuICAgKiBzZWFyY2ggZG9tYWluIG5hbWVzIGdlbmVyYXRlZCBmcm9tIHRoZSBjaG9zZW4gRE5TIHBvbGljeS5cbiAgICogRHVwbGljYXRlIGRvbWFpbiBuYW1lcyBhcmUgcmVtb3ZlZC5cbiAgICpcbiAgICogS3ViZXJuZXRlcyBhbGxvd3MgZm9yIGF0IG1vc3QgNiBzZWFyY2ggZG9tYWlucy5cbiAgICovXG4gIHJlYWRvbmx5IHNlYXJjaGVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2Ygb2JqZWN0cyB3aGVyZSBlYWNoIG9iamVjdCBtYXkgaGF2ZSBhIG5hbWUgcHJvcGVydHkgKHJlcXVpcmVkKVxuICAgKiBhbmQgYSB2YWx1ZSBwcm9wZXJ0eSAob3B0aW9uYWwpLiBUaGUgY29udGVudHMgaW4gdGhpcyBwcm9wZXJ0eVxuICAgKiB3aWxsIGJlIG1lcmdlZCB0byB0aGUgb3B0aW9ucyBnZW5lcmF0ZWQgZnJvbSB0aGUgc3BlY2lmaWVkIEROUyBwb2xpY3kuXG4gICAqIER1cGxpY2F0ZSBlbnRyaWVzIGFyZSByZW1vdmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgb3B0aW9ucz86IERuc09wdGlvbltdO1xufVxuXG4vKipcbiAqIEhvbGRzIGRucyBzZXR0aW5ncyBvZiB0aGUgcG9kLlxuICovXG5leHBvcnQgY2xhc3MgUG9kRG5zIHtcblxuICAvKipcbiAgICogVGhlIEROUyBwb2xpY3kgb2YgdGhpcyBwb2QuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcG9saWN5OiBEbnNQb2xpY3k7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25maWd1cmVkIGhvc3RuYW1lIG9mIHRoZSBwb2QuIFVuZGVmaW5lZCBtZWFucyBpdHMgc2V0IHRvIGEgc3lzdGVtLWRlZmluZWQgdmFsdWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaG9zdG5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25maWd1cmVkIHN1YmRvbWFpbiBvZiB0aGUgcG9kLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN1YmRvbWFpbj86IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhlIHBvZHMgaG9zdG5hbWUgaXMgc2V0IHRvIGl0cyBGUUROLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGhvc3RuYW1lQXNGUUROOiBib29sZWFuO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX25hbWVzZXJ2ZXJzOiBzdHJpbmdbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBfc2VhcmNoZXM6IHN0cmluZ1tdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9vcHRpb25zOiBEbnNPcHRpb25bXTtcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogUG9kRG5zUHJvcHMgPSB7fSkge1xuICAgIHRoaXMuaG9zdG5hbWUgPSBwcm9wcy5ob3N0bmFtZTtcbiAgICB0aGlzLnN1YmRvbWFpbiA9IHByb3BzLnN1YmRvbWFpbjtcbiAgICB0aGlzLnBvbGljeSA9IHByb3BzLnBvbGljeSA/PyBEbnNQb2xpY3kuQ0xVU1RFUl9GSVJTVDtcbiAgICB0aGlzLmhvc3RuYW1lQXNGUUROID0gcHJvcHMuaG9zdG5hbWVBc0ZRRE4gPz8gZmFsc2U7XG4gICAgdGhpcy5fbmFtZXNlcnZlcnMgPSBwcm9wcy5uYW1lc2VydmVycyA/PyBbXTtcbiAgICB0aGlzLl9zZWFyY2hlcyA9IHByb3BzLnNlYXJjaGVzID8/IFtdO1xuICAgIHRoaXMuX29wdGlvbnMgPSBwcm9wcy5vcHRpb25zID8/IFtdO1xuICB9XG5cbiAgLyoqXG4gICAqIE5hbWVzZXJ2ZXJzIGRlZmluZWQgZm9yIHRoaXMgcG9kLlxuICAgKi9cbiAgcHVibGljIGdldCBuYW1lc2VydmVycygpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIFsuLi50aGlzLl9uYW1lc2VydmVyc107XG4gIH1cblxuICAvKipcbiAgICogU2VhcmNoIGRvbWFpbnMgZGVmaW5lZCBmb3IgdGhpcyBwb2QuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHNlYXJjaGVzKCk6IHN0cmluZ1tdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX3NlYXJjaGVzXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDdXN0b20gZG5zIG9wdGlvbnMgZGVmaW5lZCBmb3IgdGhpcyBwb2QuXG4gICAqL1xuICBwdWJsaWMgZ2V0IG9wdGlvbnMoKTogRG5zT3B0aW9uW10ge1xuICAgIHJldHVybiBbLi4udGhpcy5fb3B0aW9uc107XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgbmFtZXNlcnZlci5cbiAgICovXG4gIHB1YmxpYyBhZGROYW1lc2VydmVyKC4uLm5hbWVzZXJ2ZXJzOiBzdHJpbmdbXSkge1xuICAgIHRoaXMuX25hbWVzZXJ2ZXJzLnB1c2goLi4ubmFtZXNlcnZlcnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHNlYXJjaCBkb21haW4uXG4gICAqL1xuICBwdWJsaWMgYWRkU2VhcmNoKC4uLnNlYXJjaGVzOiBzdHJpbmdbXSkge1xuICAgIHRoaXMuX3NlYXJjaGVzLnB1c2goLi4uc2VhcmNoZXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGN1c3RvbSBvcHRpb24uXG4gICAqL1xuICBwdWJsaWMgYWRkT3B0aW9uKC4uLm9wdGlvbnM6IERuc09wdGlvbltdKSB7XG4gICAgdGhpcy5fb3B0aW9ucy5wdXNoKC4uLm9wdGlvbnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogeyBob3N0bmFtZT86IHN0cmluZzsgc3ViZG9tYWluPzogc3RyaW5nOyBob3N0bmFtZUFzRlFETjogYm9vbGVhbjsgcG9saWN5OiBzdHJpbmc7IGNvbmZpZzogazhzLlBvZERuc0NvbmZpZyB9IHtcblxuICAgIGlmICh0aGlzLnBvbGljeSA9PT0gRG5zUG9saWN5Lk5PTkUgJiYgdGhpcy5uYW1lc2VydmVycy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignV2hlbiBkbnMgcG9saWN5IGlzIHNldCB0byBOT05FLCBhdCBsZWFzdCBvbmUgbmFtZXNlcnZlciBpcyByZXF1aXJlZCcpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLm5hbWVzZXJ2ZXJzLmxlbmd0aCA+IDMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlcmUgY2FuIGJlIGF0IG1vc3QgMyBuYW1lc2VydmVycyBzcGVjaWZpZWQnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5zZWFyY2hlcy5sZW5ndGggPiA2KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZXJlIGNhbiBiZSBhdCBtb3N0IDYgc2VhcmNoIGRvbWFpbnMgc3BlY2lmaWVkJyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIGhvc3RuYW1lOiB0aGlzLmhvc3RuYW1lLFxuICAgICAgc3ViZG9tYWluOiB0aGlzLnN1YmRvbWFpbixcbiAgICAgIGhvc3RuYW1lQXNGUUROOiB0aGlzLmhvc3RuYW1lQXNGUUROLFxuICAgICAgcG9saWN5OiB0aGlzLnBvbGljeSxcbiAgICAgIGNvbmZpZzoge1xuICAgICAgICBuYW1lc2VydmVyczogdW5kZWZpbmVkSWZFbXB0eSh0aGlzLm5hbWVzZXJ2ZXJzKSxcbiAgICAgICAgc2VhcmNoZXM6IHVuZGVmaW5lZElmRW1wdHkodGhpcy5zZWFyY2hlcyksXG4gICAgICAgIG9wdGlvbnM6IHVuZGVmaW5lZElmRW1wdHkodGhpcy5vcHRpb25zKSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG59XG5cbi8qKlxuICogSG9sZHMgcG9kLWxldmVsIHNlY3VyaXR5IGF0dHJpYnV0ZXMgYW5kIGNvbW1vbiBjb250YWluZXIgc2V0dGluZ3MuXG4gKi9cbmV4cG9ydCBjbGFzcyBQb2RTZWN1cml0eUNvbnRleHQge1xuXG4gIHB1YmxpYyByZWFkb25seSBlbnN1cmVOb25Sb290OiBib29sZWFuO1xuICBwdWJsaWMgcmVhZG9ubHkgdXNlcj86IG51bWJlcjtcbiAgcHVibGljIHJlYWRvbmx5IGdyb3VwPzogbnVtYmVyO1xuICBwdWJsaWMgcmVhZG9ubHkgZnNHcm91cD86IG51bWJlcjtcbiAgcHVibGljIHJlYWRvbmx5IGZzR3JvdXBDaGFuZ2VQb2xpY3k6IEZzR3JvdXBDaGFuZ2VQb2xpY3k7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfc3lzY3RsczogU3lzY3RsW10gPSBbXTtcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogUG9kU2VjdXJpdHlDb250ZXh0UHJvcHMgPSB7fSkge1xuICAgIHRoaXMuZW5zdXJlTm9uUm9vdCA9IHByb3BzLmVuc3VyZU5vblJvb3QgPz8gZmFsc2U7XG4gICAgdGhpcy5mc0dyb3VwQ2hhbmdlUG9saWN5ID0gcHJvcHMuZnNHcm91cENoYW5nZVBvbGljeSA/PyBGc0dyb3VwQ2hhbmdlUG9saWN5LkFMV0FZUztcbiAgICB0aGlzLnVzZXIgPSBwcm9wcy51c2VyO1xuICAgIHRoaXMuZ3JvdXAgPSBwcm9wcy5ncm91cDtcbiAgICB0aGlzLmZzR3JvdXAgPSBwcm9wcy5mc0dyb3VwO1xuXG4gICAgZm9yIChjb25zdCBzeXNjdGwgb2YgcHJvcHMuc3lzY3RscyA/PyBbXSkge1xuICAgICAgdGhpcy5fc3lzY3Rscy5wdXNoKHN5c2N0bCk7XG4gICAgfVxuXG4gIH1cblxuICBwdWJsaWMgZ2V0IHN5c2N0bHMoKTogU3lzY3RsW10ge1xuICAgIHJldHVybiBbLi4udGhpcy5fc3lzY3Rsc107XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiBrOHMuUG9kU2VjdXJpdHlDb250ZXh0IHtcbiAgICByZXR1cm4ge1xuICAgICAgcnVuQXNHcm91cDogdGhpcy5ncm91cCxcbiAgICAgIHJ1bkFzVXNlcjogdGhpcy51c2VyLFxuICAgICAgZnNHcm91cDogdGhpcy5mc0dyb3VwLFxuICAgICAgcnVuQXNOb25Sb290OiB0aGlzLmVuc3VyZU5vblJvb3QsXG4gICAgICBmc0dyb3VwQ2hhbmdlUG9saWN5OiB0aGlzLmZzR3JvdXBDaGFuZ2VQb2xpY3ksXG4gICAgICBzeXNjdGxzOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMuX3N5c2N0bHMpLFxuICAgIH07XG4gIH1cblxufVxuXG4vKipcbiAqIFJlc3RhcnQgcG9saWN5IGZvciBhbGwgY29udGFpbmVycyB3aXRoaW4gdGhlIHBvZC5cbiAqL1xuZXhwb3J0IGVudW0gUmVzdGFydFBvbGljeSB7XG4gIC8qKlxuICAgKiBBbHdheXMgcmVzdGFydCB0aGUgcG9kIGFmdGVyIGl0IGV4aXRzLlxuICAgKi9cbiAgQUxXQVlTID0gJ0Fsd2F5cycsXG5cbiAgLyoqXG4gICAqIE9ubHkgcmVzdGFydCBpZiB0aGUgcG9kIGV4aXRzIHdpdGggYSBub24temVybyBleGl0IGNvZGUuXG4gICAqL1xuICBPTl9GQUlMVVJFID0gJ09uRmFpbHVyZScsXG5cbiAgLyoqXG4gICAqIE5ldmVyIHJlc3RhcnQgdGhlIHBvZC5cbiAgICovXG4gIE5FVkVSID0gJ05ldmVyJ1xufVxuXG5leHBvcnQgZW51bSBGc0dyb3VwQ2hhbmdlUG9saWN5IHtcblxuICAvKipcbiAgICogT25seSBjaGFuZ2UgcGVybWlzc2lvbnMgYW5kIG93bmVyc2hpcCBpZiBwZXJtaXNzaW9uIGFuZCBvd25lcnNoaXAgb2Ygcm9vdCBkaXJlY3RvcnkgZG9lc1xuICAgKiBub3QgbWF0Y2ggd2l0aCBleHBlY3RlZCBwZXJtaXNzaW9ucyBvZiB0aGUgdm9sdW1lLlxuICAgKiBUaGlzIGNvdWxkIGhlbHAgc2hvcnRlbiB0aGUgdGltZSBpdCB0YWtlcyB0byBjaGFuZ2Ugb3duZXJzaGlwIGFuZCBwZXJtaXNzaW9uIG9mIGEgdm9sdW1lXG4gICAqL1xuICBPTl9ST09UX01JU01BVENIID0gJ09uUm9vdE1pc21hdGNoJyxcblxuICAvKipcbiAgICogQWx3YXlzIGNoYW5nZSBwZXJtaXNzaW9uIGFuZCBvd25lcnNoaXAgb2YgdGhlIHZvbHVtZSB3aGVuIHZvbHVtZSBpcyBtb3VudGVkLlxuICAgKi9cbiAgQUxXQVlTID0gJ0Fsd2F5cydcbn1cblxuLyoqXG4gKiBDdXN0b20gRE5TIG9wdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEbnNPcHRpb24ge1xuXG4gIC8qKlxuICAgKiBPcHRpb24gbmFtZS5cbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogT3B0aW9uIHZhbHVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHZhbHVlLlxuICAgKi9cbiAgcmVhZG9ubHkgdmFsdWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogUG9kIEROUyBwb2xpY2llcy5cbiAqL1xuZXhwb3J0IGVudW0gRG5zUG9saWN5IHtcblxuICAvKipcbiAgICogQW55IEROUyBxdWVyeSB0aGF0IGRvZXMgbm90IG1hdGNoIHRoZSBjb25maWd1cmVkIGNsdXN0ZXIgZG9tYWluIHN1ZmZpeCxcbiAgICogc3VjaCBhcyBcInd3dy5rdWJlcm5ldGVzLmlvXCIsIGlzIGZvcndhcmRlZCB0byB0aGVcbiAgICogdXBzdHJlYW0gbmFtZXNlcnZlciBpbmhlcml0ZWQgZnJvbSB0aGUgbm9kZS5cbiAgICogQ2x1c3RlciBhZG1pbmlzdHJhdG9ycyBtYXkgaGF2ZSBleHRyYSBzdHViLWRvbWFpbiBhbmQgdXBzdHJlYW0gRE5TIHNlcnZlcnMgY29uZmlndXJlZC5cbiAgICovXG4gIENMVVNURVJfRklSU1QgPSAnQ2x1c3RlckZpcnN0JyxcblxuICAvKipcbiAgICogRm9yIFBvZHMgcnVubmluZyB3aXRoIGhvc3ROZXR3b3JrLCB5b3Ugc2hvdWxkXG4gICAqIGV4cGxpY2l0bHkgc2V0IGl0cyBETlMgcG9saWN5IFwiQ2x1c3RlckZpcnN0V2l0aEhvc3ROZXRcIi5cbiAgICovXG4gIENMVVNURVJfRklSU1RfV0lUSF9IT1NUX05FVCA9ICdDbHVzdGVyRmlyc3RXaXRoSG9zdE5ldCcsXG5cbiAgLyoqXG4gICAqIFRoZSBQb2QgaW5oZXJpdHMgdGhlIG5hbWUgcmVzb2x1dGlvbiBjb25maWd1cmF0aW9uXG4gICAqIGZyb20gdGhlIG5vZGUgdGhhdCB0aGUgcG9kcyBydW4gb24uXG4gICAqL1xuICBERUZBVUxUID0gJ0RlZmF1bHQnLFxuXG4gIC8qKlxuICAgKiBJdCBhbGxvd3MgYSBQb2QgdG8gaWdub3JlIEROUyBzZXR0aW5ncyBmcm9tIHRoZSBLdWJlcm5ldGVzIGVudmlyb25tZW50LlxuICAgKiBBbGwgRE5TIHNldHRpbmdzIGFyZSBzdXBwb3NlZCB0byBiZSBwcm92aWRlZCB1c2luZyB0aGUgZG5zQ29uZmlnXG4gICAqIGZpZWxkIGluIHRoZSBQb2QgU3BlYy5cbiAgICovXG4gIE5PTkUgPSAnTm9uZScsXG5cbn1cblxuLyoqXG4gKiBIb3N0QWxpYXMgaG9sZHMgdGhlIG1hcHBpbmcgYmV0d2VlbiBJUCBhbmQgaG9zdG5hbWVzIHRoYXQgd2lsbCBiZSBpbmplY3RlZCBhc1xuICogYW4gZW50cnkgaW4gdGhlIHBvZCdzIC9ldGMvaG9zdHMgZmlsZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBIb3N0QWxpYXMge1xuICAvKipcbiAgICogSG9zdG5hbWVzIGZvciB0aGUgY2hvc2VuIElQIGFkZHJlc3MuXG4gICAqL1xuICByZWFkb25seSBob3N0bmFtZXM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBJUCBhZGRyZXNzIG9mIHRoZSBob3N0IGZpbGUgZW50cnkuXG4gICAqL1xuICByZWFkb25seSBpcDogc3RyaW5nO1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBxdWVyeSB0aGF0IGNhbiBiZSBwZXJmb3JtZWQgYWdhaW5zdCBub2RlcyB3aXRoIGxhYmVscy5cbiAqL1xuZXhwb3J0IGNsYXNzIE5vZGVMYWJlbFF1ZXJ5IHtcblxuICAvKipcbiAgICogUmVxdWlyZXMgdmFsdWUgb2YgbGFiZWwgYGtleWAgdG8gZXF1YWwgYHZhbHVlYC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgaXMoa2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gTm9kZUxhYmVsUXVlcnkuaW4oa2V5LCBbdmFsdWVdKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXF1aXJlcyB2YWx1ZSBvZiBsYWJlbCBga2V5YCB0byBiZSBvbmUgb2YgYHZhbHVlc2AuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGluKGtleTogc3RyaW5nLCB2YWx1ZXM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIG5ldyBOb2RlTGFiZWxRdWVyeShrZXksICdJbicsIHZhbHVlcyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdmFsdWUgb2YgbGFiZWwgYGtleWAgdG8gYmUgbm9uZSBvZiBgdmFsdWVzYC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgbm90SW4oa2V5OiBzdHJpbmcsIHZhbHVlczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gbmV3IE5vZGVMYWJlbFF1ZXJ5KGtleSwgJ05vdEluJywgdmFsdWVzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXF1aXJlcyBsYWJlbCBga2V5YCB0byBleGlzdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZXhpc3RzKGtleTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBOb2RlTGFiZWxRdWVyeShrZXksICdFeGlzdHMnLCB1bmRlZmluZWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVpcmVzIGxhYmVsIGBrZXlgIHRvIG5vdCBleGlzdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZG9lc05vdEV4aXN0KGtleTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBOb2RlTGFiZWxRdWVyeShrZXksICdEb2VzTm90RXhpc3QnLCB1bmRlZmluZWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcXVpcmVzIHZhbHVlIG9mIGxhYmVsIGBrZXlgIHRvIGdyZWF0ZXIgdGhhbiBhbGwgZWxlbWVudHMgaW4gYHZhbHVlc2AuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGd0KGtleTogc3RyaW5nLCB2YWx1ZXM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIG5ldyBOb2RlTGFiZWxRdWVyeShrZXksICdHdCcsIHZhbHVlcyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdmFsdWUgb2YgbGFiZWwgYGtleWAgdG8gbGVzcyB0aGFuIGFsbCBlbGVtZW50cyBpbiBgdmFsdWVzYC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgbHQoa2V5OiBzdHJpbmcsIHZhbHVlczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gbmV3IE5vZGVMYWJlbFF1ZXJ5KGtleSwgJ0x0JywgdmFsdWVzKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoXG4gICAgcHVibGljIHJlYWRvbmx5IGtleTogc3RyaW5nLFxuICAgIHB1YmxpYyByZWFkb25seSBvcGVyYXRvcjogc3RyaW5nLFxuICAgIHB1YmxpYyByZWFkb25seSB2YWx1ZXM/OiBzdHJpbmdbXSkge1xuICB9XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIHF1ZXJ5IHRoYXQgY2FuIGJlIHBlcmZvcm1lZCBhZ2FpbnN0IHJlc291cmNlcyB3aXRoIGxhYmVscy5cbiAqL1xuZXhwb3J0IGNsYXNzIExhYmVsRXhwcmVzc2lvbiB7XG5cbiAgLyoqXG4gICAqIFJlcXVpcmVzIHZhbHVlIG9mIGxhYmVsIGBrZXlgIHRvIGJlIG9uZSBvZiBgdmFsdWVzYC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgaW4oa2V5OiBzdHJpbmcsIHZhbHVlczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gbmV3IExhYmVsRXhwcmVzc2lvbihrZXksICdJbicsIHZhbHVlcyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgdmFsdWUgb2YgbGFiZWwgYGtleWAgdG8gYmUgbm9uZSBvZiBgdmFsdWVzYC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgbm90SW4oa2V5OiBzdHJpbmcsIHZhbHVlczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gbmV3IExhYmVsRXhwcmVzc2lvbihrZXksICdOb3RJbicsIHZhbHVlcyk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgbGFiZWwgYGtleWAgdG8gZXhpc3QuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGV4aXN0cyhrZXk6IHN0cmluZykge1xuICAgIHJldHVybiBuZXcgTGFiZWxFeHByZXNzaW9uKGtleSwgJ0V4aXN0cycsIHVuZGVmaW5lZCk7XG4gIH1cblxuICAvKipcbiAgICogUmVxdWlyZXMgbGFiZWwgYGtleWAgdG8gbm90IGV4aXN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBkb2VzTm90RXhpc3Qoa2V5OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gbmV3IExhYmVsRXhwcmVzc2lvbihrZXksICdEb2VzTm90RXhpc3QnLCB1bmRlZmluZWQpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihcbiAgICBwdWJsaWMgcmVhZG9ubHkga2V5OiBzdHJpbmcsXG4gICAgcHVibGljIHJlYWRvbmx5IG9wZXJhdG9yOiBzdHJpbmcsXG4gICAgcHVibGljIHJlYWRvbmx5IHZhbHVlcz86IHN0cmluZ1tdKSB7XG4gIH1cblxufVxuXG4vKipcbiAqIFRhaW50IGVmZmVjdHMuXG4gKi9cbmV4cG9ydCBlbnVtIFRhaW50RWZmZWN0IHtcbiAgLyoqXG4gICAqIFRoaXMgbWVhbnMgdGhhdCBubyBwb2Qgd2lsbCBiZSBhYmxlIHRvIHNjaGVkdWxlXG4gICAqIG9udG8gdGhlIG5vZGUgdW5sZXNzIGl0IGhhcyBhIG1hdGNoaW5nIHRvbGVyYXRpb24uXG4gICAqL1xuICBOT19TQ0hFRFVMRSA9ICdOb1NjaGVkdWxlJyxcblxuICAvKipcbiAgICogVGhpcyBpcyBhIFwicHJlZmVyZW5jZVwiIG9yIFwic29mdFwiIHZlcnNpb24gb2YgYE5PX1NDSEVEVUxFYCAtLSB0aGUgc3lzdGVtXG4gICAqIHdpbGwgdHJ5IHRvIGF2b2lkIHBsYWNpbmcgYSBwb2QgdGhhdCBkb2VzIG5vdCB0b2xlcmF0ZSB0aGUgdGFpbnQgb24gdGhlIG5vZGUsXG4gICAqIGJ1dCBpdCBpcyBub3QgcmVxdWlyZWRcbiAgICovXG4gIFBSRUZFUl9OT19TQ0hFRFVMRSA9ICdQcmVmZXJOb1NjaGVkdWxlJyxcblxuICAvKipcbiAgICogVGhpcyBhZmZlY3RzIHBvZHMgdGhhdCBhcmUgYWxyZWFkeSBydW5uaW5nIG9uIHRoZSBub2RlIGFzIGZvbGxvd3M6XG4gICAqXG4gICAqIC0gUG9kcyB0aGF0IGRvIG5vdCB0b2xlcmF0ZSB0aGUgdGFpbnQgYXJlIGV2aWN0ZWQgaW1tZWRpYXRlbHkuXG4gICAqIC0gUG9kcyB0aGF0IHRvbGVyYXRlIHRoZSB0YWludCB3aXRob3V0IHNwZWNpZnlpbmcgYGR1cmF0aW9uYCByZW1haW4gYm91bmQgZm9yZXZlci5cbiAgICogLSBQb2RzIHRoYXQgdG9sZXJhdGUgdGhlIHRhaW50IHdpdGggYSBzcGVjaWZpZWQgYGR1cmF0aW9uYCByZW1haW4gYm91bmQgZm9yXG4gICAqICAgdGhlIHNwZWNpZmllZCBhbW91bnQgb2YgdGltZS5cbiAgICovXG4gIE5PX0VYRUNVVEUgPSAnTm9FeGVjdXRlJyxcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgTm9kZVRhaW50UXVlcnlgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5vZGVUYWludFF1ZXJ5T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgdGFpbnQgZWZmZWN0IHRvIG1hdGNoLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGFsbCBlZmZlY3RzIGFyZSBtYXRjaGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZWZmZWN0PzogVGFpbnRFZmZlY3Q7XG5cbiAgLyoqXG4gICAqIEhvdyBtdWNoIHRpbWUgc2hvdWxkIGEgcG9kIHRoYXQgdG9sZXJhdGVzIHRoZSBgTk9fRVhFQ1VURWAgZWZmZWN0XG4gICAqIGJlIGJvdW5kIHRvIHRoZSBub2RlLiBPbmx5IGFwcGxpZXMgZm9yIHRoZSBgTk9fRVhFQ1VURWAgZWZmZWN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGJvdW5kIGZvcmV2ZXIuXG4gICAqL1xuICByZWFkb25seSBldmljdEFmdGVyPzogRHVyYXRpb247XG59XG5cbi8qKlxuICogVGFpbnQgcXVlcmllcyB0aGF0IGNhbiBiZSBwZXJmb21lZCBhZ2FpbnN0IG5vZGVzLlxuICovXG5leHBvcnQgY2xhc3MgTm9kZVRhaW50UXVlcnkge1xuXG4gIC8qKlxuICAgKiBNYXRjaGVzIGEgdGFpbnQgd2l0aCBhIHNwZWNpZmljIGtleSBhbmQgdmFsdWUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGlzKGtleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nLCBvcHRpb25zOiBOb2RlVGFpbnRRdWVyeU9wdGlvbnMgPSB7fSk6IE5vZGVUYWludFF1ZXJ5IHtcbiAgICByZXR1cm4gbmV3IE5vZGVUYWludFF1ZXJ5KCdFcXVhbCcsIGtleSwgdmFsdWUsIG9wdGlvbnMuZWZmZWN0LCBvcHRpb25zLmV2aWN0QWZ0ZXIpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgYSB0YWluIHdpdGggYW55IHZhbHVlIG9mIGEgc3BlY2lmaWMga2V5LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBleGlzdHMoa2V5OiBzdHJpbmcsIG9wdGlvbnM6IE5vZGVUYWludFF1ZXJ5T3B0aW9ucyA9IHt9KTogTm9kZVRhaW50UXVlcnkge1xuICAgIHJldHVybiBuZXcgTm9kZVRhaW50UXVlcnkoJ0V4aXN0cycsIGtleSwgdW5kZWZpbmVkLCBvcHRpb25zLmVmZmVjdCwgb3B0aW9ucy5ldmljdEFmdGVyKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaGVzIGFueSB0YWludC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYW55KCk6IE5vZGVUYWludFF1ZXJ5IHtcbiAgICByZXR1cm4gbmV3IE5vZGVUYWludFF1ZXJ5KCdFeGlzdHMnKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoXG4gICAgcHVibGljIHJlYWRvbmx5IG9wZXJhdG9yOiBzdHJpbmcsXG4gICAgcHVibGljIHJlYWRvbmx5IGtleT86IHN0cmluZyxcbiAgICBwdWJsaWMgcmVhZG9ubHkgdmFsdWU/OiBzdHJpbmcsXG4gICAgcHVibGljIHJlYWRvbmx5IGVmZmVjdD86IHN0cmluZyxcbiAgICBwdWJsaWMgcmVhZG9ubHkgZXZpY3RBZnRlcj86IER1cmF0aW9uLFxuICApIHtcbiAgICBpZiAoZXZpY3RBZnRlciAmJiBlZmZlY3QgIT09IFRhaW50RWZmZWN0Lk5PX0VYRUNVVEUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBcXCdOT19FWEVDVVRFXFwnIGVmZmVjdHMgY2FuIHNwZWNpZnkgXFwnZXZpY3RBZnRlclxcJycpO1xuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBQb2RzLmFsbGAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUG9kc0FsbE9wdGlvbnMge1xuXG4gIC8qKlxuICAgKiBOYW1lc3BhY2VzIHRoZSBwb2RzIGFyZSBhbGxvd2VkIHRvIGJlIGluLlxuICAgKiBVc2UgYE5hbWVzcGFjZXMuYWxsKClgIHRvIGFsbG93IGFsbCBuYW1lc3BhY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHVuc2V0LCBpbXBsaWVzIHRoZSBuYW1lc3BhY2Ugb2YgdGhlIHJlc291cmNlIHRoaXMgc2VsZWN0aW9uIGlzIHVzZWQgaW4uXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VzPzogbmFtZXNwYWNlLk5hbWVzcGFjZXM7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYFBvZHMuc2VsZWN0YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb2RzU2VsZWN0T3B0aW9ucyB7XG5cbiAgLyoqXG4gICAqIExhYmVscyB0aGUgcG9kcyBtdXN0IGhhdmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gc3RyaWN0IGxhYmVscyByZXF1aXJlbWVudHMuXG4gICAqL1xuICByZWFkb25seSBsYWJlbHM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgICogRXhwcmVzc2lvbnMgdGhlIHBvZHMgbXVzdCBzYXRpc2lmeS5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCAtIG5vIGV4cHJlc3Npb25zIHJlcXVpcmVtZW50cy5cbiAgICAqL1xuICByZWFkb25seSBleHByZXNzaW9ucz86IExhYmVsRXhwcmVzc2lvbltdO1xuXG4gIC8qKlxuICAgKiBOYW1lc3BhY2VzIHRoZSBwb2RzIGFyZSBhbGxvd2VkIHRvIGJlIGluLlxuICAgKiBVc2UgYE5hbWVzcGFjZXMuYWxsKClgIHRvIGFsbG93IGFsbCBuYW1lc3BhY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHVuc2V0LCBpbXBsaWVzIHRoZSBuYW1lc3BhY2Ugb2YgdGhlIHJlc291cmNlIHRoaXMgc2VsZWN0aW9uIGlzIHVzZWQgaW4uXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2VzPzogbmFtZXNwYWNlLk5hbWVzcGFjZXM7XG5cbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgZ3JvdXAgb2YgcG9kcy5cbiAqL1xuZXhwb3J0IGNsYXNzIFBvZHMgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJUG9kU2VsZWN0b3Ige1xuXG4gIC8qKlxuICAgKiBTZWxlY3QgcG9kcyBpbiB0aGUgY2x1c3RlciB3aXRoIHZhcmlvdXMgc2VsZWN0b3JzLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBzZWxlY3Qoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgb3B0aW9uczogUG9kc1NlbGVjdE9wdGlvbnMpOiBQb2RzIHtcbiAgICByZXR1cm4gbmV3IFBvZHMoc2NvcGUsIGlkLCBvcHRpb25zLmV4cHJlc3Npb25zLCBvcHRpb25zLmxhYmVscywgb3B0aW9ucy5uYW1lc3BhY2VzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTZWxlY3QgYWxsIHBvZHMuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFsbChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBvcHRpb25zOiBQb2RzQWxsT3B0aW9ucyA9IHt9KSB7XG4gICAgcmV0dXJuIFBvZHMuc2VsZWN0KHNjb3BlLCBpZCwgeyBuYW1lc3BhY2VzOiBvcHRpb25zLm5hbWVzcGFjZXMgfSk7XG4gIH1cblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgZXhwcmVzc2lvbnM/OiBMYWJlbEV4cHJlc3Npb25bXSxcbiAgICBwcml2YXRlIHJlYWRvbmx5IGxhYmVscz86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0sXG4gICAgcHJpdmF0ZSByZWFkb25seSBuYW1lc3BhY2VzPzogbmFtZXNwYWNlLklOYW1lc3BhY2VTZWxlY3Rvcikge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKipcbiAgICogQHNlZSBJUG9kU2VsZWN0b3IudG9Qb2RTZWxlY3RvckNvbmZpZygpXG4gICAqL1xuICBwdWJsaWMgdG9Qb2RTZWxlY3RvckNvbmZpZygpOiBQb2RTZWxlY3RvckNvbmZpZyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGxhYmVsU2VsZWN0b3I6IExhYmVsU2VsZWN0b3Iub2YoeyBleHByZXNzaW9uczogdGhpcy5leHByZXNzaW9ucywgbGFiZWxzOiB0aGlzLmxhYmVscyB9KSxcbiAgICAgIG5hbWVzcGFjZXM6IHRoaXMubmFtZXNwYWNlcz8udG9OYW1lc3BhY2VTZWxlY3RvckNvbmZpZygpLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQHNlZSBJTmV0d29ya1BvbGljeVBlZXIudG9OZXR3b3JrUG9saWN5UGVlckNvbmZpZygpXG4gICAqL1xuICBwdWJsaWMgdG9OZXR3b3JrUG9saWN5UGVlckNvbmZpZygpOiBuZXR3b3JrcG9saWN5Lk5ldHdvcmtQb2xpY3lQZWVyQ29uZmlnIHtcbiAgICByZXR1cm4geyBwb2RTZWxlY3RvcjogdGhpcy50b1BvZFNlbGVjdG9yQ29uZmlnKCkgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAc2VlIElOZXR3b3JrUG9saWN5UGVlci50b1BvZFNlbGVjdG9yKClcbiAgICovXG4gIHB1YmxpYyB0b1BvZFNlbGVjdG9yKCk6IElQb2RTZWxlY3RvciB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxufVxuXG4vKipcbiAqIEEgbm9kZSB0aGF0IGlzIG1hdGNoZWQgYnkgbGFiZWwgc2VsZWN0b3JzLlxuICovXG5leHBvcnQgY2xhc3MgTGFiZWxlZE5vZGUge1xuICBwdWJsaWMgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IGxhYmVsU2VsZWN0b3I6IE5vZGVMYWJlbFF1ZXJ5W10pIHt9O1xufVxuXG4vKipcbiAqIEEgbm9kZSB0aGF0IGlzIG1hdGNoZWQgYnkgdGFpbnQgc2VsZWN0b3JzLlxuICovXG5leHBvcnQgY2xhc3MgVGFpbnRlZE5vZGUge1xuICBwdWJsaWMgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHRhaW50U2VsZWN0b3I6IE5vZGVUYWludFF1ZXJ5W10pIHt9O1xufVxuXG4vKipcbiAqIEEgbm9kZSB0aGF0IGlzIG1hdGNoZWQgYnkgaXRzIG5hbWUuXG4gKi9cbmV4cG9ydCBjbGFzcyBOYW1lZE5vZGUge1xuICBwdWJsaWMgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZykge307XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIG5vZGUgaW4gdGhlIGNsdXN0ZXIuXG4gKi9cbmV4cG9ydCBjbGFzcyBOb2RlIHtcblxuICAvKipcbiAgICogTWF0Y2ggYSBub2RlIGJ5IGl0cyBsYWJlbHMuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGxhYmVsZWQoLi4ubGFiZWxTZWxlY3RvcjogTm9kZUxhYmVsUXVlcnlbXSk6IExhYmVsZWROb2RlIHtcbiAgICByZXR1cm4gbmV3IExhYmVsZWROb2RlKGxhYmVsU2VsZWN0b3IpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoIGEgbm9kZSBieSBpdHMgbmFtZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgbmFtZWQobm9kZU5hbWU6IHN0cmluZyk6IE5hbWVkTm9kZSB7XG4gICAgcmV0dXJuIG5ldyBOYW1lZE5vZGUobm9kZU5hbWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoIGEgbm9kZSBieSBpdHMgdGFpbnRzLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyB0YWludGVkKC4uLnRhaW50U2VsZWN0b3I6IE5vZGVUYWludFF1ZXJ5W10pOiBUYWludGVkTm9kZSB7XG4gICAgcmV0dXJuIG5ldyBUYWludGVkTm9kZSh0YWludFNlbGVjdG9yKTtcbiAgfVxuXG59XG5cbi8qKlxuICogQXZhaWxhYmxlIHRvcG9sb2d5IGRvbWFpbnMuXG4gKi9cbmV4cG9ydCBjbGFzcyBUb3BvbG9neSB7XG5cbiAgLyoqXG4gICAqIEEgaG9zdG5hbWUgcmVwcmVzZW50cyBhIHNpbmdsZSBub2RlIGluIHRoZSBjbHVzdGVyLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL3JlZmVyZW5jZS9sYWJlbHMtYW5ub3RhdGlvbnMtdGFpbnRzLyNrdWJlcm5ldGVzaW9ob3N0bmFtZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBIT1NUTkFNRSA9IG5ldyBUb3BvbG9neSgna3ViZXJuZXRlcy5pby9ob3N0bmFtZScpO1xuXG4gIC8qKlxuICAgKiBBIHpvbmUgcmVwcmVzZW50cyBhIGxvZ2ljYWwgZmFpbHVyZSBkb21haW4uIEl0IGlzIGNvbW1vbiBmb3IgS3ViZXJuZXRlcyBjbHVzdGVycyB0b1xuICAgKiBzcGFuIG11bHRpcGxlIHpvbmVzIGZvciBpbmNyZWFzZWQgYXZhaWxhYmlsaXR5LiBXaGlsZSB0aGUgZXhhY3QgZGVmaW5pdGlvbiBvZiBhIHpvbmUgaXNcbiAgICogbGVmdCB0byBpbmZyYXN0cnVjdHVyZSBpbXBsZW1lbnRhdGlvbnMsIGNvbW1vbiBwcm9wZXJ0aWVzIG9mIGEgem9uZSBpbmNsdWRlIHZlcnkgbG93XG4gICAqIG5ldHdvcmsgbGF0ZW5jeSB3aXRoaW4gYSB6b25lLCBuby1jb3N0IG5ldHdvcmsgdHJhZmZpYyB3aXRoaW4gYSB6b25lLCBhbmQgZmFpbHVyZVxuICAgKiBpbmRlcGVuZGVuY2UgZnJvbSBvdGhlciB6b25lcy4gRm9yIGV4YW1wbGUsIG5vZGVzIHdpdGhpbiBhIHpvbmUgbWlnaHQgc2hhcmUgYSBuZXR3b3JrXG4gICAqIHN3aXRjaCwgYnV0IG5vZGVzIGluIGRpZmZlcmVudCB6b25lcyBzaG91bGQgbm90LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL3JlZmVyZW5jZS9sYWJlbHMtYW5ub3RhdGlvbnMtdGFpbnRzLyN0b3BvbG9neWt1YmVybmV0ZXNpb3pvbmVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgWk9ORSA9IG5ldyBUb3BvbG9neSgndG9wb2xvZ3kua3ViZXJuZXRlcy5pby96b25lJyk7XG5cbiAgLyoqXG4gICAqIEEgcmVnaW9uIHJlcHJlc2VudHMgYSBsYXJnZXIgZG9tYWluLCBtYWRlIHVwIG9mIG9uZSBvciBtb3JlIHpvbmVzLiBJdCBpcyB1bmNvbW1vblxuICAgKiBmb3IgS3ViZXJuZXRlcyBjbHVzdGVycyB0byBzcGFuIG11bHRpcGxlIHJlZ2lvbnMuIFdoaWxlIHRoZSBleGFjdCBkZWZpbml0aW9uIG9mIGFcbiAgICogem9uZSBvciByZWdpb24gaXMgbGVmdCB0byBpbmZyYXN0cnVjdHVyZSBpbXBsZW1lbnRhdGlvbnMsIGNvbW1vbiBwcm9wZXJ0aWVzIG9mIGEgcmVnaW9uXG4gICAqIGluY2x1ZGUgaGlnaGVyIG5ldHdvcmsgbGF0ZW5jeSBiZXR3ZWVuIHRoZW0gdGhhbiB3aXRoaW4gdGhlbSwgbm9uLXplcm8gY29zdCBmb3IgbmV0d29ya1xuICAgKiB0cmFmZmljIGJldHdlZW4gdGhlbSwgYW5kIGZhaWx1cmUgaW5kZXBlbmRlbmNlIGZyb20gb3RoZXIgem9uZXMgb3IgcmVnaW9ucy5cbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIG5vZGVzIHdpdGhpbiBhIHJlZ2lvbiBtaWdodCBzaGFyZSBwb3dlciBpbmZyYXN0cnVjdHVyZSAoZS5nLiBhIFVQUyBvciBnZW5lcmF0b3IpLCBidXRcbiAgICogbm9kZXMgaW4gZGlmZmVyZW50IHJlZ2lvbnMgdHlwaWNhbGx5IHdvdWxkIG5vdC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9yZWZlcmVuY2UvbGFiZWxzLWFubm90YXRpb25zLXRhaW50cy8jdG9wb2xvZ3lrdWJlcm5ldGVzaW9yZWdpb25cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUkVHSU9OID0gbmV3IFRvcG9sb2d5KCd0b3BvbG9neS5rdWJlcm5ldGVzLmlvL3JlZ2lvbicpO1xuXG4gIC8qKlxuICAgKiBDdXN0b20ga2V5IGZvciB0aGUgbm9kZSBsYWJlbCB0aGF0IHRoZSBzeXN0ZW0gdXNlcyB0byBkZW5vdGUgdGhlIHRvcG9sb2d5IGRvbWFpbi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgY3VzdG9tKGtleTogc3RyaW5nKTogVG9wb2xvZ3kge1xuICAgIHJldHVybiBuZXcgVG9wb2xvZ3koa2V5KTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IGtleTogc3RyaW5nKSB7fTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgUG9kU2NoZWR1bGluZy5jb2xvY2F0ZWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUG9kU2NoZWR1bGluZ0NvbG9jYXRlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBXaGljaCB0b3BvbG9neSB0byBjb2xvYXRlIG9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRvcG9sb2d5LkhPU1ROQU1FXG4gICAqL1xuICByZWFkb25seSB0b3BvbG9neT86IFRvcG9sb2d5O1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgdGhlIGNvLWxvY2F0aW9uIGlzIG9wdGlvbmFsIChzb2Z0KSwgd2l0aCB0aGlzIHdlaWdodCBzY29yZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyB3ZWlnaHQuIGNvLWxvY2F0aW9uIGlzIGFzc3VtZWQgdG8gYmUgcmVxdWlyZWQgKGhhcmQpLlxuICAgKi9cbiAgcmVhZG9ubHkgd2VpZ2h0PzogbnVtYmVyO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBQb2RTY2hlZHVsaW5nLnNlcGFyYXRlYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb2RTY2hlZHVsaW5nU2VwYXJhdGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIFdoaWNoIHRvcG9sb2d5IHRvIHNlcGFyYXRlIG9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRvcG9sb2d5LkhPU1ROQU1FXG4gICAqL1xuICByZWFkb25seSB0b3BvbG9neT86IFRvcG9sb2d5O1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgdGhlIHNlcGFyYXRpb24gaXMgb3B0aW9uYWwgKHNvZnQpLCB3aXRoIHRoaXMgd2VpZ2h0IHNjb3JlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHdlaWdodC4gc2VwYXJhdGlvbiBpcyBhc3N1bWVkIHRvIGJlIHJlcXVpcmVkIChoYXJkKS5cbiAgICovXG4gIHJlYWRvbmx5IHdlaWdodD86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgUG9kU2NoZWR1bGluZy5hdHRyYWN0YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb2RTY2hlZHVsaW5nQXR0cmFjdE9wdGlvbnMge1xuICAvKipcbiAgICogSW5kaWNhdGVzIHRoZSBhdHRyYWN0aW9uIGlzIG9wdGlvbmFsIChzb2Z0KSwgd2l0aCB0aGlzIHdlaWdodCBzY29yZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyB3ZWlnaHQuIGFzc2lnbm1lbnQgaXMgYXNzdW1lZCB0byBiZSByZXF1aXJlZCAoaGFyZCkuXG4gICAqL1xuICByZWFkb25seSB3ZWlnaHQ/OiBudW1iZXI7XG59XG5cbi8qKlxuICogQ29udHJvbHMgdGhlIHBvZCBzY2hlZHVsaW5nIHN0cmF0ZWd5LlxuICovXG5leHBvcnQgY2xhc3MgUG9kU2NoZWR1bGluZyB7XG5cbiAgcHJpdmF0ZSBfbm9kZUFmZmluaXR5UHJlZmVycmVkOiBrOHMuUHJlZmVycmVkU2NoZWR1bGluZ1Rlcm1bXSA9IFtdO1xuICBwcml2YXRlIF9ub2RlQWZmaW5pdHlSZXF1aXJlZDogazhzLk5vZGVTZWxlY3RvclRlcm1bXSA9IFtdO1xuICBwcml2YXRlIF9wb2RBZmZpbml0eVByZWZlcnJlZDogazhzLldlaWdodGVkUG9kQWZmaW5pdHlUZXJtW10gPSBbXTtcbiAgcHJpdmF0ZSBfcG9kQWZmaW5pdHlSZXF1aXJlZDogazhzLlBvZEFmZmluaXR5VGVybVtdID0gW107XG4gIHByaXZhdGUgX3BvZEFudGlBZmZpbml0eVByZWZlcnJlZDogazhzLldlaWdodGVkUG9kQWZmaW5pdHlUZXJtW10gPSBbXTtcbiAgcHJpdmF0ZSBfcG9kQW50aUFmZmluaXR5UmVxdWlyZWQ6IGs4cy5Qb2RBZmZpbml0eVRlcm1bXSA9IFtdO1xuICBwcml2YXRlIF90b2xlcmF0aW9uczogazhzLlRvbGVyYXRpb25bXSA9IFtdO1xuICBwcml2YXRlIF9ub2RlTmFtZT86IHN0cmluZztcblxuICBjb25zdHJ1Y3Rvcihwcm90ZWN0ZWQgcmVhZG9ubHkgaW5zdGFuY2U6IEFic3RyYWN0UG9kKSB7fVxuXG4gIC8qKlxuICAgKiBBc3NpZ24gdGhpcyBwb2QgYSBzcGVjaWZpYyBub2RlIGJ5IG5hbWUuXG4gICAqXG4gICAqIFRoZSBzY2hlZHVsZXIgaWdub3JlcyB0aGUgUG9kLCBhbmQgdGhlIGt1YmVsZXQgb24gdGhlIG5hbWVkIG5vZGVcbiAgICogdHJpZXMgdG8gcGxhY2UgdGhlIFBvZCBvbiB0aGF0IG5vZGUuIE92ZXJydWxlcyBhbnkgYWZmaW5pdHkgcnVsZXMgb2YgdGhlIHBvZC5cbiAgICpcbiAgICogU29tZSBsaW1pdGF0aW9ucyBvZiBzdGF0aWMgYXNzaWdubWVudCBhcmU6XG4gICAqXG4gICAqIC0gSWYgdGhlIG5hbWVkIG5vZGUgZG9lcyBub3QgZXhpc3QsIHRoZSBQb2Qgd2lsbCBub3QgcnVuLCBhbmQgaW4gc29tZVxuICAgKiAgIGNhc2VzIG1heSBiZSBhdXRvbWF0aWNhbGx5IGRlbGV0ZWQuXG4gICAqIC0gSWYgdGhlIG5hbWVkIG5vZGUgZG9lcyBub3QgaGF2ZSB0aGUgcmVzb3VyY2VzIHRvIGFjY29tbW9kYXRlIHRoZSBQb2QsXG4gICAqICAgdGhlIFBvZCB3aWxsIGZhaWwgYW5kIGl0cyByZWFzb24gd2lsbCBpbmRpY2F0ZSB3aHksIGZvciBleGFtcGxlIE91dE9mbWVtb3J5IG9yIE91dE9mY3B1LlxuICAgKiAtIE5vZGUgbmFtZXMgaW4gY2xvdWQgZW52aXJvbm1lbnRzIGFyZSBub3QgYWx3YXlzIHByZWRpY3RhYmxlIG9yIHN0YWJsZS5cbiAgICpcbiAgICogV2lsbCB0aHJvdyBpcyB0aGUgcG9kIGlzIGFscmVhZHkgYXNzaWduZWQgdG8gbmFtZWQgbm9kZS5cbiAgICpcbiAgICogVW5kZXIgdGhlIGhvb2QsIHRoaXMgbWV0aG9kIHV0aWxpemVzIHRoZSBgbm9kZU5hbWVgIHByb3BlcnR5LlxuICAgKi9cbiAgcHVibGljIGFzc2lnbihub2RlOiBOYW1lZE5vZGUpIHtcblxuICAgIGlmICh0aGlzLl9ub2RlTmFtZSkge1xuICAgICAgLy8gZGlzYWxsb3cgb3ZlcnJpZGluZyBhbiBzdGF0aWMgbm9kZSBhc3NpZ25tZW50XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBhc3NpZ24gJHt0aGlzLmluc3RhbmNlLnBvZE1ldGFkYXRhLm5hbWV9IHRvIG5vZGUgJHtub2RlLm5hbWV9LiBJdCBpcyBhbHJlYWR5IGFzc2lnbmVkIHRvIG5vZGUgJHt0aGlzLl9ub2RlTmFtZX1gKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5fbm9kZU5hbWUgPSBub2RlLm5hbWU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93IHRoaXMgcG9kIHRvIHRvbGVyYXRlIHRhaW50cyBtYXRjaGluZyB0aGVzZSB0b2xlcmF0aW9ucy5cbiAgICpcbiAgICogWW91IGNhbiBwdXQgbXVsdGlwbGUgdGFpbnRzIG9uIHRoZSBzYW1lIG5vZGUgYW5kIG11bHRpcGxlIHRvbGVyYXRpb25zIG9uIHRoZSBzYW1lIHBvZC5cbiAgICogVGhlIHdheSBLdWJlcm5ldGVzIHByb2Nlc3NlcyBtdWx0aXBsZSB0YWludHMgYW5kIHRvbGVyYXRpb25zIGlzIGxpa2UgYSBmaWx0ZXI6IHN0YXJ0IHdpdGhcbiAgICogYWxsIG9mIGEgbm9kZSdzIHRhaW50cywgdGhlbiBpZ25vcmUgdGhlIG9uZXMgZm9yIHdoaWNoIHRoZSBwb2QgaGFzIGEgbWF0Y2hpbmcgdG9sZXJhdGlvbjtcbiAgICogdGhlIHJlbWFpbmluZyB1bi1pZ25vcmVkIHRhaW50cyBoYXZlIHRoZSBpbmRpY2F0ZWQgZWZmZWN0cyBvbiB0aGUgcG9kLiBJbiBwYXJ0aWN1bGFyOlxuICAgKlxuICAgKiAtIGlmIHRoZXJlIGlzIGF0IGxlYXN0IG9uZSB1bi1pZ25vcmVkIHRhaW50IHdpdGggZWZmZWN0IE5vU2NoZWR1bGUgdGhlbiBLdWJlcm5ldGVzIHdpbGxcbiAgICogICBub3Qgc2NoZWR1bGUgdGhlIHBvZCBvbnRvIHRoYXQgbm9kZVxuICAgKiAtIGlmIHRoZXJlIGlzIG5vIHVuLWlnbm9yZWQgdGFpbnQgd2l0aCBlZmZlY3QgTm9TY2hlZHVsZSBidXQgdGhlcmUgaXMgYXQgbGVhc3Qgb25lIHVuLWlnbm9yZWRcbiAgICogICB0YWludCB3aXRoIGVmZmVjdCBQcmVmZXJOb1NjaGVkdWxlIHRoZW4gS3ViZXJuZXRlcyB3aWxsIHRyeSB0byBub3Qgc2NoZWR1bGUgdGhlIHBvZCBvbnRvIHRoZSBub2RlXG4gICAqIC0gaWYgdGhlcmUgaXMgYXQgbGVhc3Qgb25lIHVuLWlnbm9yZWQgdGFpbnQgd2l0aCBlZmZlY3QgTm9FeGVjdXRlIHRoZW4gdGhlIHBvZCB3aWxsIGJlIGV2aWN0ZWQgZnJvbVxuICAgKiAgIHRoZSBub2RlIChpZiBpdCBpcyBhbHJlYWR5IHJ1bm5pbmcgb24gdGhlIG5vZGUpLCBhbmQgd2lsbCBub3QgYmUgc2NoZWR1bGVkIG9udG8gdGhlIG5vZGUgKGlmIGl0IGlzXG4gICAqICAgbm90IHlldCBydW5uaW5nIG9uIHRoZSBub2RlKS5cbiAgICpcbiAgICogVW5kZXIgdGhlIGhvb2QsIHRoaXMgbWV0aG9kIHV0aWxpemVzIHRoZSBgdG9sZXJhdGlvbnNgIHByb3BlcnR5LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3NjaGVkdWxpbmctZXZpY3Rpb24vdGFpbnQtYW5kLXRvbGVyYXRpb24vXG4gICAqL1xuICBwdWJsaWMgdG9sZXJhdGUobm9kZTogVGFpbnRlZE5vZGUpIHtcbiAgICBmb3IgKGNvbnN0IHF1ZXJ5IG9mIG5vZGUudGFpbnRTZWxlY3Rvcikge1xuXG4gICAgICB0aGlzLl90b2xlcmF0aW9ucy5wdXNoKHtcbiAgICAgICAga2V5OiBxdWVyeS5rZXksXG4gICAgICAgIHZhbHVlOiBxdWVyeS52YWx1ZSxcbiAgICAgICAgZWZmZWN0OiBxdWVyeS5lZmZlY3QsXG4gICAgICAgIG9wZXJhdG9yOiBxdWVyeS5vcGVyYXRvcixcbiAgICAgICAgdG9sZXJhdGlvblNlY29uZHM6IHF1ZXJ5LmV2aWN0QWZ0ZXI/LnRvU2Vjb25kcygpLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEF0dHJhY3QgdGhpcyBwb2QgdG8gYSBub2RlIG1hdGNoZWQgYnkgc2VsZWN0b3JzLlxuICAgKiBZb3UgY2FuIHNlbGVjdCBhIG5vZGUgYnkgdXNpbmcgYE5vZGUubGFiZWxlZCgpYC5cbiAgICpcbiAgICogQXR0cmFjdGluZyB0byBtdWx0aXBsZSBub2RlcyAoaS5lIGludm9raW5nIHRoaXMgbWV0aG9kIG11bHRpcGxlIHRpbWVzKSBhY3RzIGFzXG4gICAqIGFuIE9SIGNvbmRpdGlvbiwgbWVhbmluZyB0aGUgcG9kIHdpbGwgYmUgYXNzaWduZWQgdG8gZWl0aGVyIG9uZSBvZiB0aGUgbm9kZXMuXG4gICAqXG4gICAqIFVuZGVyIHRoZSBob29kLCB0aGlzIG1ldGhvZCB1dGlsaXplcyB0aGUgYG5vZGVBZmZpbml0eWAgcHJvcGVydHkuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc2NoZWR1bGluZy1ldmljdGlvbi9hc3NpZ24tcG9kLW5vZGUvI25vZGUtYWZmaW5pdHlcbiAgICovXG4gIHB1YmxpYyBhdHRyYWN0KG5vZGU6IExhYmVsZWROb2RlLCBvcHRpb25zOiBQb2RTY2hlZHVsaW5nQXR0cmFjdE9wdGlvbnMgPSB7fSkge1xuXG4gICAgY29uc3QgdGVybSA9IHRoaXMuY3JlYXRlTm9kZUFmZmluaXR5VGVybShub2RlKTtcblxuICAgIGlmIChvcHRpb25zLndlaWdodCkge1xuICAgICAgdGhpcy52YWxpZGF0ZVdlaWdodChvcHRpb25zLndlaWdodCk7XG4gICAgICB0aGlzLl9ub2RlQWZmaW5pdHlQcmVmZXJyZWQucHVzaCh7IHdlaWdodDogb3B0aW9ucy53ZWlnaHQsIHByZWZlcmVuY2U6IHRlcm0gfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuX25vZGVBZmZpbml0eVJlcXVpcmVkLnB1c2godGVybSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENvLWxvY2F0ZSB0aGlzIHBvZCB3aXRoIGEgc2NoZWR1bGluZyBzZWxlY3Rpb24uXG4gICAqXG4gICAqIEEgc2VsZWN0aW9uIGNhbiBiZSBvbmUgb2Y6XG4gICAqXG4gICAqIC0gQW4gaW5zdGFuY2Ugb2YgYSBgUG9kYC5cbiAgICogLSBBbiBpbnN0YW5jZSBvZiBhIGBXb3JrbG9hZGAgKGUuZyBgRGVwbG95bWVudGAsIGBTdGF0ZWZ1bFNldGApLlxuICAgKiAtIEFuIHVuLW1hbmFnZWQgcG9kIHRoYXQgY2FuIGJlIHNlbGVjdGVkIHZpYSBgUG9kcy5zZWxlY3QoKWAuXG4gICAqXG4gICAqIENvLWxvY2F0aW5nIHdpdGggbXVsdGlwbGUgc2VsZWN0aW9ucyAoKGkuZSBpbnZva2luZyB0aGlzIG1ldGhvZCBtdWx0aXBsZSB0aW1lcykpIGFjdHMgYXNcbiAgICogYW4gQU5EIGNvbmRpdGlvbi4gbWVhbmluZyB0aGUgcG9kIHdpbGwgYmUgYXNzaWduZWQgdG8gYSBub2RlIHRoYXQgc2F0aXNmaWVzIGFsbFxuICAgKiBzZWxlY3Rpb25zIChpLmUgcnVucyBhdCBsZWFzdCBvbmUgcG9kIHRoYXQgc2F0aXNpZmllcyBlYWNoIHNlbGVjdGlvbikuXG4gICAqXG4gICAqIFVuZGVyIHRoZSBob29kLCB0aGlzIG1ldGhvZCB1dGlsaXplcyB0aGUgYHBvZEFmZmluaXR5YCBwcm9wZXJ0eS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zY2hlZHVsaW5nLWV2aWN0aW9uL2Fzc2lnbi1wb2Qtbm9kZS8jaW50ZXItcG9kLWFmZmluaXR5LWFuZC1hbnRpLWFmZmluaXR5XG4gICAqL1xuICBwdWJsaWMgY29sb2NhdGUoc2VsZWN0b3I6IElQb2RTZWxlY3Rvciwgb3B0aW9uczogUG9kU2NoZWR1bGluZ0NvbG9jYXRlT3B0aW9ucyA9IHt9KSB7XG5cbiAgICBjb25zdCB0b3BvbG9neSA9IG9wdGlvbnMudG9wb2xvZ3kgPz8gVG9wb2xvZ3kuSE9TVE5BTUU7XG4gICAgY29uc3QgdGVybSA9IHRoaXMuY3JlYXRlUG9kQWZmaW5pdHlUZXJtKHRvcG9sb2d5LCBzZWxlY3Rvcik7XG5cbiAgICBpZiAob3B0aW9ucy53ZWlnaHQpIHtcbiAgICAgIHRoaXMudmFsaWRhdGVXZWlnaHQob3B0aW9ucy53ZWlnaHQpO1xuICAgICAgdGhpcy5fcG9kQWZmaW5pdHlQcmVmZXJyZWQucHVzaCh7IHdlaWdodDogb3B0aW9ucy53ZWlnaHQsIHBvZEFmZmluaXR5VGVybTogdGVybSB9KTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5fcG9kQWZmaW5pdHlSZXF1aXJlZC5wdXNoKHRlcm0pO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBTZXBlcmF0ZSB0aGlzIHBvZCBmcm9tIGEgc2NoZWR1bGluZyBzZWxlY3Rpb24uXG4gICAqXG4gICAqIEEgc2VsZWN0aW9uIGNhbiBiZSBvbmUgb2Y6XG4gICAqXG4gICAqIC0gQW4gaW5zdGFuY2Ugb2YgYSBgUG9kYC5cbiAgICogLSBBbiBpbnN0YW5jZSBvZiBhIGBXb3JrbG9hZGAgKGUuZyBgRGVwbG95bWVudGAsIGBTdGF0ZWZ1bFNldGApLlxuICAgKiAtIEFuIHVuLW1hbmFnZWQgcG9kIHRoYXQgY2FuIGJlIHNlbGVjdGVkIHZpYSBgUG9kcy5zZWxlY3QoKWAuXG4gICAqXG4gICAqIFNlcGVyYXRpbmcgZnJvbSBtdWx0aXBsZSBzZWxlY3Rpb25zIGFjdHMgYXMgYW4gQU5EIGNvbmRpdGlvbi4gbWVhbmluZyB0aGUgcG9kXG4gICAqIHdpbGwgbm90IGJlIGFzc2lnbmVkIHRvIGEgbm9kZSB0aGF0IHNhdGlzZmllcyBhbGwgc2VsZWN0aW9ucyAoaS5lIHJ1bnMgYXQgbGVhc3Qgb25lIHBvZCB0aGF0IHNhdGlzaWZpZXMgZWFjaCBzZWxlY3Rpb24pLlxuICAgKlxuICAgKiBVbmRlciB0aGUgaG9vZCwgdGhpcyBtZXRob2QgdXRpbGl6ZXMgdGhlIGBwb2RBbnRpQWZmaW5pdHlgIHByb3BlcnR5LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3NjaGVkdWxpbmctZXZpY3Rpb24vYXNzaWduLXBvZC1ub2RlLyNpbnRlci1wb2QtYWZmaW5pdHktYW5kLWFudGktYWZmaW5pdHlcbiAgICovXG4gIHB1YmxpYyBzZXBhcmF0ZShzZWxlY3RvcjogSVBvZFNlbGVjdG9yLCBvcHRpb25zOiBQb2RTY2hlZHVsaW5nU2VwYXJhdGVPcHRpb25zID0ge30pIHtcblxuICAgIGNvbnN0IHRvcG9sb2d5ID0gb3B0aW9ucy50b3BvbG9neSA/PyBUb3BvbG9neS5IT1NUTkFNRTtcbiAgICBjb25zdCB0ZXJtID0gdGhpcy5jcmVhdGVQb2RBZmZpbml0eVRlcm0odG9wb2xvZ3ksIHNlbGVjdG9yKTtcblxuICAgIGlmIChvcHRpb25zLndlaWdodCkge1xuICAgICAgdGhpcy52YWxpZGF0ZVdlaWdodChvcHRpb25zLndlaWdodCk7XG4gICAgICB0aGlzLl9wb2RBbnRpQWZmaW5pdHlQcmVmZXJyZWQucHVzaCh7IHdlaWdodDogb3B0aW9ucy53ZWlnaHQsIHBvZEFmZmluaXR5VGVybTogdGVybSB9KTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5fcG9kQW50aUFmZmluaXR5UmVxdWlyZWQucHVzaCh0ZXJtKTtcbiAgICB9XG5cbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlUG9kQWZmaW5pdHlUZXJtKHRvcG9sb2d5OiBUb3BvbG9neSwgc2VsZWN0b3I6IElQb2RTZWxlY3Rvcik6IGs4cy5Qb2RBZmZpbml0eVRlcm0ge1xuICAgIGNvbnN0IGNvbmZpZyA9IHNlbGVjdG9yLnRvUG9kU2VsZWN0b3JDb25maWcoKTtcbiAgICByZXR1cm4ge1xuICAgICAgdG9wb2xvZ3lLZXk6IHRvcG9sb2d5LmtleSxcbiAgICAgIGxhYmVsU2VsZWN0b3I6IGNvbmZpZy5sYWJlbFNlbGVjdG9yLl90b0t1YmUoKSxcbiAgICAgIG5hbWVzcGFjZVNlbGVjdG9yOiBjb25maWcubmFtZXNwYWNlcz8ubGFiZWxTZWxlY3Rvcj8uX3RvS3ViZSgpLFxuICAgICAgbmFtZXNwYWNlczogY29uZmlnLm5hbWVzcGFjZXM/Lm5hbWVzLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZU5vZGVBZmZpbml0eVRlcm0obm9kZTogTGFiZWxlZE5vZGUpOiBrOHMuTm9kZVNlbGVjdG9yVGVybSB7XG4gICAgcmV0dXJuIHsgbWF0Y2hFeHByZXNzaW9uczogbm9kZS5sYWJlbFNlbGVjdG9yLm1hcChzID0+ICh7IGtleTogcy5rZXksIG9wZXJhdG9yOiBzLm9wZXJhdG9yISwgdmFsdWVzOiBzLnZhbHVlcyB9KSkgfTtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVXZWlnaHQod2VpZ2h0OiBudW1iZXIpIHtcbiAgICBpZiAod2VpZ2h0IDwgMSB8fCB3ZWlnaHQgPiAxMDApIHtcbiAgICAgIC8vIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3NjaGVkdWxpbmctZXZpY3Rpb24vYXNzaWduLXBvZC1ub2RlLyNub2RlLWFmZmluaXR5LXdlaWdodFxuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGFmZmluaXR5IHdlaWdodDogJHt3ZWlnaHR9LiBNdXN0IGJlIGluIHJhbmdlIDEtMTAwYCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogeyBhZmZpbml0eT86IGs4cy5BZmZpbml0eTsgbm9kZU5hbWU/OiBzdHJpbmc7IHRvbGVyYXRpb25zPzogazhzLlRvbGVyYXRpb25bXSB9IHtcblxuICAgIGNvbnN0IGF0TGVhc3RPbmUgPSAoLi4uYXJyYXlzOiBBcnJheTxhbnk+W10pID0+IHtcbiAgICAgIHJldHVybiBhcnJheXMuZmxhdCgpLmxlbmd0aCA+IDA7XG4gICAgfTtcblxuICAgIGNvbnN0IGhhc05vZGVBZmZpbml0eSA9IGF0TGVhc3RPbmUodGhpcy5fbm9kZUFmZmluaXR5UHJlZmVycmVkLCB0aGlzLl9ub2RlQWZmaW5pdHlSZXF1aXJlZCk7XG4gICAgY29uc3QgaGFzUG9kQWZmaW5pdHkgPSBhdExlYXN0T25lKHRoaXMuX3BvZEFmZmluaXR5UHJlZmVycmVkLCB0aGlzLl9wb2RBZmZpbml0eVJlcXVpcmVkKTtcbiAgICBjb25zdCBoYXNQb2RBbnRpQWZmaW50eSA9IGF0TGVhc3RPbmUodGhpcy5fcG9kQW50aUFmZmluaXR5UHJlZmVycmVkLCB0aGlzLl9wb2RBbnRpQWZmaW5pdHlSZXF1aXJlZCk7XG4gICAgY29uc3QgaGFzQWZmaW5pdHkgPSBoYXNOb2RlQWZmaW5pdHkgfHwgaGFzUG9kQWZmaW5pdHkgfHwgaGFzUG9kQW50aUFmZmludHk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgYWZmaW5pdHk6IGhhc0FmZmluaXR5ID8ge1xuICAgICAgICBub2RlQWZmaW5pdHk6IGhhc05vZGVBZmZpbml0eSA/IHtcbiAgICAgICAgICBwcmVmZXJyZWREdXJpbmdTY2hlZHVsaW5nSWdub3JlZER1cmluZ0V4ZWN1dGlvbjogdW5kZWZpbmVkSWZFbXB0eSh0aGlzLl9ub2RlQWZmaW5pdHlQcmVmZXJyZWQpLFxuICAgICAgICAgIHJlcXVpcmVkRHVyaW5nU2NoZWR1bGluZ0lnbm9yZWREdXJpbmdFeGVjdXRpb246IHRoaXMuX25vZGVBZmZpbml0eVJlcXVpcmVkLmxlbmd0aCA+IDAgPyB7XG4gICAgICAgICAgICBub2RlU2VsZWN0b3JUZXJtczogdGhpcy5fbm9kZUFmZmluaXR5UmVxdWlyZWQsXG4gICAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgICAgcG9kQWZmaW5pdHk6IGhhc1BvZEFmZmluaXR5ID8ge1xuICAgICAgICAgIHByZWZlcnJlZER1cmluZ1NjaGVkdWxpbmdJZ25vcmVkRHVyaW5nRXhlY3V0aW9uOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMuX3BvZEFmZmluaXR5UHJlZmVycmVkKSxcbiAgICAgICAgICByZXF1aXJlZER1cmluZ1NjaGVkdWxpbmdJZ25vcmVkRHVyaW5nRXhlY3V0aW9uOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMuX3BvZEFmZmluaXR5UmVxdWlyZWQpLFxuICAgICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgICBwb2RBbnRpQWZmaW5pdHk6IGhhc1BvZEFudGlBZmZpbnR5ID8ge1xuICAgICAgICAgIHByZWZlcnJlZER1cmluZ1NjaGVkdWxpbmdJZ25vcmVkRHVyaW5nRXhlY3V0aW9uOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMuX3BvZEFudGlBZmZpbml0eVByZWZlcnJlZCksXG4gICAgICAgICAgcmVxdWlyZWREdXJpbmdTY2hlZHVsaW5nSWdub3JlZER1cmluZ0V4ZWN1dGlvbjogdW5kZWZpbmVkSWZFbXB0eSh0aGlzLl9wb2RBbnRpQWZmaW5pdHlSZXF1aXJlZCksXG4gICAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgbm9kZU5hbWU6IHRoaXMuX25vZGVOYW1lLFxuICAgICAgdG9sZXJhdGlvbnM6IHVuZGVmaW5lZElmRW1wdHkodGhpcy5fdG9sZXJhdGlvbnMpLFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBJc29sYXRpb24gZGV0ZXJtaW5lcyB3aGljaCBwb2xpY2llcyBhcmUgY3JlYXRlZFxuICogd2hlbiBhbGxvd2luZyBjb25uZWN0aW9ucyBmcm9tIGEgYSBwb2QgLyB3b3JrbG9hZCB0byBwZWVycy5cbiAqL1xuZXhwb3J0IGVudW0gUG9kQ29ubmVjdGlvbnNJc29sYXRpb24ge1xuXG4gIC8qKlxuICAgKiBPbmx5IGNyZWF0ZXMgbmV0d29yayBwb2xpY2llcyB0aGF0IHNlbGVjdCB0aGUgcG9kLlxuICAgKi9cbiAgUE9EID0gJ1BPRCcsXG5cbiAgLyoqXG4gICAqIE9ubHkgY3JlYXRlcyBuZXR3b3JrIHBvbGljaWVzIHRoYXQgc2VsZWN0IHRoZSBwZWVyLlxuICAgKi9cbiAgUEVFUiA9ICdQRUVSJyxcblxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBQb2RDb25uZWN0aW9ucy5hbGxvd1RvYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQb2RDb25uZWN0aW9uc0FsbG93VG9PcHRpb25zIHtcblxuICAvKipcbiAgICogV2hpY2ggaXNvbGF0aW9uIHNob3VsZCBiZSBhcHBsaWVkIHRvIGVzdGFibGlzaCB0aGUgY29ubmVjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1bnNldCwgaXNvbGF0ZXMgYm90aCB0aGUgcG9kIGFuZCB0aGUgcGVlci5cbiAgICovXG4gIHJlYWRvbmx5IGlzb2xhdGlvbj86IFBvZENvbm5lY3Rpb25zSXNvbGF0aW9uO1xuXG4gIC8qKlxuICAgKiBQb3J0cyB0byBhbGxvdyBvdXRnb2luZyB0cmFmZmljIHRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIElmIHRoZSBwZWVyIGlzIGEgbWFuYWdlZCBwb2QsIHRha2UgaXRzIHBvcnRzLiBPdGhlcndpc2UsIGFsbCBwb3J0cyBhcmUgYWxsb3dlZC5cbiAgICovXG4gIHJlYWRvbmx5IHBvcnRzPzogbmV0d29ya3BvbGljeS5OZXR3b3JrUG9saWN5UG9ydFtdO1xuXG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYFBvZENvbm5lY3Rpb25zLmFsbG93RnJvbWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUG9kQ29ubmVjdGlvbnNBbGxvd0Zyb21PcHRpb25zIHtcblxuICAvKipcbiAgICogV2hpY2ggaXNvbGF0aW9uIHNob3VsZCBiZSBhcHBsaWVkIHRvIGVzdGFibGlzaCB0aGUgY29ubmVjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1bnNldCwgaXNvbGF0ZXMgYm90aCB0aGUgcG9kIGFuZCB0aGUgcGVlci5cbiAgICovXG4gIHJlYWRvbmx5IGlzb2xhdGlvbj86IFBvZENvbm5lY3Rpb25zSXNvbGF0aW9uO1xuXG4gIC8qKlxuICAgKiBQb3J0cyB0byBhbGxvdyBpbmNvbWluZyB0cmFmZmljIHRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRoZSBwb2QgcG9ydHMuXG4gICAqL1xuICByZWFkb25seSBwb3J0cz86IG5ldHdvcmtwb2xpY3kuTmV0d29ya1BvbGljeVBvcnRbXTtcblxufVxuXG4vKipcbiAqIENvbnRyb2xzIG5ldHdvcmsgaXNvbGF0aW9uIHJ1bGVzIGZvciBpbnRlci1wb2QgY29tbXVuaWNhdGlvbi5cbiAqL1xuZXhwb3J0IGNsYXNzIFBvZENvbm5lY3Rpb25zIHtcblxuICBjb25zdHJ1Y3Rvcihwcm90ZWN0ZWQgcmVhZG9ubHkgaW5zdGFuY2U6IEFic3RyYWN0UG9kKSB7fVxuXG4gIC8qKlxuICAgKiBBbGxvdyBuZXR3b3JrIHRyYWZmaWMgZnJvbSB0aGlzIHBvZCB0byB0aGUgcGVlci5cbiAgICpcbiAgICogQnkgZGVmYXVsdCwgdGhpcyB3aWxsIGNyZWF0ZSBhbiBlZ3Jlc3MgbmV0d29yayBwb2xpY3kgZm9yIHRoaXMgcG9kLCBhbmQgYW4gaW5ncmVzc1xuICAgKiBuZXR3b3JrIHBvbGljeSBmb3IgdGhlIHBlZXIuIFRoaXMgaXMgcmVxdWlyZWQgaWYgYm90aCBzaWRlcyBhcmUgYWxyZWFkeSBpc29sYXRlZC5cbiAgICogVXNlIGBvcHRpb25zLmlzb2xhdGlvbmAgdG8gY29udHJvbCB0aGlzIGJlaGF2aW9yLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKlxuICAgKiAvLyBjcmVhdGUgb25seSBhbiBlZ3Jlc3MgcG9saWN5IHRoYXQgc2VsZWN0cyB0aGUgJ3dlYicgcG9kIHRvIGFsbG93IG91dGdvaW5nIHRyYWZmaWNcbiAgICogLy8gdG8gdGhlICdyZWRpcycgcG9kLiB0aGlzIHJlcXVpcmVzIHRoZSAncmVkaXMnIHBvZCB0byBub3QgYmUgaXNvbGF0ZWQgZm9yIGluZ3Jlc3MuXG4gICAqIHdlYi5jb25uZWN0aW9ucy5hbGxvd1RvKHJlZGlzLCB7IGlzb2xhdGlvbjogSXNvbGF0aW9uLlBPRCB9KVxuICAgKlxuICAgKiAvLyBjcmVhdGUgb25seSBhbiBpbmdyZXNzIHBvbGljeSB0aGF0IHNlbGVjdHMgdGhlICdyZWRpcycgcGVlciB0byBhbGxvdyBpbmNvbWluZyB0cmFmZmljXG4gICAqIC8vIGZyb20gdGhlICd3ZWInIHBvZC4gdGhpcyByZXF1aXJlcyB0aGUgJ3dlYicgcG9kIHRvIG5vdCBiZSBpc29sYXRlZCBmb3IgZWdyZXNzLlxuICAgKiB3ZWIuY29ubmVjdGlvbnMuYWxsb3dUbyhyZWRpcywgeyBpc29sYXRpb246IElzb2xhdGlvbi5QRUVSIH0pXG4gICAqXG4gICAqL1xuICBwdWJsaWMgYWxsb3dUbyhwZWVyOiBuZXR3b3JrcG9saWN5LklOZXR3b3JrUG9saWN5UGVlciwgb3B0aW9uczogUG9kQ29ubmVjdGlvbnNBbGxvd1RvT3B0aW9ucyA9IHt9KSB7XG4gICAgcmV0dXJuIHRoaXMuYWxsb3coJ0VncmVzcycsIHBlZXIsIHsgcG9ydHM6IHRoaXMuZXh0cmFjdFBvcnRzKHBlZXIpLCAuLi5vcHRpb25zIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93IG5ldHdvcmsgdHJhZmZpYyBmcm9tIHRoZSBwZWVyIHRvIHRoaXMgcG9kLlxuICAgKlxuICAgKiBCeSBkZWZhdWx0LCB0aGlzIHdpbGwgY3JlYXRlIGFuIGluZ3Jlc3MgbmV0d29yayBwb2xpY3kgZm9yIHRoaXMgcG9kLCBhbmQgYW4gZWdyZXNzXG4gICAqIG5ldHdvcmsgcG9saWN5IGZvciB0aGUgcGVlci4gVGhpcyBpcyByZXF1aXJlZCBpZiBib3RoIHNpZGVzIGFyZSBhbHJlYWR5IGlzb2xhdGVkLlxuICAgKiBVc2UgYG9wdGlvbnMuaXNvbGF0aW9uYCB0byBjb250cm9sIHRoaXMgYmVoYXZpb3IuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqXG4gICAqIC8vIGNyZWF0ZSBvbmx5IGFuIGVncmVzcyBwb2xpY3kgdGhhdCBzZWxlY3RzIHRoZSAnd2ViJyBwb2QgdG8gYWxsb3cgb3V0Z29pbmcgdHJhZmZpY1xuICAgKiAvLyB0byB0aGUgJ3JlZGlzJyBwb2QuIHRoaXMgcmVxdWlyZXMgdGhlICdyZWRpcycgcG9kIHRvIG5vdCBiZSBpc29sYXRlZCBmb3IgaW5ncmVzcy5cbiAgICogcmVkaXMuY29ubmVjdGlvbnMuYWxsb3dGcm9tKHdlYiwgeyBpc29sYXRpb246IElzb2xhdGlvbi5QRUVSIH0pXG4gICAqXG4gICAqIC8vIGNyZWF0ZSBvbmx5IGFuIGluZ3Jlc3MgcG9saWN5IHRoYXQgc2VsZWN0cyB0aGUgJ3JlZGlzJyBwZWVyIHRvIGFsbG93IGluY29taW5nIHRyYWZmaWNcbiAgICogLy8gZnJvbSB0aGUgJ3dlYicgcG9kLiB0aGlzIHJlcXVpcmVzIHRoZSAnd2ViJyBwb2QgdG8gbm90IGJlIGlzb2xhdGVkIGZvciBlZ3Jlc3MuXG4gICAqIHJlZGlzLmNvbm5lY3Rpb25zLmFsbG93RnJvbSh3ZWIsIHsgaXNvbGF0aW9uOiBJc29sYXRpb24uUE9EIH0pXG4gICAqXG4gICAqL1xuICBwdWJsaWMgYWxsb3dGcm9tKHBlZXI6IG5ldHdvcmtwb2xpY3kuSU5ldHdvcmtQb2xpY3lQZWVyLCBvcHRpb25zOiBQb2RDb25uZWN0aW9uc0FsbG93RnJvbU9wdGlvbnMgPSB7fSkge1xuICAgIHJldHVybiB0aGlzLmFsbG93KCdJbmdyZXNzJywgcGVlciwgeyBwb3J0czogdGhpcy5leHRyYWN0UG9ydHModGhpcy5pbnN0YW5jZSksIC4uLm9wdGlvbnMgfSk7XG4gIH1cblxuICBwcml2YXRlIGFsbG93KGRpcmVjdGlvbjogJ0luZ3Jlc3MnIHwgJ0VncmVzcycsIHBlZXI6IG5ldHdvcmtwb2xpY3kuSU5ldHdvcmtQb2xpY3lQZWVyLCBvcHRpb25zOiBQb2RDb25uZWN0aW9uc0FsbG93VG9PcHRpb25zIHwgUG9kQ29ubmVjdGlvbnNBbGxvd0Zyb21PcHRpb25zID0ge30pIHtcblxuICAgIGNvbnN0IGNvbmZpZyA9IHBlZXIudG9OZXR3b3JrUG9saWN5UGVlckNvbmZpZygpO1xuICAgIG5ldHdvcmtwb2xpY3kudmFsaWRhdGVQZWVyQ29uZmlnKGNvbmZpZyk7XG5cbiAgICBjb25zdCBwZWVyQWRkcmVzcyA9IGFkZHJlc3MocGVlcik7XG5cbiAgICBpZiAoIW9wdGlvbnMuaXNvbGF0aW9uIHx8IG9wdGlvbnMuaXNvbGF0aW9uID09PSBQb2RDb25uZWN0aW9uc0lzb2xhdGlvbi5QT0QpIHtcblxuICAgICAgY29uc3Qgc3JjID0gbmV3IG5ldHdvcmtwb2xpY3kuTmV0d29ya1BvbGljeSh0aGlzLmluc3RhbmNlLCBgQWxsb3cke2RpcmVjdGlvbn0ke3BlZXJBZGRyZXNzfWAsIHtcbiAgICAgICAgc2VsZWN0b3I6IHRoaXMuaW5zdGFuY2UsXG4gICAgICAgIC8vIHRoZSBwb2xpY3kgbXVzdCBiZSBkZWZpbmVkIGluIHRoZSBuYW1lc3BhY2Ugb2YgdGhlIHBvZFxuICAgICAgICAvLyBzbyBpdCBjYW4gc2VsZWN0IGl0LlxuICAgICAgICBtZXRhZGF0YTogeyBuYW1lc3BhY2U6IHRoaXMuaW5zdGFuY2UubWV0YWRhdGEubmFtZXNwYWNlIH0sXG4gICAgICB9KTtcblxuICAgICAgc3dpdGNoIChkaXJlY3Rpb24pIHtcbiAgICAgICAgY2FzZSAnRWdyZXNzJzpcbiAgICAgICAgICBzcmMuYWRkRWdyZXNzUnVsZShwZWVyLCBvcHRpb25zLnBvcnRzKTtcbiAgICAgICAgICBicmVhaztcbiAgICAgICAgY2FzZSAnSW5ncmVzcyc6XG4gICAgICAgICAgc3JjLmFkZEluZ3Jlc3NSdWxlKHBlZXIsIG9wdGlvbnMucG9ydHMpO1xuICAgICAgfVxuXG4gICAgfVxuXG4gICAgaWYgKCFvcHRpb25zLmlzb2xhdGlvbiB8fCBvcHRpb25zLmlzb2xhdGlvbiA9PT0gUG9kQ29ubmVjdGlvbnNJc29sYXRpb24uUEVFUikge1xuXG4gICAgICBpZiAoY29uZmlnLmlwQmxvY2spIHtcbiAgICAgICAgLy8gZm9yIGFuIGlwIGJsb2NrIHdlIGRvbid0IG5lZWQgdG8gY3JlYXRlIHRoZSBvcHBvc2l0ZSBwb2xpY2llc1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IHBvZFNlbGVjdG9yID0gcGVlci50b1BvZFNlbGVjdG9yKCk7XG4gICAgICBpZiAoIXBvZFNlbGVjdG9yKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIGNyZWF0ZSBwb2xpY2llcyBmb3IgcGVlciAnJHtwZWVyLm5vZGUuYWRkcn0nIHNpbmNlIGl0cyBub3QgYSBwb2Qgc2VsZWN0b3JgKTtcbiAgICAgIH1cblxuICAgICAgY29uc3Qgb3Bwb3NpdGVEaXJlY3Rpb24gPSBkaXJlY3Rpb24gPT09ICdFZ3Jlc3MnID8gJ0luZ3Jlc3MnIDogJ0VncmVzcyc7XG5cbiAgICAgIGNvbnN0IHBvZFNlbGVjdG9yQ29uZmlnID0gcG9kU2VsZWN0b3IudG9Qb2RTZWxlY3RvckNvbmZpZygpO1xuICAgICAgbGV0IG5hbWVzcGFjZXM6IChzdHJpbmcgfCB1bmRlZmluZWQpW107XG5cbiAgICAgIGlmICghcG9kU2VsZWN0b3JDb25maWcubmFtZXNwYWNlcykge1xuXG4gICAgICAgIC8vIGlmIHRoZSBwZWVyIGRvZXNuJ3Qgc3BlY2lmeSBuYW1lc3BhY2VzLCB3ZSBhc3N1bWUgdGhlIHNhbWUgbmFtZXNwYWNlLlxuICAgICAgICBuYW1lc3BhY2VzID0gW3RoaXMuaW5zdGFuY2UubWV0YWRhdGEubmFtZXNwYWNlXTtcblxuICAgICAgfSBlbHNlIHtcblxuICAgICAgICAvLyBhIHBlZXIgY2Fubm90IHNwZWNpZnkgbmFtZXNwYWNlcyBieSBsYWJlbHMgYmVjYXVzZVxuICAgICAgICAvLyB3ZSB3b24ndCBiZSBhYmxlIHRvIGV4dHJhY3QgdGhlIG5hbWVzIG9mIHRob3NlIG5hbWVzcGFjZXMuXG4gICAgICAgIGlmIChwb2RTZWxlY3RvckNvbmZpZy5uYW1lc3BhY2VzLmxhYmVsU2VsZWN0b3IgJiYgIXBvZFNlbGVjdG9yQ29uZmlnLm5hbWVzcGFjZXMubGFiZWxTZWxlY3Rvci5pc0VtcHR5KCkpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuYWJsZSB0byBjcmVhdGUgYW4gJHtvcHBvc2l0ZURpcmVjdGlvbn0gcG9saWN5IGZvciBwZWVyICcke3BlZXIubm9kZS5wYXRofScgKHBvZD0ke3RoaXMuaW5zdGFuY2UubmFtZX0pLiBQZWVyIG11c3Qgc3BlY2lmeSBuYW1lc3BhY2VzIG9ubHkgYnkgbmFtZWApO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gYSBwZWVyIG11c3Qgc3BlY2lmeSBuYW1lc3BhY2VzIGJ5IG5hbWUuXG4gICAgICAgIGlmICghcG9kU2VsZWN0b3JDb25maWcubmFtZXNwYWNlcy5uYW1lcykge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIGNyZWF0ZSBhbiAke29wcG9zaXRlRGlyZWN0aW9ufSBwb2xpY3kgZm9yIHBlZXIgJyR7cGVlci5ub2RlLnBhdGh9JyAocG9kPSR7dGhpcy5pbnN0YW5jZS5uYW1lfSkuIFBlZXIgbXVzdCBzcGVjaWZ5IG5hbWVzcGFjZSBuYW1lc2ApO1xuICAgICAgICB9XG5cbiAgICAgICAgbmFtZXNwYWNlcyA9IHBvZFNlbGVjdG9yQ29uZmlnLm5hbWVzcGFjZXMubmFtZXM7XG4gICAgICB9XG5cbiAgICAgIGZvciAoY29uc3QgbmFtZSBvZiBuYW1lc3BhY2VzKSB7XG4gICAgICAgIHN3aXRjaCAoZGlyZWN0aW9uKSB7XG4gICAgICAgICAgY2FzZSAnRWdyZXNzJzpcbiAgICAgICAgICAgIG5ldyBuZXR3b3JrcG9saWN5Lk5ldHdvcmtQb2xpY3kodGhpcy5pbnN0YW5jZSwgYEFsbG93SW5ncmVzcyR7bmFtZX0ke3BlZXJBZGRyZXNzfWAsIHtcbiAgICAgICAgICAgICAgc2VsZWN0b3I6IHBvZFNlbGVjdG9yLFxuICAgICAgICAgICAgICBtZXRhZGF0YTogeyBuYW1lc3BhY2U6IG5hbWUgfSxcbiAgICAgICAgICAgICAgaW5ncmVzczogeyBydWxlczogW3sgcGVlcjogdGhpcy5pbnN0YW5jZSwgcG9ydHM6IG9wdGlvbnMucG9ydHMgfV0gfSxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgY2FzZSAnSW5ncmVzcyc6XG4gICAgICAgICAgICBuZXcgbmV0d29ya3BvbGljeS5OZXR3b3JrUG9saWN5KHRoaXMuaW5zdGFuY2UsIGBBbGxvd0VncmVzcyR7bmFtZX0ke3BlZXJBZGRyZXNzfWAsIHtcbiAgICAgICAgICAgICAgc2VsZWN0b3I6IHBvZFNlbGVjdG9yLFxuICAgICAgICAgICAgICBtZXRhZGF0YTogeyBuYW1lc3BhY2U6IG5hbWUgfSxcbiAgICAgICAgICAgICAgZWdyZXNzOiB7IHJ1bGVzOiBbeyBwZWVyOiB0aGlzLmluc3RhbmNlLCBwb3J0czogb3B0aW9ucy5wb3J0cyB9XSB9LFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBkaXJlY3Rpb246ICR7ZGlyZWN0aW9ufWApO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGV4dHJhY3RQb3J0cyhzZWxlY3Rvcj86IG5ldHdvcmtwb2xpY3kuSU5ldHdvcmtQb2xpY3lQZWVyKTogbmV0d29ya3BvbGljeS5OZXR3b3JrUG9saWN5UG9ydFtdIHtcbiAgICByZXR1cm4gY29udGFpbmVyLmV4dHJhY3RDb250YWluZXJQb3J0cyhzZWxlY3RvcikubWFwKG4gPT4gbmV0d29ya3BvbGljeS5OZXR3b3JrUG9saWN5UG9ydC50Y3Aobi5udW1iZXIpKTtcbiAgfVxufVxuIl19