"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? false;
        this.user = props.user ?? 25000;
        this.group = props.group ?? 26000;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-22.ContainerSecurityContext", version: "2.0.0-rc.110" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-22.EnvValue", version: "2.0.0-rc.110" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numbers = this._ports.map(p => p.number);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        if (numbers.includes(port.number)) {
            throw new Error(`Port with number ${port.number} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol?.toString(),
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-22.Container", version: "2.0.0-rc.110" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-22.Cpu", version: "2.0.0-rc.110" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-22.EnvFrom", version: "2.0.0-rc.110" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-22.Env", version: "2.0.0-rc.110" };
//# sourceMappingURL=data:application/json;base64,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