from contextlib import contextmanager
import os
from shutil import copy as copy_file
from typing import Callable, List, Optional, Tuple, Union
from dcicutils.file_utils import normalize_path
from dcicutils.tmpfile_utils import create_temporary_file_name, temporary_file
from submitr.rclone.rclone_store import RCloneStore
from submitr.rclone.rclone_amazon import RCloneAmazon
from submitr.rclone.rclone_commands import RCloneCommands
from submitr.rclone.rclone_installation import RCloneInstallation
from submitr.rclone.rclone_utils import cloud_path
from submitr.utils import DEBUGGING

# Notes on the basic structure of this rclone support code.
#
# - rclone_store.RCloneStore
#   Abstract base class representing a cloud provider target (a source or a destination).
#   The RCloneAmazon and RCloneGoogle classes are derived from this. Contains the credentials
#   for accessing the specific cloud provider (AWS credentials file with access key info for
#   Amazon, and the exorted service account file for Google); and functions to obtain various
#   info like whether or not a given path/bucket/file/directory exists, the size of a given
#   file (key), its checksum, et cetera, and a ping function. It has a name (auto-generated
#   if unspecified) used as the identifier in the specific rclone commands to refer to its
#   config info contained within the rclone config file (also auto-generated by this class).
#
# - rclone_store.RCloneStoreRegistry
#   Class to collect info on all defined RCloneStore derived class, by decorating said derived
#   with the RCloneStore.register decorator (which calls into RCloneStoreRegistry.register).
#
# - rclone_amazon.RCloneAmazon
# - rclone_google.RCloneGoogle
#   Specific implementations of RCloneTraget for Amazon and Google, respectively.
#
# - rcloner.RCloner
#   Main class implementing the copy functionality from one cloud provider to another.
#   Though our only real use case currently is for Google to Amazon, all other combinations
#   related to Amazon, Google, and the local filesystem are also supported, as it pretty
#   much just fell out (e.g. local file to Google, Google to local file, Amazon to Google).
#
# - rcloner.RCloneCommands
#   Implementation of the lower-level rclone commands. These exec a sub-process to the
#   rclone command; which itself can be installed via the rclone_installation module.
#   The function here are called out to from RCloneStore and RCloner. These use an
#   rclone config file which is auto-generated (to a temporary file) by RCloneStore.
#
# - rclone_installation.RCloneInstallation
#   Module to check for and actually install rclone automatically with little/no user
#   interaction (just confirming okay to go ahead). This is done by ddownloading the
#   tarball, unpacking it, and placing the single rclone executable in the user's local
#   application directory (e.g. ~/Library/Application Support/edu.harvard.hms/smaht-submitr
#   on MacOS, or on Linux ~/.local/share/edu.harvard.hms/smaht-submitr). Note that no
#   testing has been done on Windows.
#
# For troubleshooting purposes, if you set your SMAHT_DEBUG environment variable to true,
# then details of the RCloneCommand actions/results are printed to stdout.


class RCloner(RCloneCommands, RCloneInstallation):

    def __init__(self, source: Optional[RCloneStore] = None, destination: Optional[RCloneStore] = None) -> None:
        self._source_config = source if isinstance(source, RCloneStore) else None
        self._destination_config = destination if isinstance(destination, RCloneStore) else None

    @property
    def source(self) -> Optional[RCloneStore]:
        return self._source_config

    @source.setter
    def source(self, value: RCloneStore) -> None:
        if isinstance(value, RCloneStore) or value is None:
            self._source_config = value

    @property
    def destination(self) -> Optional[RCloneStore]:
        return self._destination_config

    @destination.setter
    def destination(self, value: RCloneStore) -> None:
        if isinstance(value, RCloneStore) or value is None:
            self._destination_config = value

    @property
    def config_lines(self) -> List[str]:
        lines = []
        if (isinstance(source := self.source, RCloneStore) and
            isinstance(source_config_lines := source.config_lines(), list)):  # noqa
            lines.extend(source_config_lines)
        if (isinstance(destination_config := self.destination, RCloneStore) and
            isinstance(destination_config_lines := destination_config.config_lines(), list)):  # noqa
            if lines:
                lines.append("")  # not necessary but sporting
            lines.extend(destination_config_lines)
        return lines

    @contextmanager
    def config_file(self, persist: bool = False) -> str:
        with temporary_file(suffix=".conf") as temporary_config_file_name:
            os.chmod(temporary_config_file_name, 0o600)  # for security
            RCloneStore.write_config_file(temporary_config_file_name, self.config_lines)
            if (persist is True) or DEBUGGING():
                # This is just for dryrun for testing/troubleshooting.
                persistent_config_file_name = create_temporary_file_name(suffix=".conf")
                copy_file(temporary_config_file_name, persistent_config_file_name)
                os.chmod(persistent_config_file_name, 0o600)  # for security
                yield persistent_config_file_name
            else:
                yield temporary_config_file_name

    def copy(self, source: str, destination: Optional[str] = None, metadata: Optional[Callable] = None,
             progress: Optional[Callable] = None, dryrun: bool = False, copyto: bool = True,
             process_info: Optional[dict] = None, return_output: bool = False,
             raise_exception: bool = True) -> Union[bool, Tuple[bool, List[str]]]:
        """
        Uses rclone to copy the given source file to the given destination. All manner of variation is
        encapsulated within this simple statement. Depends on whether or not a source and/or destination
        configuration (RCloneStore) has been specified and whether or not a bucket is specified in the
        that configuration et cetera. If no configuration is specified then we assume the local file
        system is the source and/or destination. TODO: Expand on these notes.

        If self.source and/or self.destination is None then it means the the source and/or
        destination arguments here refer to local files; i.e. when no RCloneStore is
        specified we assume the (degenerate) case of local file.

        Note the we assume (by default) that the destination path is to a *file*, not a "directory" (such
        as they are in cloud storage); and we therefore use the rclone 'copyto' command rather than 'copy'.
        This keeps it simple (otherwise it gets surprisingly confusing with 'copy' WRT whether or not the
        destination is a file or "directory" et cetera); and in any case this is our only actual use-case.
        Can force to use 'copy' by passing False as the copyto argument.
        """
        # Just FYI WRT copy/copyto:
        # - Using 'copy' when the cloud destination is a file gives error: "is a file not a directory".
        # - Using 'copyto' when the cloud destination is a "directory" creates a *file* of that name;
        #   along side the "directory" of the same name (which is odd and almost certainly unwanted).
        if isinstance(destination_config := self.destination, RCloneStore):
            # Here a destination cloud configuration has been defined for this RCloner object;
            # meaning we are copying to some cloud destination (and not to a local file destination).
            is_destination_folder = destination_config.is_path_folder(destination)
            is_destination_bucket_only = destination_config.is_path_bucket_only(destination)
            if not (destination := destination_config.path(destination)):
                raise Exception(f"No cloud destination specified.")
            if isinstance(source_config := self.source, RCloneStore):
                # Here both a source and destination cloud configuration have been defined for this RCloner
                # object; meaning we are copying from one cloud source to another cloud destination; i.e. e.g.
                # from either Amazon S3 or Google Cloud Storage to either Amazon S3 or Google Cloud Storage.
                if not (source := source_config.path(source)):
                    raise Exception(f"No cloud source specified.")
                if (is_destination_folder and (copyto is True)) or is_destination_bucket_only:
                    # If the given destination looks like a folder (i.e. ends with a slash) or is
                    # only a bucket (i.e. contains on slash), then we are copying the source file
                    # into this folder or bucket; we explicitly then change the destination folder
                    # or bucket to be the actual path to the destination key by joining/suffixing
                    # given source file (base) name to the destination folder or bucket. FYI note
                    # in general we can not (straightforwardly) tell if a cloud path refers to a
                    # folder, so we use/respect a trailing slash in a path to mean that it is.
                    destination = cloud_path.join(destination, cloud_path.basename(source))
                    copyto = True
                with self.config_file(persist=dryrun is True) as source_and_destination_config_file:  # noqa
                    command_args = [f"{source_config.name}:{source}", f"{destination_config.name}:{destination}"]
                    source_s3 = isinstance(source_config, RCloneAmazon)
                    destination_s3 = isinstance(destination_config, RCloneAmazon)
                    return RCloneCommands.copy_command(command_args,
                                                       config=source_and_destination_config_file,
                                                       copyto=copyto,
                                                       source_s3=source_s3, destination_s3=destination_s3,
                                                       metadata=metadata, progress=progress, dryrun=dryrun,
                                                       process_info=process_info,
                                                       return_output=return_output,
                                                       raise_exception=raise_exception)
            else:
                # Here only a destination config cloud configuration has been defined for this RCloner
                # object; meaning we are copying from a local file source to some cloud destination;
                # i.e. e.g. from a local file to either Amazon S3 or Google Cloud Storage.
                if not (source := normalize_path(source)):
                    raise Exception(f"No file source specified.")
                elif not os.path.isfile(source):
                    raise Exception(f"Source file not found: {source}")
                if (is_destination_folder and (copyto is True)) or is_destination_bucket_only:
                    # If the given destination looks like a folder (i.e. ends with a slash) or is
                    # only a bucket (i.e. contains on slash), then we are copying the source file
                    # into this folder or bucket; we explicitly then change the destination folder
                    # or bucket to be the actual path to the destination key by joining/suffixing
                    # given source file (base) name to the destination folder or bucket. FYI note
                    # in general we can not (straightforwardly) tell if a cloud path refers to a
                    # folder, so we use/respect a trailing slash in a path to mean that it is.
                    destination = cloud_path.join(destination, os.path.basename(source))
                    copyto = True
                with destination_config.config_file(persist=dryrun is True) as destination_config_file:
                    command_args = [source, f"{destination_config.name}:{destination}"]
                    destination_s3 = isinstance(destination_config, RCloneAmazon)
                    return RCloneCommands.copy_command(command_args,
                                                       config=destination_config_file,
                                                       copyto=copyto, destination_s3=destination_s3,
                                                       metadata=metadata, progress=progress, dryrun=dryrun,
                                                       process_info=process_info,
                                                       return_output=return_output,
                                                       raise_exception=raise_exception)
        elif isinstance(source_config := self.source, RCloneStore):
            # Here only a source cloud configuration has been defined for this RCloner object;
            # meaning we are copying from some cloud source to a local file destination;
            # i.e. e.g. from either Amazon S3 or Google Cloud Storage to a local file.
            if source_config.bucket:
                # A path/bucket in the source RCloneStore is nothing more than an alternative
                # way of manually placing it at the beginning of the given source argument.
                source = cloud_path.join(source_config.bucket, source)
            if not (source := cloud_path.normalize(source)):
                raise Exception(f"No cloud source specified.")
            elif cloud_path.is_bucket_only(source):
                raise Exception(f"No cloud source key/file specified (only bucket: {source}).")
            elif not (destination := normalize_path(destination)):
                raise Exception(f"No file destination specified.")
            elif os.path.isdir(destination):
                # Normal/usually-desired case e.g.: cp s3://bucket/subfolder/file to destination/file
                if not os.access(destination, os.W_OK):
                    raise Exception(f"Destination directory is not writable: {destination}")
                destination = os.path.join(destination, cloud_path.basename(source))
            elif not (dirname := os.path.dirname(destination)):
                if not os.access(dirname := os.getcwd(), os.W_OK):
                    raise Exception(f"Destination directory is not writable: {dirname}")
            elif not os.path.isdir(dirname):
                raise Exception(f"Destination directory does not exist: {dirname}")
            elif not os.access(dirname, os.W_OK):
                raise Exception(f"Destination directory is not writable: {dirname}")
            source_s3 = isinstance(source_config, RCloneAmazon)
            with source_config.config_file(persist=dryrun is True) as source_config_file:  # noqa
                command_args = [f"{source_config.name}:{source}", destination]
                return RCloneCommands.copy_command(command_args,
                                                   config=source_config_file,
                                                   copyto=True, source_s3=source_s3,
                                                   progress=progress, dryrun=dryrun,
                                                   process_info=process_info,
                                                   return_output=return_output,
                                                   raise_exception=raise_exception)
        else:
            # Here not source or destination cloud configuration has been defined for this RCloner;
            # object; meaning this is (degenerate case of a) simple local file to file copy.
            if not (source := normalize_path(source)):
                raise Exception(f"No file source specified.")
            elif not os.path.isfile(source):
                raise Exception(f"Source file does not exist: {source}")
            elif not (destination := normalize_path(destination)):
                raise Exception(f"No file destination specified.")
            elif os.path.isdir(destination):
                if not os.access(destination, os.W_OK):
                    raise Exception(f"Destination directory is not writable: {destination}")
                destination = os.path.join(destination, cloud_path.basename(source))
            elif not (dirname := os.path.dirname(destination)):
                if not os.access(dirname := os.getcwd(), os.W_OK):
                    raise Exception(f"Destination directory is not writable: {dirname}")
            elif not os.path.isdir(dirname):
                raise Exception(f"Destination directory does not exist: {dirname}")
            elif not os.access(dirname, os.W_OK):
                raise Exception(f"Destination directory is not writable: {dirname}")
            command_args = [source, destination]
            return RCloneCommands.copy_command(command_args,
                                               copyto=True,
                                               progress=progress, dryrun=dryrun,
                                               process_info=process_info,
                                               return_output=return_output,
                                               raise_exception=raise_exception)

    def copy_to_bucket(self, *args, **kwargs) -> Union[bool, Tuple[bool, List[str]]]:
        kwargs["copyto"] = False
        return self.copy(*args, **kwargs)

    def copy_to_key(self, *args, **kwargs) -> Union[bool, Tuple[bool, List[str]]]:
        kwargs["copyto"] = True
        return self.copy(*args, **kwargs)
