"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmptyDirMedium = exports.Volume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Volume represents a named volume in a pod that may be accessed by any
 * container in the pod.
 *
 * Docker also has a concept of volumes, though it is somewhat looser and less
 * managed. In Docker, a volume is simply a directory on disk or in another
 * Container. Lifetimes are not managed and until very recently there were only
 * local-disk-backed volumes. Docker now provides volume drivers, but the
 * functionality is very limited for now (e.g. as of Docker 1.7 only one volume
 * driver is allowed per Container and there is no way to pass parameters to
 * volumes).
 *
 * A Kubernetes volume, on the other hand, has an explicit lifetime - the same
 * as the Pod that encloses it. Consequently, a volume outlives any Containers
 * that run within the Pod, and data is preserved across Container restarts. Of
 * course, when a Pod ceases to exist, the volume will cease to exist, too.
 * Perhaps more importantly than this, Kubernetes supports many types of
 * volumes, and a Pod can use any number of them simultaneously.
 *
 * At its core, a volume is just a directory, possibly with some data in it,
 * which is accessible to the Containers in a Pod. How that directory comes to
 * be, the medium that backs it, and the contents of it are determined by the
 * particular volume type used.
 *
 * To use a volume, a Pod specifies what volumes to provide for the Pod (the
 * .spec.volumes field) and where to mount those into Containers (the
 * .spec.containers[*].volumeMounts field).
 *
 * A process in a container sees a filesystem view composed from their Docker
 * image and volumes. The Docker image is at the root of the filesystem
 * hierarchy, and any volumes are mounted at the specified paths within the
 * image. Volumes can not mount onto other volumes
 */
class Volume {
    constructor(name, config) {
        this.name = name;
        this.config = config;
    }
    /**
     * Populate the volume from a ConfigMap.
     *
     * The configMap resource provides a way to inject configuration data into
     * Pods. The data stored in a ConfigMap object can be referenced in a volume
     * of type configMap and then consumed by containerized applications running
     * in a Pod.
     *
     * When referencing a configMap object, you can simply provide its name in the
     * volume to reference it. You can also customize the path to use for a
     * specific entry in the ConfigMap.
     *
     * @param configMap The config map to use to populate the volume.
     * @param options Options
     */
    static fromConfigMap(configMap, options = {}) {
        var _b;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `configmap-${configMap.name}`, {
            configMap: {
                name: configMap.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * An emptyDir volume is first created when a Pod is assigned to a Node, and
     * exists as long as that Pod is running on that node. As the name says, it is
     * initially empty. Containers in the Pod can all read and write the same
     * files in the emptyDir volume, though that volume can be mounted at the same
     * or different paths in each Container. When a Pod is removed from a node for
     * any reason, the data in the emptyDir is deleted forever.
     *
     * @see http://kubernetes.io/docs/user-guide/volumes#emptydir
     *
     * @param options - Additional options.
     */
    static fromEmptyDir(name, options = {}) {
        return new Volume(name, {
            emptyDir: {
                medium: options.medium,
                sizeLimit: options.sizeLimit
                    ? `${options.sizeLimit.toMebibytes()}Mi`
                    : undefined,
            },
        });
    }
    /**
     * Populate the volume from a Secret.
     *
     * A secret volume is used to pass sensitive information, such as passwords, to Pods.
     * You can store secrets in the Kubernetes API and mount them as files for use by pods
     * without coupling to Kubernetes directly.
     *
     * secret volumes are backed by tmpfs (a RAM-backed filesystem)
     * so they are never written to non-volatile storage.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#secret
     *
     * @param secret The secret to use to populate the volume.
     * @param options Options
     */
    static fromSecret(secret, options = {}) {
        var _b;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `secret-${secret.name}`, {
            secret: {
                secretName: secret.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            name: this.name,
            ...this.config,
        };
    }
}
exports.Volume = Volume;
_a = JSII_RTTI_SYMBOL_1;
Volume[_a] = { fqn: "cdk8s-plus-22.Volume", version: "1.0.0-beta.140" };
/**
  * @internal
 */
Volume.renderItems = (items) => {
    if (!items) {
        return undefined;
    }
    const result = new Array();
    for (const key of Object.keys(items).sort()) {
        result.push({
            key,
            path: items[key].path,
            mode: items[key].mode,
        });
    }
    return result;
};
/**
 * The medium on which to store the volume.
 */
var EmptyDirMedium;
(function (EmptyDirMedium) {
    /**
     * The default volume of the backing node.
     */
    EmptyDirMedium["DEFAULT"] = "";
    /**
     * Mount a tmpfs (RAM-backed filesystem) for you instead. While tmpfs is very
     * fast, be aware that unlike disks, tmpfs is cleared on node reboot and any
     * files you write will count against your Container's memory limit.
     */
    EmptyDirMedium["MEMORY"] = "Memory";
})(EmptyDirMedium = exports.EmptyDirMedium || (exports.EmptyDirMedium = {}));
//# sourceMappingURL=data:application/json;base64,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