"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RestartPolicy = exports.Pod = exports.PodTemplate = exports.PodSpec = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const container_1 = require("./container");
const k8s = require("./imports/k8s");
/**
 * Provides read/write capabilities ontop of a `PodSpecProps`.
 */
class PodSpec {
    constructor(props = {}) {
        this._containers = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    addContainer(container) {
        const impl = new container_1.Container(container);
        this._containers.push(impl);
        return impl;
    }
    addVolume(volume) {
        const existingVolume = this._volumes.get(volume.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${volume.name} already exists`);
        }
        this._volumes.set(volume.name, volume);
    }
    /**
     * @internal
     */
    _toPodSpec() {
        var _d;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        for (const container of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                addVolume(mount.volume);
            }
            containers.push(container._toKube());
        }
        for (const volume of this.volumes) {
            addVolume(volume);
        }
        function addVolume(volume) {
            const existingVolume = volumes.get(volume.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== volume) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${volume.name}`);
            }
            volumes.set(volume.name, volume);
        }
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_d = this.serviceAccount) === null || _d === void 0 ? void 0 : _d.name,
            containers: containers,
            volumes: Array.from(volumes.values()).map(v => v._toKube()),
        };
    }
}
exports.PodSpec = PodSpec;
_a = JSII_RTTI_SYMBOL_1;
PodSpec[_a] = { fqn: "cdk8s-plus-22.PodSpec", version: "1.0.0-beta.140" };
/**
 * Provides read/write capabilities ontop of a `PodTemplateProps`.
 */
class PodTemplate extends PodSpec {
    constructor(props = {}) {
        super(props);
        this.podMetadata = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadata);
    }
    /**
     * @internal
     */
    _toPodTemplateSpec() {
        return {
            metadata: this.podMetadata.toJson(),
            spec: this._toPodSpec(),
        };
    }
}
exports.PodTemplate = PodTemplate;
_b = JSII_RTTI_SYMBOL_1;
PodTemplate[_b] = { fqn: "cdk8s-plus-22.PodTemplate", version: "1.0.0-beta.140" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._spec._toPodSpec() }),
        });
        this._spec = new PodSpec(props);
    }
    get containers() {
        return this._spec.containers;
    }
    get volumes() {
        return this._spec.volumes;
    }
    get restartPolicy() {
        return this._spec.restartPolicy;
    }
    get serviceAccount() {
        return this._spec.serviceAccount;
    }
    addContainer(container) {
        return this._spec.addContainer(container);
    }
    addVolume(volume) {
        return this._spec.addVolume(volume);
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-22.Pod", version: "1.0.0-beta.140" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
//# sourceMappingURL=data:application/json;base64,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