"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const k8s = require("./imports/k8s");
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_a = JSII_RTTI_SYMBOL_1;
EnvValue[_a] = { fqn: "cdk8s-plus-22.EnvValue", version: "1.0.0-beta.140" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        var _d, _e, _f, _g;
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        this.name = (_d = props.name) !== null && _d !== void 0 ? _d : 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._args = props.args;
        this._env = (_e = props.env) !== null && _e !== void 0 ? _e : {};
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup;
        this.resources = props.resources;
        this.workingDir = props.workingDir;
        this.mounts = (_f = props.volumeMounts) !== null && _f !== void 0 ? _f : [];
        this.imagePullPolicy = (_g = props.imagePullPolicy) !== null && _g !== void 0 ? _g : ImagePullPolicy.ALWAYS;
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Add an environment value to the container. The variable value can come
     * from various dynamic sources such a secrets of config maps.
     *
     * @see EnvValue.fromXXX
     *
     * @param name - The variable name.
     * @param value - The variable value.
     */
    addEnv(name, value) {
        this._env[name] = value;
    }
    /**
     * The environment variables for this container.
     *
     * Returns a copy. To add environment variables use `addEnv()`.
     */
    get env() {
        return { ...this._env };
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param volume - The volume to mount.
     */
    mount(path, volume, options = {}) {
        this.mounts.push({ path, volume, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        var _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        // Resource requests and limits
        const cpuLimit = (_f = (_e = (_d = this.resources) === null || _d === void 0 ? void 0 : _d.cpu) === null || _e === void 0 ? void 0 : _e.limit) === null || _f === void 0 ? void 0 : _f.amount;
        const cpuRequest = (_j = (_h = (_g = this.resources) === null || _g === void 0 ? void 0 : _g.cpu) === null || _h === void 0 ? void 0 : _h.request) === null || _j === void 0 ? void 0 : _j.amount;
        const memoryLimit = (_l = (_k = this.resources) === null || _k === void 0 ? void 0 : _k.memory) === null || _l === void 0 ? void 0 : _l.limit;
        const memoryRequest = (_o = (_m = this.resources) === null || _m === void 0 ? void 0 : _m.memory) === null || _o === void 0 ? void 0 : _o.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: limits,
                requests: requests,
            };
        }
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports,
            volumeMounts,
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: renderEnv(this._env),
            readinessProbe: (_p = this._readiness) === null || _p === void 0 ? void 0 : _p._toKube(this),
            livenessProbe: (_q = this._liveness) === null || _q === void 0 ? void 0 : _q._toKube(this),
            startupProbe: (_r = this._startup) === null || _r === void 0 ? void 0 : _r._toKube(this),
            resources: resourceRequirements,
        };
    }
}
exports.Container = Container;
_b = JSII_RTTI_SYMBOL_1;
Container[_b] = { fqn: "cdk8s-plus-22.Container", version: "1.0.0-beta.140" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_c = JSII_RTTI_SYMBOL_1;
Cpu[_c] = { fqn: "cdk8s-plus-22.Cpu", version: "1.0.0-beta.140" };
function renderEnv(env) {
    const result = new Array();
    for (const [name, v] of Object.entries(env)) {
        result.push({
            name,
            value: v.value,
            valueFrom: v.valueFrom,
        });
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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