# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyairvisual']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.0', 'numpy>=1.22.0', 'pysmb>=1.2.6,<2.0.0']

setup_kwargs = {
    'name': 'pyairvisual',
    'version': '2022.12.1',
    'description': 'A simple API for AirVisual air quality data',
    'long_description': '# ☀️ pyairvisual: a thin Python wrapper for the AirVisual© API\n\n[![CI][ci-badge]][ci]\n[![PyPI][pypi-badge]][pypi]\n[![Version][version-badge]][version]\n[![License][license-badge]][license]\n[![Code Coverage][codecov-badge]][codecov]\n[![Maintainability][maintainability-badge]][maintainability]\n\n<a href="https://www.buymeacoffee.com/bachya1208P" target="_blank"><img src="https://cdn.buymeacoffee.com/buttons/default-orange.png" alt="Buy Me A Coffee" height="41" width="174"></a>\n\n`pyairvisual` is a simple, clean, well-tested library for interacting with\n[AirVisual][airvisual] to retrieve air quality information.\n\n- [Python Versions](#python-versions)\n- [Installation](#installation)\n- [API Key](#api-key)\n  - [Community](#community)\n  - [Startup](#startup)\n  - [Enterprise](#enterprise)\n- [Usage](#usage)\n  - [Using the Cloud API](#using-the-cloud-api)\n  - [Working with Node/Pro Units](#working-with-node-pro-units)\n- [Contributing](#contributing)\n\n# Python Versions\n\n`pyairvisual` is currently supported on:\n\n- Python 3.9\n- Python 3.10\n- Python 3.11\n\n# Installation\n\n```bash\npip install pyairvisual\n```\n\n# API Key\n\nYou can get an AirVisual API key from [the AirVisual API site][airvisual-api].\nDepending on the plan you choose, more functionality will be available from the API:\n\n## Community\n\nThe Community Plan gives access to:\n\n- List supported countries\n- List supported states\n- List supported cities\n- Get data from the nearest city based on IP address\n- Get data from the nearest city based on latitude/longitude\n- Get data from a specific city\n\n## Startup\n\nThe Startup Plan gives access to:\n\n- List supported stations in a city\n- Get data from the nearest station based on IP address\n- Get data from the nearest station based on latitude/longitude\n- Get data from a specific station\n\n## Enterprise\n\nThe Enterprise Plan gives access to:\n\n- Get a global city ranking of air quality\n\n# Usage\n\n## Using the Cloud API\n\n```python\nimport asyncio\n\nfrom pyairvisual.cloud_api import CloudAPI\n\n\nasync def main() -> None:\n    """Run!"""\n    cloud_api = CloudAPI("<YOUR_AIRVISUAL_API_KEY>")\n\n    # Get data based on the city nearest to your IP address:\n    data = await cloud_api.air_quality.nearest_city()\n\n    # ...or get data based on the city nearest to a latitude/longitude:\n    data = await cloud_api.air_quality.nearest_city(\n        latitude=39.742599, longitude=-104.9942557\n    )\n\n    # ...or get it explicitly:\n    data = await cloud_api.air_quality.city(\n        city="Los Angeles", state="California", country="USA"\n    )\n\n    # If you have the appropriate API key, you can also get data based on\n    # station (nearest or explicit):\n    data = await cloud_api.air_quality.nearest_station()\n    data = await cloud_api.air_quality.nearest_station(\n        latitude=39.742599, longitude=-104.9942557\n    )\n    data = await cloud_api.air_quality.station(\n        station="US Embassy in Beijing",\n        city="Beijing",\n        state="Beijing",\n        country="China",\n    )\n\n    # With the appropriate API key, you can get an air quality ranking:\n    data = await cloud_api.air_quality.ranking()\n\n    # pyairvisual gives you several methods to look locations up:\n    countries = await cloud_api.supported.countries()\n    states = await cloud_api.supported.states("USA")\n    cities = await cloud_api.supported.cities("USA", "Colorado")\n    stations = await cloud_api.supported.stations("USA", "Colorado", "Denver")\n\n\nasyncio.run(main())\n```\n\nBy default, the library creates a new connection to AirVisual with each coroutine. If\nyou are calling a large number of coroutines (or merely want to squeeze out every second\nof runtime savings possible), an [`aiohttp`][aiohttp] `ClientSession` can be used for\nconnection pooling:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pyairvisual.cloud_api import CloudAPI\n\n\nasync def main() -> None:\n    """Run!"""\n    async with ClientSession() as session:\n        cloud_api = CloudAPI("<YOUR_AIRVISUAL_API_KEY>", session=session)\n\n        # ...\n\n\nasyncio.run(main())\n```\n\n## Working with Node/Pro Units\n\n`pyairvisual` also allows users to interact with [Node/Pro units][airvisual-pro], both via\nthe cloud API:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pyairvisual.cloud_api import CloudAPI\n\n\nasync def main() -> None:\n    """Run!"""\n    cloud_api = CloudAPI("<YOUR_AIRVISUAL_API_KEY>")\n\n    # The Node/Pro unit ID can be retrieved from the "API" section of the cloud\n    # dashboard:\n    data = await cloud_api.node.get_by_node_id("<NODE_ID>")\n\n\nasyncio.run(main())\n```\n\n...or over the local network via Samba (the unit password can be found\n[on the device itself][airvisual-samba-instructions]):\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom pyairvisual.node import NodeSamba\n\n\nasync def main() -> None:\n    """Run!"""\n    async with NodeSamba("<IP_ADDRESS_OR_HOST>", "<PASSWORD>") as node:\n        measurements = await node.async_get_latest_measurements()\n\n        # Can take some optional parameters:\n        #   1. include_trends: include trends (defaults to True)\n        #   2. measurements_to_use: the number of measurements to use when calculating\n        #      trends (defaults to -1, which means "use all measurements")\n        history = await node.async_get_history()\n\n\nasyncio.run(main())\n```\n\nCheck out the examples, the tests, and the source files themselves for method\nsignatures and more examples.\n\n# Contributing\n\nThanks to all of [our contributors][contributors] so far!\n\n1. [Check for open features/bugs][issues] or [initiate a discussion on one][new-issue].\n2. [Fork the repository][fork].\n3. (_optional, but highly recommended_) Create a virtual environment: `python3 -m venv .venv`\n4. (_optional, but highly recommended_) Enter the virtual environment: `source ./.venv/bin/activate`\n5. Install the dev environment: `script/setup`\n6. Code your new feature or bug fix on a new branch.\n7. Write tests that cover your new functionality.\n8. Run tests and ensure 100% code coverage: `poetry run pytest --cov pyairvisual tests`\n9. Update `README.md` with any new documentation.\n10. Submit a pull request!\n\n[aiohttp]: https://github.com/aio-libs/aiohttp\n[airvisual]: https://www.airvisual.com/\n[airvisual-api]: https://www.airvisual.com/user/api\n[airvisual-pro]: https://www.airvisual.com/air-quality-monitor\n[airvisual-samba-instructions]: https://support.airvisual.com/en/articles/3029331-download-the-airvisual-node-pro-s-data-using-samba\n[ci-badge]: https://github.com/bachya/pyairvisual/workflows/CI/badge.svg\n[ci]: https://github.com/bachya/pyairvisual/actions\n[codecov-badge]: https://codecov.io/gh/bachya/pyairvisual/branch/dev/graph/badge.svg\n[codecov]: https://codecov.io/gh/bachya/pyairvisual\n[contributors]: https://github.com/bachya/pyairvisual/graphs/contributors\n[fork]: https://github.com/bachya/pyairvisual/fork\n[issues]: https://github.com/bachya/pyairvisual/issues\n[license-badge]: https://img.shields.io/pypi/l/pyairvisual.svg\n[license]: https://github.com/bachya/pyairvisual/blob/main/LICENSE\n[maintainability-badge]: https://api.codeclimate.com/v1/badges/a03c9e96f19a3dc37f98/maintainability\n[maintainability]: https://codeclimate.com/github/bachya/pyairvisual/maintainability\n[new-issue]: https://github.com/bachya/pyairvisual/issues/new\n[new-issue]: https://github.com/bachya/pyairvisual/issues/new\n[pypi-badge]: https://img.shields.io/pypi/v/pyairvisual.svg\n[pypi]: https://pypi.python.org/pypi/pyairvisual\n[version-badge]: https://img.shields.io/pypi/pyversions/pyairvisual.svg\n[version]: https://pypi.python.org/pypi/pyairvisual\n',
    'author': 'Aaron Bach',
    'author_email': 'bachya1208@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/bachya/pyairvisual',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9.0,<4.0.0',
}


setup(**setup_kwargs)
