"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostedRotationType = exports.HostedRotation = exports.RotationSchedule = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const secretsmanager_generated_1 = require("./secretsmanager.generated");
/**
 * A rotation schedule.
 */
class RotationSchedule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_RotationScheduleProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, RotationSchedule);
            }
            throw error;
        }
        if ((!props.rotationLambda && !props.hostedRotation) || (props.rotationLambda && props.hostedRotation)) {
            throw new Error('One of `rotationLambda` or `hostedRotation` must be specified.');
        }
        if (props.rotationLambda?.permissionsNode.defaultChild) {
            if (props.secret.encryptionKey) {
                props.secret.encryptionKey.grantEncryptDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, props.rotationLambda.grantPrincipal));
            }
            props.rotationLambda.grantInvoke(new iam.ServicePrincipal('secretsmanager.amazonaws.com'));
            props.rotationLambda.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'secretsmanager:DescribeSecret',
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:PutSecretValue',
                    'secretsmanager:UpdateSecretVersionStage',
                ],
                resources: [props.secret.secretFullArn ? props.secret.secretFullArn : `${props.secret.secretArn}-??????`],
            }));
            props.rotationLambda.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'secretsmanager:GetRandomPassword',
                ],
                resources: ['*'],
            }));
        }
        let automaticallyAfterDays = undefined;
        if (props.automaticallyAfter?.toMilliseconds() !== 0) {
            automaticallyAfterDays = props.automaticallyAfter?.toDays() || 30;
        }
        let rotationRules = undefined;
        if (automaticallyAfterDays !== undefined) {
            rotationRules = {
                automaticallyAfterDays,
            };
        }
        new secretsmanager_generated_1.CfnRotationSchedule(this, 'Resource', {
            secretId: props.secret.secretArn,
            rotationLambdaArn: props.rotationLambda?.functionArn,
            hostedRotationLambda: props.hostedRotation?.bind(props.secret, this),
            rotationRules,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
    }
}
exports.RotationSchedule = RotationSchedule;
_a = JSII_RTTI_SYMBOL_1;
RotationSchedule[_a] = { fqn: "@aws-cdk/aws-secretsmanager.RotationSchedule", version: "1.183.0" };
/**
 * A hosted rotation
 */
class HostedRotation {
    constructor(type, props, masterSecret) {
        this.type = type;
        this.props = props;
        this.masterSecret = masterSecret;
        if (type.isMultiUser && !masterSecret) {
            throw new Error('The `masterSecret` must be specified when using the multi user scheme.');
        }
    }
    /** MySQL Single User */
    static mysqlSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mysqlSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MYSQL_SINGLE_USER, options);
    }
    /** MySQL Multi User */
    static mysqlMultiUser(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mysqlMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MYSQL_MULTI_USER, options, options.masterSecret);
    }
    /** PostgreSQL Single User */
    static postgreSqlSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.postgreSqlSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.POSTGRESQL_SINGLE_USER, options);
    }
    /** PostgreSQL Multi User */
    static postgreSqlMultiUser(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.postgreSqlMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.POSTGRESQL_MULTI_USER, options, options.masterSecret);
    }
    /** Oracle Single User */
    static oracleSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.oracleSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.ORACLE_SINGLE_USER, options);
    }
    /** Oracle Multi User */
    static oracleMultiUser(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.oracleMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.ORACLE_MULTI_USER, options, options.masterSecret);
    }
    /** MariaDB Single User */
    static mariaDbSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mariaDbSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MARIADB_SINGLE_USER, options);
    }
    /** MariaDB Multi User */
    static mariaDbMultiUser(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mariaDbMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MARIADB_MULTI_USER, options, options.masterSecret);
    }
    /** SQL Server Single User */
    static sqlServerSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sqlServerSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.SQLSERVER_SINGLE_USER, options);
    }
    /** SQL Server Multi User */
    static sqlServerMultiUser(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sqlServerMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.SQLSERVER_MULTI_USER, options, options.masterSecret);
    }
    /** Redshift Single User */
    static redshiftSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.redshiftSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.REDSHIFT_SINGLE_USER, options);
    }
    /** Redshift Multi User */
    static redshiftMultiUser(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.redshiftMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.REDSHIFT_MULTI_USER, options, options.masterSecret);
    }
    /** MongoDB Single User */
    static mongoDbSingleUser(options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SingleUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mongoDbSingleUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MONGODB_SINGLE_USER, options);
    }
    /** MongoDB Multi User */
    static mongoDbMultiUser(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_MultiUserHostedRotationOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.mongoDbMultiUser);
            }
            throw error;
        }
        return new HostedRotation(HostedRotationType.MONGODB_MULTI_USER, options, options.masterSecret);
    }
    /**
     * Binds this hosted rotation to a secret
     */
    bind(secret, scope) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_ISecret(secret);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-secretsmanager-rotationschedule-hostedrotationlambda.html
        core_1.Stack.of(scope).addTransform('AWS::SecretsManager-2020-07-23');
        if (!this.props.vpc && this.props.securityGroups) {
            throw new Error('`vpc` must be specified when specifying `securityGroups`.');
        }
        if (this.props.vpc) {
            this._connections = new ec2.Connections({
                securityGroups: this.props.securityGroups || [new ec2.SecurityGroup(scope, 'SecurityGroup', {
                        vpc: this.props.vpc,
                    })],
            });
        }
        // Prevent master secret deletion when rotation is in place
        if (this.masterSecret) {
            this.masterSecret.denyAccountRootDelete();
        }
        return {
            rotationType: this.type.name,
            kmsKeyArn: secret.encryptionKey?.keyArn,
            masterSecretArn: this.masterSecret?.secretArn,
            masterSecretKmsKeyArn: this.masterSecret?.encryptionKey?.keyArn,
            rotationLambdaName: this.props.functionName,
            vpcSecurityGroupIds: this._connections?.securityGroups?.map(s => s.securityGroupId).join(','),
            vpcSubnetIds: this.props.vpc?.selectSubnets(this.props.vpcSubnets).subnetIds.join(','),
        };
    }
    /**
     * Security group connections for this hosted rotation
     */
    get connections() {
        if (!this.props.vpc) {
            throw new Error('Cannot use connections for a hosted rotation that is not deployed in a VPC');
        }
        // If we are in a vpc and bind() has been called _connections should be defined
        if (!this._connections) {
            throw new Error('Cannot use connections for a hosted rotation that has not been bound to a secret');
        }
        return this._connections;
    }
}
exports.HostedRotation = HostedRotation;
_b = JSII_RTTI_SYMBOL_1;
HostedRotation[_b] = { fqn: "@aws-cdk/aws-secretsmanager.HostedRotation", version: "1.183.0" };
/**
 * Hosted rotation type
 */
class HostedRotationType {
    /**
     * @param name The type of rotation
     * @param isMultiUser Whether the rotation uses the mutli user scheme
     */
    constructor(name, isMultiUser) {
        this.name = name;
        this.isMultiUser = isMultiUser;
    }
}
exports.HostedRotationType = HostedRotationType;
_c = JSII_RTTI_SYMBOL_1;
HostedRotationType[_c] = { fqn: "@aws-cdk/aws-secretsmanager.HostedRotationType", version: "1.183.0" };
/** MySQL Single User */
HostedRotationType.MYSQL_SINGLE_USER = new HostedRotationType('MySQLSingleUser');
/** MySQL Multi User */
HostedRotationType.MYSQL_MULTI_USER = new HostedRotationType('MySQLMultiUser', true);
/** PostgreSQL Single User */
HostedRotationType.POSTGRESQL_SINGLE_USER = new HostedRotationType('PostgreSQLSingleUser');
/** PostgreSQL Multi User */
HostedRotationType.POSTGRESQL_MULTI_USER = new HostedRotationType('PostgreSQLMultiUser', true);
/** Oracle Single User */
HostedRotationType.ORACLE_SINGLE_USER = new HostedRotationType('OracleSingleUser');
/** Oracle Multi User */
HostedRotationType.ORACLE_MULTI_USER = new HostedRotationType('OracleMultiUser', true);
/** MariaDB Single User */
HostedRotationType.MARIADB_SINGLE_USER = new HostedRotationType('MariaDBSingleUser');
/** MariaDB Multi User */
HostedRotationType.MARIADB_MULTI_USER = new HostedRotationType('MariaDBMultiUser', true);
/** SQL Server Single User */
HostedRotationType.SQLSERVER_SINGLE_USER = new HostedRotationType('SQLServerSingleUser');
/** SQL Server Multi User */
HostedRotationType.SQLSERVER_MULTI_USER = new HostedRotationType('SQLServerMultiUser', true);
/** Redshift Single User */
HostedRotationType.REDSHIFT_SINGLE_USER = new HostedRotationType('RedshiftSingleUser');
/** Redshift Multi User */
HostedRotationType.REDSHIFT_MULTI_USER = new HostedRotationType('RedshiftMultiUser', true);
/** MongoDB Single User */
HostedRotationType.MONGODB_SINGLE_USER = new HostedRotationType('MongoDBSingleUser');
/** MongoDB Multi User */
HostedRotationType.MONGODB_MULTI_USER = new HostedRotationType('MongoDBMultiUser', true);
//# sourceMappingURL=data:application/json;base64,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