"use strict";
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const aws_prototyping_1 = require("../src/packs/aws-prototyping");
const expectedErrors = [
    "AwsPrototyping-AutoScalingLaunchConfigPublicIpDisabled",
    "AwsPrototyping-Cloud9InstanceNoIngressSystemsManager",
    "AwsPrototyping-CloudFrontDistributionGeoRestrictions",
    "AwsPrototyping-CloudFrontDistributionS3OriginAccessIdentity",
    "AwsPrototyping-CodeBuildProjectEnvVarAwsCred",
    "AwsPrototyping-CognitoUserPoolNoUnauthenticatedLogins",
    "AwsPrototyping-CognitoUserPoolStrongPasswordPolicy",
    "AwsPrototyping-DMSReplicationNotPublic",
    "AwsPrototyping-DocumentDBCredentialsInSecretsManager",
    "AwsPrototyping-EC2InstancesInVPC",
    "AwsPrototyping-EC2RestrictedCommonPorts",
    "AwsPrototyping-EC2RestrictedInbound",
    "AwsPrototyping-EC2RestrictedSSH",
    "AwsPrototyping-ECROpenAccess",
    "AwsPrototyping-EKSClusterNoEndpointPublicAccess",
    "AwsPrototyping-ElastiCacheClusterInVPC",
    "AwsPrototyping-ElasticBeanstalkManagedUpdatesEnabled",
    "AwsPrototyping-EventBusOpenAccess",
    "AwsPrototyping-IAMPolicyNoStatementsWithAdminAccess",
    "AwsPrototyping-LambdaFunctionPublicAccessProhibited",
    "AwsPrototyping-LambdaFunctionUrlAuth",
    "AwsPrototyping-NeptuneClusterAutomaticMinorVersionUpgrade",
    "AwsPrototyping-OpenSearchInVPCOnly",
    "AwsPrototyping-OpenSearchNoUnsignedOrAnonymousAccess",
    "AwsPrototyping-RDSAutomaticMinorVersionUpgradeEnabled",
    "AwsPrototyping-RDSInstancePublicAccess",
    "AwsPrototyping-RDSRestrictedInbound",
    "AwsPrototyping-RedshiftClusterInVPC",
    "AwsPrototyping-RedshiftClusterPublicAccess",
    "AwsPrototyping-RedshiftClusterVersionUpgrade",
    "AwsPrototyping-S3BucketLevelPublicAccessProhibited",
    "AwsPrototyping-S3BucketPublicReadProhibited",
    "AwsPrototyping-S3BucketPublicWriteProhibited",
    "AwsPrototyping-S3WebBucketOAIAccess",
    "AwsPrototyping-VPCDefaultSecurityGroupClosed",
    "AwsPrototyping-VPCSubnetAutoAssignPublicIpDisabled",
];
const expectedWarnings = [
    "AwsPrototyping-APIGWAssociatedWithWAF",
    "AwsPrototyping-APIGWAuthorization",
    "AwsPrototyping-APIGWRequestValidation",
    "AwsPrototyping-AthenaWorkgroupEncryptedQueryResults",
    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionNoOutdatedSSL",
    "AwsPrototyping-CloudFrontDistributionWAFIntegration",
    "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
    "AwsPrototyping-CodeBuildProjectManagedImages",
    "AwsPrototyping-CodeBuildProjectPrivilegedModeDisabled",
    "AwsPrototyping-CognitoUserPoolAdvancedSecurityModeEnforced",
    "AwsPrototyping-CognitoUserPoolAPIGWAuthorizer",
    "AwsPrototyping-CognitoUserPoolMFA",
    "AwsPrototyping-DocumentDBClusterEncryptionAtRest",
    "AwsPrototyping-DocumentDBClusterNonDefaultPort",
    "AwsPrototyping-DAXEncrypted",
    "AwsPrototyping-EC2EBSVolumeEncrypted",
    "AwsPrototyping-EC2InstanceNoPublicIp",
    "AwsPrototyping-EC2InstanceProfileAttached",
    "AwsPrototyping-EC2SecurityGroupDescription",
    "AwsPrototyping-ECSTaskDefinitionContainerLogging",
    "AwsPrototyping-ECSTaskDefinitionNoEnvironmentVariables",
    "AwsPrototyping-EFSEncrypted",
    "AwsPrototyping-EKSClusterControlPlaneLogs",
    "AwsPrototyping-ElastiCacheClusterNonDefaultPort",
    "AwsPrototyping-ElastiCacheRedisClusterEncryption",
    "AwsPrototyping-ElastiCacheRedisClusterRedisAuth",
    "AwsPrototyping-ElasticBeanstalkEC2InstanceLogsToS3",
    "AwsPrototyping-ElasticBeanstalkVPCSpecified",
    "AwsPrototyping-ALBHttpToHttpsRedirection",
    "AwsPrototyping-ELBTlsHttpsListenersOnly",
    "AwsPrototyping-EMRAuthEC2KeyPairOrKerberos",
    "AwsPrototyping-EMREncryptionInTransit",
    "AwsPrototyping-EMRLocalDiskEncryption",
    "AwsPrototyping-GlueEncryptedCloudWatchLogs",
    "AwsPrototyping-GlueJobBookmarkEncrypted",
    "AwsPrototyping-IAMNoManagedPolicies",
    "AwsPrototyping-IAMNoWildcardPermissions",
    "AwsPrototyping-IAMPolicyNoStatementsWithFullAccess",
    "AwsPrototyping-KinesisDataFirehoseSSE",
    "AwsPrototyping-KinesisDataStreamDefaultKeyWhenSSE",
    "AwsPrototyping-KinesisDataStreamSSE",
    "AwsPrototyping-KMSBackingKeyRotationEnabled",
    "AwsPrototyping-LambdaLatestVersion",
    "AwsPrototyping-MediaStoreContainerCORSPolicy",
    "AwsPrototyping-MediaStoreContainerHasContainerPolicy",
    "AwsPrototyping-MediaStoreContainerSSLRequestsOnly",
    "AwsPrototyping-MSKBrokerToBrokerTLS",
    "AwsPrototyping-MSKClientToBrokerTLS",
    "AwsPrototyping-NeptuneClusterEncryptionAtRest",
    "AwsPrototyping-NeptuneClusterIAMAuth",
    "AwsPrototyping-OpenSearchAllowlistedIPs",
    "AwsPrototyping-OpenSearchEncryptedAtRest",
    "AwsPrototyping-OpenSearchNodeToNodeEncryption",
    "AwsPrototyping-QuicksightSSLConnections",
    "AwsPrototyping-AuroraMySQLPostgresIAMAuth",
    "AwsPrototyping-RDSNonDefaultPort",
    "AwsPrototyping-RDSStorageEncrypted",
    "AwsPrototyping-RedshiftClusterEncryptionAtRest",
    "AwsPrototyping-RedshiftClusterNonDefaultPort",
    "AwsPrototyping-RedshiftClusterNonDefaultUsername",
    "AwsPrototyping-RedshiftRequireTlsSSL",
    "AwsPrototyping-S3BucketLoggingEnabled",
    "AwsPrototyping-S3BucketServerSideEncryptionEnabled",
    "AwsPrototyping-S3BucketSSLRequestsOnly",
    "AwsPrototyping-SageMakerNotebookInstanceKMSKeyConfigured",
    "AwsPrototyping-SageMakerNotebookInVPC",
    "AwsPrototyping-SageMakerNotebookNoDirectInternetAccess",
    "AwsPrototyping-SecretsManagerRotationEnabled",
    "AwsPrototyping-SNSEncryptedKMS",
    "AwsPrototyping-SNSTopicSSLPublishOnly",
    "AwsPrototyping-SQSQueueSSE",
    "AwsPrototyping-SQSQueueSSLRequestsOnly",
    "AwsPrototyping-TimestreamDatabaseCustomerManagedKey",
    "AwsPrototyping-VPCNoNACLs",
];
describe("Check NagPack Details", () => {
    describe("AwsPrototyping-CDK", () => {
        class AwsPrototypingChecksExtended extends aws_prototyping_1.AwsPrototypingChecks {
            constructor() {
                super(...arguments);
                this.actualWarnings = new Array();
                this.actualErrors = new Array();
            }
            applyRule(params) {
                const ruleSuffix = params.ruleSuffixOverride
                    ? params.ruleSuffixOverride
                    : params.rule.name;
                const ruleId = `${pack.readPackName}-${ruleSuffix}`;
                if (params.level === cdk_nag_1.NagMessageLevel.WARN) {
                    this.actualWarnings.push(ruleId);
                }
                else {
                    this.actualErrors.push(ruleId);
                }
            }
        }
        const pack = new AwsPrototypingChecksExtended();
        test("Pack Name is correct", () => {
            expect(pack.readPackName).toStrictEqual("AwsPrototyping");
        });
        test("Pack contains expected warning and error rules", () => {
            jest.spyOn(pack, "applyRule");
            const stack = new aws_cdk_lib_1.Stack();
            aws_cdk_lib_1.Aspects.of(stack).add(pack);
            new aws_cdk_lib_1.CfnResource(stack, "rTestResource", { type: "foo" });
            assert_1.SynthUtils.synthesize(stack).messages;
            expect(pack.actualWarnings.sort()).toEqual(expectedWarnings.sort());
            expect(pack.actualErrors.sort()).toEqual(expectedErrors.sort());
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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