

import pandas as pd
import numpy as np
from sklearn.ensemble import RandomForestClassifier
from sklearn.metrics import accuracy_score
from deap import base, creator, tools, algorithms

def prepare_data(data, target_column):
    X = data.drop(target_column, axis=1).values
    y = data[target_column].values
    return X, y

def setup_deap(X):
    if not hasattr(creator, 'FitnessMax'):
        creator.create("FitnessMax", base.Fitness, weights=(1.0,))
    
    if not hasattr(creator, 'Individual'):
        creator.create("Individual", list, fitness=creator.FitnessMax)
    
    toolbox = base.Toolbox()
    toolbox.register("attr_bool", np.random.choice, [0, 1])
    toolbox.register("individual", tools.initRepeat, creator.Individual, toolbox.attr_bool, n=len(X[0]))
    toolbox.register("population", tools.initRepeat, list, toolbox.individual)
    
    return toolbox

def eval_individual(individual, X, y):
    selected_features = [i for i, gene in enumerate(individual) if gene]
    if not selected_features:
        return 0.0,
    
    X_subset = X[:, selected_features]
    clf = RandomForestClassifier(random_state=42, n_jobs=-1)
    clf.fit(X_subset, y)
    y_pred = clf.predict(X_subset)
    accuracy = accuracy_score(y, y_pred)
    
    return accuracy,

def run_ga(toolbox, X, y, data_columns, population_size, generations):
    toolbox.register("evaluate", eval_individual, X=X, y=y)
    toolbox.register("mate", tools.cxTwoPoint)
    toolbox.register("mutate", tools.mutFlipBit, indpb=0.1)
    toolbox.register("select", tools.selTournament, tournsize=3)
    
    population = toolbox.population(n=population_size)
    
    for gen in range(generations):
        fitness_values = list(map(toolbox.evaluate, population))
        for ind, fit in zip(population, fitness_values):
            ind.fitness.values = fit
        
        offspring = toolbox.select(population, len(population))
        offspring = list(map(toolbox.clone, offspring))
        
        for child1, child2 in zip(offspring[::2], offspring[1::2]):
            if np.random.rand() < 0.7:
                toolbox.mate(child1, child2)
                del child1.fitness.values
                del child2.fitness.values
        
        for mutant in offspring:
            if np.random.rand() < 0.2:
                toolbox.mutate(mutant)
                del mutant.fitness.values
        
        fitness_values = list(map(toolbox.evaluate, offspring))
        for ind, fit in zip(offspring, fitness_values):
            ind.fitness.values = fit
        
        population[:] = offspring
        fits = [ind.fitness.values[0] for ind in population]
        length = len(population)
        mean = sum(fits) / length
        sum2 = sum(x*x for x in fits)
        std = abs(sum2 / length - mean**2)**0.5
        
        print(f"Generation {gen + 1} Summary:")
        print(f"  Min Fitness: {min(fits)}")
        print(f"  Max Fitness: {max(fits)}")
        print(f"  Avg Fitness: {mean}")
        print(f"  Std Fitness: {std}")
    
    best_individual = tools.selBest(population, k=1)[0]
    print("\nBest Individual:", best_individual)
    print("Best Fitness:", best_individual.fitness.values)
    
    selected_features_indices = [i for i, gene in enumerate(best_individual) if gene]
    selected_columns = data_columns[selected_features_indices]
    print("Selected Columns:", selected_columns)
    
    return selected_columns

def save_selected_data(original_data, selected_columns, target_column, output_path="train_data_selected.csv"):
    selected_data = original_data[selected_columns].copy()
    selected_data[target_column] = original_data[target_column]
    selected_data.to_csv(output_path, index=False)
