"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionHandler = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const helpers_internal_1 = require("aws-cdk-lib/assertions/lib/helpers-internal");
const base_1 = require("./base");
class AssertionHandler extends base_1.CustomResourceHandler {
    async processEvent(request) {
        let actual = decodeCall(request.actual);
        const expected = decodeCall(request.expected);
        let result;
        const matcher = new MatchCreator(expected).getMatcher();
        console.log(`Testing equality between ${JSON.stringify(request.actual)} and ${JSON.stringify(request.expected)}`);
        const matchResult = matcher.test(actual);
        matchResult.finished();
        if (matchResult.hasFailed()) {
            result = {
                failed: true,
                assertion: JSON.stringify({
                    status: 'fail',
                    message: [
                        ...matchResult.toHumanStrings(),
                        JSON.stringify(matchResult.target, undefined, 2),
                    ].join('\n'),
                }),
            };
            if (request.failDeployment) {
                throw new Error(result.assertion);
            }
        }
        else {
            result = {
                assertion: JSON.stringify({
                    status: 'success',
                }),
            };
        }
        return result;
    }
}
exports.AssertionHandler = AssertionHandler;
class MatchCreator {
    constructor(obj) {
        this.parsedObj = {
            matcher: obj,
        };
    }
    /**
     * Return a Matcher that can be tested against the actual results.
     * This will convert the encoded matchers into their corresponding
     * assertions matcher.
     *
     * For example:
     *
     * ExpectedResult.objectLike({
     *   Messages: [{
     *     Body: Match.objectLike({
     *       Elements: Match.arrayWith([{ Asdf: 3 }]),
     *       Payload: Match.serializedJson({ key: 'value' }),
     *     }),
     *   }],
     * });
     *
     * Will be encoded as:
     * {
     *   $ObjectLike: {
     *     Messages: [{
     *       Body: {
     *         $ObjectLike: {
     *           Elements: {
     *             $ArrayWith: [{ Asdf: 3 }],
     *           },
     *           Payload: {
     *             $SerializedJson: { key: 'value' }
     *           }
     *         },
     *       },
     *     }],
     *   },
     * }
     *
     * Which can then be parsed by this function. For each key (recursively)
     * the parser will check if the value has one of the encoded matchers as a key
     * and if so, it will set the value as the Matcher. So,
     *
     * {
     *   Body: {
     *     $ObjectLike: {
     *       Elements: {
     *         $ArrayWith: [{ Asdf: 3 }],
     *       },
     *       Payload: {
     *         $SerializedJson: { key: 'value' }
     *       }
     *     },
     *   },
     * }
     *
     * Will be converted to
     * {
     *   Body: Match.objectLike({
     *     Elements: Match.arrayWith([{ Asdf: 3 }]),
     *     Payload: Match.serializedJson({ key: 'value' }),
     *   }),
     * }
     */
    getMatcher() {
        try {
            const final = JSON.parse(JSON.stringify(this.parsedObj), function (_k, v) {
                const nested = Object.keys(v)[0];
                switch (nested) {
                    case '$ArrayWith':
                        return helpers_internal_1.Match.arrayWith(v[nested]);
                    case '$ObjectLike':
                        return helpers_internal_1.Match.objectLike(v[nested]);
                    case '$StringLike':
                        return helpers_internal_1.Match.stringLikeRegexp(v[nested]);
                    case '$SerializedJson':
                        return helpers_internal_1.Match.serializedJson(v[nested]);
                    default:
                        return v;
                }
            });
            if (helpers_internal_1.Matcher.isMatcher(final.matcher)) {
                return final.matcher;
            }
            return helpers_internal_1.Match.exact(final.matcher);
        }
        catch {
            return helpers_internal_1.Match.exact(this.parsedObj.matcher);
        }
    }
}
function decodeCall(call) {
    if (!call) {
        return undefined;
    }
    try {
        const parsed = JSON.parse(call);
        return parsed;
    }
    catch (e) {
        return call;
    }
}
//# sourceMappingURL=data:application/json;base64,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