"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
/**
 * @stability stable
 */
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener.
 *
 * Routes to a fleet of of instances in a VPC.
 *
 * @stability stable
 */
class LoadBalancer extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer.
         *
         * @stability stable
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerProps(props);
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.any({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (_c = props.crossZone) !== null && _c !== void 0 ? _c : true,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        if (props.accessLoggingPolicy !== undefined) {
            this.elb.accessLoggingPolicy = props.accessLoggingPolicy;
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer.
     *
     * @returns A ListenerPort object that controls connections to the listener port
     * @stability stable
     */
    addListener(listener) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerListener(listener);
        if (listener.sslCertificateArn && listener.sslCertificateId) {
            throw new Error('"sslCertificateId" is deprecated, please use "sslCertificateArn" only.');
        }
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const sslCertificateArn = listener.sslCertificateArn || listener.sslCertificateId;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: sslCertificateArn,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    /**
     * @stability stable
     */
    addTarget(target) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_ILoadBalancerTarget(target);
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
LoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancing.LoadBalancer", version: "1.142.0" };
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *     const listener = elb.addListener(...);
 *
 *     listener.connections.allowDefaultPortFromAnyIPv4();
 *     // or
 *     instance.connections.allowToDefaultPort(listener);
 *
 * @stability stable
 */
class ListenerPort {
    /**
     * @stability stable
     */
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
_b = JSII_RTTI_SYMBOL_1;
ListenerPort[_b] = { fqn: "@aws-cdk/aws-elasticloadbalancing.ListenerPort", version: "1.142.0" };
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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