"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuildProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const smithy_generated_output_1 = require("./components/smithy-generated-output");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyBuildProject extends projen_1.Project {
    constructor(options) {
        super(options);
        // Store whether we've synthesized the project
        this.synthed = false;
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        // Add the gradle files which the user may modify to customise what's generated
        new projen_1.SampleFile(this, "build.gradle", {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "smithy", "build.gradle"),
        });
        new projen_1.SampleFile(this, "settings.gradle", {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "smithy", "settings.gradle"),
        });
        // Create the smithy build json file
        new smithy_build_1.SmithyBuild(this, {
            version: "2.0",
            ...options.smithyBuildOptions,
            projections: {
                ...options.smithyBuildOptions?.projections,
                openapi: {
                    plugins: {
                        openapi: {
                            service: options.fullyQualifiedServiceName,
                            ...options.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
        });
        // SmithyBuild component above always writes to smithy-build.json
        this.smithyBuildConfigPath = path.join(this.outdir, "smithy-build.json");
        this.smithyBuildOutputPath = path.join(this.outdir, options.buildOutputDir);
        new smithy_generated_output_1.SmithyGeneratedOutput(this, {
            modelPath: options.modelPath,
            gradleProjectPath: this.outdir,
            smithyBuildConfigPath: this.smithyBuildConfigPath,
            outputPath: this.smithyBuildOutputPath,
            gradleWrapperPath: options.gradleWrapperPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.SmithyBuildProject = SmithyBuildProject;
//# sourceMappingURL=data:application/json;base64,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