from django.apps import apps
from django.utils.text import slugify
from pathlib import Path
from string import Template
from datetime import datetime
from hyperx.logger.hx_logger import *

_logger = load_logger("htmx_backend_views")
_logger.info("htmx_backend_views initialized")

import os, sys, re, json, hmac, hashlib, csv
from datetime import datetime




TEMPLATE_VIEW = Template(
    '''
# Auto-generated by HyperX build CLI
# Timestamp: $timestamp

# ───────────────  $model_name  ───────────────
from django.shortcuts import get_object_or_404
from django.http import HttpResponse
from django.views.decorators.http import require_http_methods
from django.core.paginator import Paginator
from django.utils.html import escape
from django.middleware.csrf import get_token
import csv

from $app_label.models import $model_name


@require_http_methods(["GET"])
def ${endpoint}_list(request):
    page = int(request.GET.get("page", 1))
    per_page = int(request.GET.get("per_page", 10))
    items = $model_name.objects.order_by("-id")
    paginator = Paginator(items, per_page)
    page_obj = paginator.get_page(page)
    rows = []
    for obj in page_obj.object_list:
        row = "  <tr>" + "".join([
            f"<td>{{escape(str(getattr(obj, f.name)))}}</td>"
            for f in $model_name._meta.fields if not f.primary_key
        ]) + f"""
          <td class="text-center">
            <hx:button get="/$endpoint/edit/{{obj.id}}/" target="#crud-zone" label="✏️ Edit" />
            <hx:button delete="/$endpoint/delete/{{obj.id}}/" target="#crud-zone" confirm="Delete this item?" label="🗑️ Delete" />
          </td>
        </tr>"""
        rows.append(row)
    table = f"""
    <hx:panel id="crud-zone">
      <table class="table table-striped table-hover align-middle">
        <thead><tr>{{"".join(f"<th>{{f.verbose_name.title()}}</th>" for f in $model_name._meta.fields if not f.primary_key)}}<th>Actions</th></tr></thead>
        <tbody>{{''.join(rows)}}</tbody>
      </table>
      <hx:pagination source="$endpoint/list" target="#crud-zone" per-page="{{per_page}}" />
    </hx:panel>"""
    return HttpResponse(table)


@require_http_methods(["POST"])
def ${endpoint}_create(request):
    obj = $model_name.objects.create(**request.POST.dict())
    return HttpResponse(f"<hx:toast message='✅ $model_name created!' level='success' />" + ${endpoint}_tablecrud())


@require_http_methods(["GET", "POST"])
def ${endpoint}_edit(request, pk):
    obj = get_object_or_404($model_name, pk=pk)
    if request.method == "POST":
        for key, val in request.POST.items():
            setattr(obj, key, val)
        obj.save()
        return HttpResponse(f"<hx:toast message='✅ $model_name updated!' level='info' />" + ${endpoint}_tablecrud())
    fields = "".join([
        f"<hx:field label='{{f.verbose_name.title()}}' name='{{f.name}}' value='{{getattr(obj,f.name)}}' />"
        for f in $model_name._meta.fields if not f.primary_key
    ])
    csrf = get_token(request)
    return HttpResponse(f"<hx:form post='/$endpoint/edit/{{pk}}/' target='#crud-zone'><input type='hidden' name='csrfmiddlewaretoken' value='{csrf}' />{fields}<button type='submit' class='btn btn-success mt-2'>Save</button></hx:form>")


@require_http_methods(["DELETE"])
def ${endpoint}_delete(request, pk):
    obj = get_object_or_404($model_name, pk=pk)
    obj.delete()
    return HttpResponse(f"<hx:toast message='🗑️ $model_name deleted.' level='danger' />" + ${endpoint}_tablecrud())


def ${endpoint}_tablecrud():
    return f"<hx:tablecrud source='$endpoint' fields='{','.join(f.name for f in $model_name._meta.fields if not f.primary_key)}' actions='edit,delete,export' per-page='10' paginate='true' />"


@require_http_methods(["GET"])
def ${endpoint}_export(request):
    response = HttpResponse(content_type="text/csv")
    response["Content-Disposition"] = f'attachment; filename="$endpoint.csv"'
    fields = [f.name for f in $model_name._meta.fields if not f.primary_key]
    writer = csv.writer(response)
    writer.writerow(fields)
    for obj in $model_name.objects.all():
        writer.writerow([getattr(obj, f) for f in fields])
    return response
'''
)

def generate_views(app_label, output, silent=False):
    models_in_app = apps.get_app_config(app_label).get_models()
    blocks = []
    for model in models_in_app:
        endpoint = slugify(model.__name__).lower() + "s"
        code = TEMPLATE_VIEW.safe_substitute(
            app_label=app_label,
            model_name=model.__name__,
            endpoint=endpoint,
            timestamp=datetime.now().isoformat(),
        )
        blocks.append(code)
    Path(output).write_text("\n\n".join(blocks))
    if not silent:
        print(f"✅ Views written → {output}")
