"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/fr_fr/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue'],
            resourceArns: [this.secretArn],
            scope: this
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/fr_fr/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, Object.assign({ secret: this }, options));
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.secretName,
        });
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
    }
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, Object.assign({ secret: this }, options));
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType
        });
        this.encryptionKey = props.secret.encryptionKey;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
//# sourceMappingURL=data:application/json;base64,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