"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrAddStep = exports.ActionOnFailure = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The action to take when the cluster step fails.
 * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
 *
 * Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
 *
 * @default CONTINUE
 *
 */
var ActionOnFailure;
(function (ActionOnFailure) {
    /**
     * Terminate the Cluster on Step Failure
     */
    ActionOnFailure["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    /**
     * Cancel Step execution and enter WAITING state
     */
    ActionOnFailure["CANCEL_AND_WAIT"] = "CANCEL_AND_WAIT";
    /**
     * Continue to the next Step
     */
    ActionOnFailure["CONTINUE"] = "CONTINUE";
})(ActionOnFailure = exports.ActionOnFailure || (exports.ActionOnFailure = {}));
/**
 * A Step Functions Task to add a Step to an EMR Cluster
 *
 * The StepConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the StepId
 *
 */
class EmrAddStep extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EmrAddStepProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EmrAddStep);
            }
            throw error;
        }
        this.actionOnFailure = props.actionOnFailure ?? ActionOnFailure.CONTINUE;
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'addStep', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
                Step: {
                    Name: this.props.name,
                    ActionOnFailure: this.actionOnFailure.valueOf(),
                    HadoopJarStep: {
                        Jar: this.props.jar,
                        MainClass: this.props.mainClass,
                        Args: this.props.args,
                        Properties: (this.props.properties === undefined) ?
                            undefined :
                            Object.entries(this.props.properties).map(kv => ({
                                Key: kv[0],
                                Value: kv[1],
                            })),
                    },
                },
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call AddStep.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:AddJobFlowSteps',
                    'elasticmapreduce:DescribeStep',
                    'elasticmapreduce:CancelSteps',
                ],
                resources: [
                    stack.formatArn({
                        service: 'elasticmapreduce',
                        resource: 'cluster',
                        resourceName: '*',
                    }),
                ],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRAddJobFlowStepsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.EmrAddStep = EmrAddStep;
_a = JSII_RTTI_SYMBOL_1;
EmrAddStep[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EmrAddStep", version: "1.160.0" };
EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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