"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.sonarqubeScanner = exports.createSonarqubeProject = exports.generateSonarqubeReports = void 0;
const cube = (path, action = "GET", args) => `curl -X ${action} -u $SONARQUBE_TOKEN: "$SONARQUBE_ENDPOINT${path}" ${args || ""}`;
const setupSonarqubeQualityGates = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_GATE=\`${cube(`/api/qualitygates/search?gateName=${defaultProfileOrGateName}`)}\``,
    `export SPECIFIC_GATE=\`${cube(`/api/qualitygates/search?gateName=${specificProfileOrGateName}`)}\``,
    `if [[ "$(echo $SPECIFIC_GATE | jq .errors)" == "null" && "$(echo $SPECIFIC_GATE | jq '.results | length')" -gt 0 ]]; then export GATE_NAME=${specificProfileOrGateName}; else export GATE_NAME=${defaultProfileOrGateName}; fi`,
    `${cube("/api/qualitygates/select?projectKey=$PROJECT_NAME&gateName=$GATE_NAME", "POST")}`,
];
const setupSonarqubeQualityProfiles = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${defaultProfileOrGateName}`)} | jq .profiles\``,
    `export SPECIFIC_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${specificProfileOrGateName}`)} | jq .profiles\``,
    `export MERGED_PROFILES=\`jq --argjson arr1 "$DEFAULT_PROFILE" --argjson arr2 "$SPECIFIC_PROFILE" -n '$arr1 + $arr2 | group_by(.language) | map(.[-1])'\``,
    `echo $MERGED_PROFILES | jq -c '.[]' | while read i; do ${cube("/api/qualityprofiles/add_project?project=$PROJECT_NAME&language=`echo $i | jq -r .language`&qualityProfile=`echo $i | jq -r .name`", "POST")}; done`,
];
const setupSonarqubePermissions = (authorizedGroup) => !authorizedGroup
    ? []
    : [
        "admin",
        "codeviewer",
        "issueadmin",
        "securityhotspotadmin",
        "scan",
        "user",
    ].map((p) => `${cube(`/api/permissions/add_group?projectKey=$PROJECT_NAME&groupName=${authorizedGroup}&permission=${p}`, "POST")}`);
const setupSonarqubeProject = ({ sonarqubeTags, sonarqubeAuthorizedGroup, sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName, }) => {
    return [
        ...setupSonarqubePermissions(sonarqubeAuthorizedGroup),
        `${cube("/api/project_branches/rename?project=$PROJECT_NAME&name=mainline", "POST")}`,
        `${cube(`/api/project_tags/set?project=$PROJECT_NAME&tags=${[
            sonarqubeAuthorizedGroup,
            ...(sonarqubeTags || []),
        ].join(",")}`, "POST")}`,
        ...setupSonarqubeQualityProfiles(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
        ...setupSonarqubeQualityGates(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
    ].join(";");
};
exports.generateSonarqubeReports = () => [
    cube("/api/bitegarden/report/pdf_issues_breakdown?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-issues-report.pdf"),
    cube("/api/bitegarden/report/pdf?resource=$PROJECT_NAME&branch=mainline", "GET", "--output reports/prototype-executive-report.pdf"),
    cube("/api/security_reports/download?project=$PROJECT_NAME", "GET", "--output reports/prototype-security-report.pdf"),
];
exports.createSonarqubeProject = (props) => [
    `CREATE_PROJECT_OUTPUT=\`${cube("/api/projects/create?name=$PROJECT_NAME&project=$PROJECT_NAME&visibility=private", "POST")}\``,
    `if [[ "$(echo $CREATE_PROJECT_OUTPUT | jq .errors)" == "null" ]]; then ${setupSonarqubeProject(props)}; fi;`,
];
exports.sonarqubeScanner = (excludeGlobsForScan) => [
    "npx sonarqube-scanner -Dsonar.login=$SONARQUBE_TOKEN",
    "-Dsonar.projectKey=$PROJECT_NAME",
    "-Dsonar.projectName=$PROJECT_NAME",
    "-Dsonar.projectVersion=`echo $RESOLVED_SOURCE_VERSION | cut -c1-7`",
    "-Dsonar.branch.name=mainline",
    "-Dsonar.host.url=$SONARQUBE_ENDPOINT",
    "-Dsonar.cfn.nag.reportFiles=reports/cfn-nag-report.json",
    "-Dsonar.dependencyCheck.htmlReportPath=reports/dependency-check-report.html",
    "-Dsonar.javascript.lcov.reportPaths=**/coverage/lcov.info",
    "-Dsonar.clover.reportPath=**/coverage/clover.xml",
    `-Dsonar.exclusions="**/reports/**,**/coverage/**${excludeGlobsForScan ? `,${excludeGlobsForScan.join(",")}` : ""}"`,
    "-Dsonar.sources=.",
].join(" ");
//# sourceMappingURL=data:application/json;base64,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