import asyncio
import typing
from uuid import uuid4

import aiohttp

from ratelimitcli.client.exceptions import APIRateLimitException
from ratelimitcli.limits.read_write import ThriftReadWrite, Base64ReadWrite
from ratelimitcli.limits import cli_thrift as T
from ratelimitcli.conf import settings


class RatelimitClient:
    """Http client for the Ratelimit API.

    Record requests against pre-existing rate limits directly.
    """

    def __init__(
        self, host=None, port=None, path=None, client_id=None, api_key=None, loop=None
    ):
        if host is None:
            host = settings["RL_HOST"]
        if port is None:
            port = settings["RL_PORT"]
        if path is None:
            base_path = settings["RL_BASE_PATH"]
        self._base_url = f"{host}:{port}{base_path}"

        if client_id is None:
            client_id = settings["RL_CLIENT_ID"]
        if api_key is None:
            api_key = settings["RL_API_KEY"]

        self._headers = {
            "X-CLIENT-ID": client_id,
            "Authorization": api_key,
            "Content-Type": "application/text",
        }
        self._client_id = client_id
        self.b64rw = Base64ReadWrite()

        self._MAX_CONNECTIONS = int(settings["RL_MAX_CONNECTIONS"])
        self._MAX_CONNECTIONS_PER_HOST = int(settings["RL_MAX_CONNECTIONS_PER_HOST"])
        self._session = None
        self._loop = loop

    @property
    def loop(self):
        if self._loop:
            return self._loop
        try:
            self._loop = asyncio.get_running_loop()
        except:
            self._loop = asyncio.new_event_loop()
        return self._loop

    def __del__(self):
        """Close underlying session."""
        if self._session:
            self.loop.run_until_complete(self._session.close())

    async def _create_session(self):
        """Create a ClientSession in an async method."""
        self._session = aiohttp.ClientSession(
            connector=aiohttp.TCPConnector(
                limit=self._MAX_CONNECTIONS,
                limit_per_host=self._MAX_CONNECTIONS,
            ),
            headers=self._headers,
        )

    async def _send(
        self, url: str, request_bytes: bytes
    ) -> typing.Optional[typing.Union[bytes, typing.NoReturn]]:
        """Send the request to the API endpoint and return the response."""
        if not self._session:
            await self._create_session()

        data = self.b64rw.to_base64(request_bytes)
        async with self._session.post(url, data=data) as resp:
            if resp.status == 429:
                raise APIRateLimitException(resp)
            response = await resp.read()
            return response if response else None

    def sync_record_request(
        self, limit_id: str
    ) -> typing.Optional[typing.Union[bytes, typing.NoReturn]]:
        """Synchronously send request to the API."""
        return self.loop.run_until_complete(self.record_request(limit_id))

    async def record_request(
        self, limit_id: str
    ) -> typing.Optional[typing.Union[bytes, typing.NoReturn]]:
        url = f"{self._base_url}{settings['RL_BUCKET_PATH']}"
        rw = _SERDE["BUCKET"]
        thrift = T.GetRemainingCapacityRequest(
            meta=T.Metadata(
                client_id=self._client_id,
                purpose="upsert_limit",
                tracking_id=str(uuid4()),
            ),
            limit_id=limit_id,
        )
        return await self._send(url, rw.to_bytes(thrift))


_SERDE = {
    "BUCKET": ThriftReadWrite(T.GetRemainingCapacityRequest),
    "CONFIG": ThriftReadWrite(T.SetLimitRequest),
}
