# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ratelimitcli',
 'ratelimitcli.billing',
 'ratelimitcli.client',
 'ratelimitcli.conf',
 'ratelimitcli.config',
 'ratelimitcli.limits']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.1,<4.0.0',
 'python-jose>=3.3.0,<4.0.0',
 'pytomlpp>=1.0.11,<2.0.0',
 'requests>=2.28.1,<3.0.0',
 'thriftpy2>=0.4.14,<0.5.0',
 'typer[all]>=0.5.0,<0.6.0']

entry_points = \
{'console_scripts': ['ratelimitcli = ratelimitcli.main:app']}

setup_kwargs = {
    'name': 'ratelimitcli',
    'version': '0.2.1',
    'description': 'A CLI to manage rate limits.',
    'long_description': '# Adding rate limits to your API\n\nThis is a python CLI that communicates with a cloud based HTTP **Ratelimit Service** to configure rate limits based on the token bucket algorithm.\n\nRedis is used as the backing storage for the rate limit configurations, and using Lua scripts, the **Ratelimit Service** can make a determination to allow or deny a request within milliseconds.\n\nThis python package also comes with:\n\n1. An HTTP client that can be used to manually test your limits.\n2. A python decorator for your HTTP endpoint functions.\n\n### Where to start\n\nThe quickstart below will guide you through configuring the CLI on your system, authenticating to the **Ratelimit Service**, creating a rate limit, and testing it.\n\nFor more detailed documentation check out [https://docs.ratelimit.xyz](https://docs.ratelimit.xyz).\n\n## Configure the CLI\n\n1. Install the RateLimit CLI.\n\n```bash\n$ pip install ratelimitcli\n```\n\n2. Configure the RateLimit CLI and follow the interactive prompts. You won\'t have an API key yet so you can just press `[ENTER]` when asked for a value.\n\n```bash\n$ ratelimitcli configure\nClient ID  [None]:\nAPI key    [None]:\nFirst name [None]: Jasmin\nLast name  [None]: Smith\nEmail name [None]: jasmin.smith@gmail.com\nUpdated config file (~/.ratelimit/config).\n```\n\n3. Request an API key and a Client ID. You\'ll be asked to enter credit card information.\n\n```\n$ ratelimitcli billing configure\nEnter a credit card number: 1234567890123456\nEnter the expiration date for that credit card: 10/12\nEnter the CCV for that credit card: 333\nNew values have been written to the config file.\n```\n\n> **Disclaimer**\n>\n> I have not yet implemented payments and as such, you won\'t be charged for anything. That said, this service is currently in a closed alpha. There is only a single credit card that will allow you to get an API key and you\'d have to get that from me.\n>\n> If this service or the concept is interesting enough for you and you don\'t mind sending an email, feel free to contact me and we can discuss your usecase.\n>\n> Oh and the fake card I show above won\'t work - just FYI.\n\n4. Check to see that the config file has been written to `$HOME/.ratelimit/config`\n\n```bash\n$ cat ~/.ratelimit/config\n```\n\nThe CLI is now configured and ready to use.\n\n---\n\n## Creating your first limit\n\n1. Create your first limit by supplying integers for the **burst limit** and the **rate limit**.\n\n```bash\n$ ratelimitcli limits upsert \\\n  --throttling-burst-limit 2 \\\n  --throttling-rate-limit 0\n```\n\n> **Burst Limit**\n>\n> The maximum available instantaneous capacity.\\\n> https://en.wikipedia.org/wiki/Token_bucket\n\n> **Rate Limit**\n>\n> The rate at which used capacity is restored.\\\n> https://en.wikipedia.org/wiki/Token_bucket\n\nYou\'ll know everything is set up when you get an API response with a limit ID that identifies the rate limit you just created.\n\nReturns\n\n```\nOk: API response ({"limit_id": "a9f9f31b-2c0b-321b-b398-f9d36kd30820"}).\n```\n\n2. We can store that limit ID into an environment variable for ease of access.\n\n```bash\nexport LIMIT_ID=a9f9f31b-2c0b-321b-b398-f9d36kd30820\n```\n\n3. Test your first rate limit.\n\n```bash\n# ok - capacity 2, using the env var\n$ ratelimitcli limits record $LIMIT_ID\n\n# ok - capacity 1, using ID directly\n$ ratelimitcli limits record \\\n  a9f9f31b-2c0b-321b-b398-f9d36kd30820\n\n# ERROR - capacity 0, we\'ve exhausted the capacity\n$ ratelimitcli limits record $LIMIT_ID\n```\n\n4. Test rate limits in a python interpreter shell. Note that in production, you most likely will not be using the `RatelimitClient` directly, nor will you be using synchronous calls to the **Ratelimit Service**.\n\n```python\n$ python\n\n# Get the limit ID from the env var\n>>> import os\n>>> limit_id = os.environ[\'LIMIT_ID\']\n>>> limit_id\n\'a9f9f31b-2c0b-321b-b398-f9d36kd30820\'\n\n# Import the client\n>>> from ratelimitcli.client.client import RatelimitClient\n\n# All args are optional.\n# Defaults are pulled from env and config file.\n>>> client = RatelimitClient()\n\n# Synchronous calls to the service.\n>>> client.sync_record_request(limit_id)\n>>> client.sync_record_request(\n... "a9f9f31b-2c0b-321b-b398-f9d36kd30820"\n... )\n```\n\n8. Use your rate limit in your code.\n\n```python\nfrom fastapi import FastAPI\n\nfrom ratelimitcli.client.exceptions import APIRateLimitException\nfrom ratelimitcli.client.decorator import RatelimitDecorator\n\napp = FastAPI()\n\ndef on_error_callback(\n  msg: str,\n  exception: APIRateLimitException\n):\n    """Callback function.\n\n    Args:\n        1. All the `*args` and `**kwargs` that were passed\n           to the original function. In this case `msg: str`.\n\n        2. The exception that was raised by RatelimitDecorator\n\n    Returns:\n        Some value that takes the place of the wrapped function\'s\n        return value.\n    """\n    return "Goodbye, World!" + msg\n\n\n@app.get("/")\n@RatelimitDecorator(\n    # The limit ID to check capacity against.\n    id="a9f9f31b-2c0b-321b-b398-f9d36kd30820",\n    # The function to call if capacity has been exhausted.\n    callback=on_error_callback,\n)\nasync def hello(msg: str):\n    return "Hello, World!" + msg\n```\n',
    'author': 'Anwar',
    'author_email': 'anwar@anwarthink.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://docs.ratelimit.xyz',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
