import requests
import typer

from ratelimitcli.limits.read_write import ThriftReadWrite, Base64ReadWrite
from ratelimitcli.limits import cli_thrift as T

from ratelimitcli.config.middlewear import check_for_config_file

app = typer.Typer(
    short_help="Create and update rate limits.",
    help="""
NAME:

    limits

DESCRIPTION

    Create and update rate limits as well as record requests against rate
    limits you have created.

SYNOPSIS

    ratelimitcli limits
""",
)

_MAX_INT = 1 << 32


def throttling_burst_limit_callback(value: int) -> int:
    error = "Invalid burst limit value. Must be between [0, 2^32]."
    if value < 0 or value > _MAX_INT:
        raise typer.BadParameter(error)
    return value


def throttling_rate_limit_callback(value: int) -> int:
    error = "Invalid rate limit value. Must be between [0, 2^32]."
    if value < 0 or value > _MAX_INT:
        raise typer.BadParameter(error)
    return value


@app.command(
    short_help="Upsert a limit configuration.",
    help="""
NAME:

    upsert

DESCRIPTION

    Upsert - create or update - a rate limit.

    A rate limit is composed of two parts: a `burst limit` and a `rate
    limit`.

    The burst limit is how much capacity you configure for your limit that
    can be immediately used by incoming requests. The rate limit is how
    quickly the used capacity will be restored. Limits that have exhausted
    their capacity will reject requests.

LIMIT CONFIG

    o Burst Limit - The maximum available instantaneous capacity.

    o Rate Limit - The rate at which used capacity is restored.

SYNOPSIS

    ratelimitcli limits upsert
""",
)
def upsert(
    throttling_burst_limit: int = typer.Option(
        ..., callback=throttling_burst_limit_callback
    ),
    throttling_rate_limit: int = typer.Option(
        ..., callback=throttling_rate_limit_callback
    ),
    limit_id: str = None,
    purpose: str = "cmdline",
    tracking_id: str = "cmdline",
):
    existing_config = check_for_config_file()

    thrift = T.SetLimitRequest(
        meta=T.Metadata(
            client_id=existing_config.client_id,
            purpose=purpose,
            tracking_id=tracking_id,
        ),
        limit_id=limit_id,
        throttling_burst_limit=throttling_burst_limit,
        throttling_rate_limit=throttling_rate_limit,
    )
    rw = ThriftReadWrite(T.SetLimitRequest)
    brw = Base64ReadWrite()
    data = brw.to_base64(rw.to_bytes(thrift))

    response = requests.post(
        "https://api.ratelimit.xyz/api/config",
        headers={
            "X-CLIENT-ID": existing_config.client_id,
            "Authorization": existing_config.api_key,
            "Content-Type": "application/text",
        },
        data=data,
    )
    if response.status_code != 200:
        typer.echo(f"Bad API response ({response.text}).", err=True)
        raise typer.Exit(code=1)
    else:
        typer.echo(f"Ok: API response ({response.text}).")


@app.command(
    short_help="Record a request against a limit.",
    help="""
NAME:

    record

DESCRIPTION

    Record a request against a limit. Limits that have exhausted their
    capacity will reject requests.

SYNOPSIS

    ratelimitcli limits record
""",
)
def record(limit_id: str, purpose: str = "cmdline", tracking_id: str = "cmdline"):
    existing_config = check_for_config_file()

    thrift = T.GetRemainingCapacityRequest(
        meta=T.Metadata(
            client_id=existing_config.client_id,
            purpose=purpose,
            tracking_id=tracking_id,
        ),
        limit_id=limit_id,
    )
    rw = ThriftReadWrite(T.GetRemainingCapacityRequest)
    brw = Base64ReadWrite()
    data = brw.to_base64(rw.to_bytes(thrift))

    response = requests.post(
        "https://api.ratelimit.xyz/api/bucket",
        headers={
            "X-CLIENT-ID": existing_config.client_id,
            "Authorization": existing_config.api_key,
            "Content-Type": "application/text",
        },
        data=data,
    )
    if response.status_code == 200:
        typer.echo(f"Ok: API response ({response.text}).")
    elif response.status_code == 429:
        typer.echo(f"Too Many Requests: API response ({response.text}).", err=True)
        raise typer.Exit(code=1)
    else:
        typer.echo(f"Error: API response ({response.text}).", err=True)
        raise typer.Exit(code=1)
