import bisect
import json
import os
from dataclasses import dataclass, fields
from typing import Any, Dict, List, Optional, Type, Union

import requests
from typing_extensions import Self

from ids_validator.models.validator_parameters import IdsArtifact
from ids_validator.utils import parse_prefixed_version


def _strtobool(value: Optional[Union[str, bool]], field_name: str) -> Optional[bool]:
    """Convert a set of strings to True or False.

    Passes through `None` or values which are already `bool`.

    Raises a ValueError including `field_name` if `value` is anything else.
    """
    # This is a modified version of distutils.util.strtobool which is deprecated
    if value is None:
        return None
    if isinstance(value, bool):
        return value

    # These values are also specified in the README.md (keep them in sync if changed)
    # These are the same values as what ts-sdk accepts as True
    true_values = ("true", "True", "1")
    false_values = ("false", "False", "0")
    if value in true_values:
        return True
    elif value in false_values:
        return False

    raise ValueError(
        f"Unrecognized boolean value for {field_name}: `{value}` ({type(value)}). "
        f"True must be one of: {true_values}. "
        f"False must be one of: {false_values}. (Case insensitive.)"
    )


def _get_env_value(key: str) -> Optional[str]:
    """
    To match the environment variable naming of ts-sdk, prepend "TS_" and make the key
    uppercase, e.g. "auth_token" is found in the "TS_AUTH_TOKEN" env var.
    """
    env_key = "TS_" + key.upper()
    return os.environ.get(env_key, None)


class ConfigKeys:
    API_URL = "api_url"
    ORG = "org"
    AUTH_TOKEN = "auth_token"
    IGNORE_SSL = "ignore_ssl"


@dataclass
class APIConfig:
    """
    TDP API configuration values which can be loaded from a file or environment variables.
    This matches the functionality of ts-sdk.

    `source` is a string shown to users in error messages to help explain where API
    config was loaded from, e.g. whether it's being loaded from environment variables
    or JSON input.
    """

    api_url: str
    org: str
    auth_token: str
    # Ignore SSL is part of the config generated by TDP when creating a service user
    ignore_ssl: bool = False
    # Text to show to users for where API config was loaded from (env vars/file/other)
    source: str = "default"

    @property
    def request_headers(self: Self) -> Dict[str, str]:
        return {
            "x-org-slug": self.org,
            "ts-auth-token": self.auth_token,
        }

    @property
    def request_configuration(self: Self) -> Dict[str, Any]:
        """
        Configure the `request.get` kwargs `headers` and `verify` from the loaded API
        configuration.
        """
        return {"headers": self.request_headers, "verify": not self.ignore_ssl}

    @classmethod
    def from_json_or_env(
        cls: Type[Self],
        json_config: Optional[dict] = None,
        json_config_source: str = "JSON config",
    ) -> Self:
        """Read API config from `json_config` if defined, otherwise from env vars."""
        if json_config is not None:
            # Load auth entirely from the file
            # ts-sdk allows a string value for the bool `ignore_ssl` which needs to be
            # converted to a bool.
            if ConfigKeys.IGNORE_SSL in json_config:
                json_config[ConfigKeys.IGNORE_SSL] = _strtobool(
                    json_config[ConfigKeys.IGNORE_SSL],
                    field_name=ConfigKeys.IGNORE_SSL,
                )
            return cls.validate_input(json_config, source=json_config_source)

        # Load auth from environment vars
        config_from_env = {
            ConfigKeys.API_URL: _get_env_value(ConfigKeys.API_URL),
            ConfigKeys.ORG: _get_env_value(ConfigKeys.ORG),
            ConfigKeys.AUTH_TOKEN: _get_env_value(ConfigKeys.AUTH_TOKEN),
            ConfigKeys.IGNORE_SSL: _strtobool(
                _get_env_value(ConfigKeys.IGNORE_SSL), field_name="TS_IGNORE_SSL"
            ),
        }
        return cls.validate_input(config_from_env, source="environment variables")

    @classmethod
    def validate_input(
        cls: Type[Self],
        api_config: dict,
        source: str,
    ) -> Self:
        """Validate all keys are present and values are not `None` for API config input."""
        # Only keep expected keys, discard `None` values
        config = {
            key: value
            for key, value in api_config.items()
            if (
                key in (field.name for field in fields(APIConfig)) and value is not None
            )
        }
        config["source"] = source

        missing_keys = {
            ConfigKeys.API_URL,
            ConfigKeys.AUTH_TOKEN,
            ConfigKeys.ORG,
        } - config.keys()

        if missing_keys:
            raise ValueError(
                f"API configuration values are missing or null for: {missing_keys}."
            )

        return cls(**config)

    def __str__(self):
        # Reveal same amount of auth_token as ts-sdk
        return f'APIConfig(api_url="{self.api_url}", org="{self.org}", auth_token="{self.auth_token[:7]}...", ignore_ssl={self.ignore_ssl}, source="{self.source}")'


class APIError(Exception):
    """Exception encountered during a TDP API call."""

    def __init__(self, config: APIConfig):
        self.message = (
            "Encountered error during a TDP API request. "
            f"Check API configuration and auth: {config}"
        )
        super().__init__(self.message)


class IdsArtifactFile:
    """IDS Artifact file slugs as defined by the TDP Artifact Management API."""

    README = "readme"
    ATHENA = "athena"
    BUILD = "build"
    ELASTICSEARCH = "elasticsearch"
    ERD = "erd"
    EXPECTED = "expected"
    MANIFEST = "manifest"
    SCHEMA = "schema"


def get_ids_versions(config: APIConfig, namespace: str, slug: str) -> List[str]:
    """Get all versions of the artifact matching a namespace and slug."""
    url = f"{config.api_url}/artifacts/ids/{namespace}/{slug}"
    response = requests.get(
        url,
        **config.request_configuration,
    )
    try:
        response.raise_for_status()
    except requests.HTTPError as error:
        raise APIError(config) from error

    artifacts = json.loads(response.text)
    if not all("version" in artifact for artifact in artifacts):
        # This will not happen if this endpoint follows its specification
        raise ValueError(
            "Unexpected API response structure, expected a list of objects containing "
            f"a 'version' key from: {url}"
        )

    return [artifact["version"] for artifact in artifacts]


def get_preceding_version(target_version: str, versions: List[str]) -> Optional[str]:
    """Find the version from `versions` which immediately precedes `target_version`.

    e.g. target_version is the local IDS version being developed, and `versions` comes
    from the list of IDS versions with the same namespace and slug already in TDP.
    Then this function finds the version which should be compared against for any
    version validation like breaking change validation.
    """
    # Before Python 3.10, bisect does not have `key` which would simplify this.
    # Instead, we keep track of a sortable `semver.Version` AND the original string
    parsed_versions = [(parse_prefixed_version(v), v) for v in versions]
    sorted_versions = sorted(parsed_versions)
    parsed_target = (parse_prefixed_version(target_version), target_version)
    insert_index = bisect.bisect(sorted_versions, parsed_target)
    if insert_index == 0:
        # The target is either the first version, or precedes the first version
        return None
    _, preceding_version = sorted_versions[insert_index - 1]
    return preceding_version


def get_ids_artifact_file(
    config: APIConfig, namespace: str, slug: str, version: str, file_slug: str
) -> str:
    """
    Retrieve a specific IDS artifact file (README.md, schema.json, etc.) as a string.
    """
    response = requests.get(
        f"{config.api_url}/artifacts/ids/{namespace}/{slug}/{version}/files/{file_slug}",
        **config.request_configuration,
    )
    try:
        response.raise_for_status()
    except requests.HTTPError as error:
        raise APIError(config) from error
    return response.text


def get_ids_artifact(
    config: APIConfig, namespace: str, slug: str, version: str
) -> IdsArtifact:
    """
    Read all relevant IDS artifact files for the given namespace/slug/version into the
    format used by ts-ids-validator.
    """
    # Note `expected.json` and other files are not currently needed for breaking
    # change validation so they are not downloaded
    files = {}
    for file_slug in (
        IdsArtifactFile.SCHEMA,
        IdsArtifactFile.ATHENA,
        IdsArtifactFile.ELASTICSEARCH,
    ):
        # The API specifies that the file content is under "content" property
        files[file_slug] = json.loads(
            get_ids_artifact_file(
                config=config,
                namespace=namespace,
                slug=slug,
                version=version,
                file_slug=file_slug,
            )
        )["content"]
    return IdsArtifact(**files)


def get_preceding_ids_artifact(
    config: APIConfig, namespace: str, slug: str, target_version: str
) -> Optional[IdsArtifact]:
    remote_versions = get_ids_versions(config=config, namespace=namespace, slug=slug)
    preceding_version = get_preceding_version(
        target_version=target_version, versions=remote_versions
    )
    if preceding_version is None:
        # There is no remote version preceding the target version
        return None
    return get_ids_artifact(
        config=config, namespace=namespace, slug=slug, version=preceding_version
    )
