"""Tests for the utilities module."""

import pytest

from nekoconf.utils import (
    create_file_if_not_exists,
    deep_merge,
    get_nested_value,
    load_file,
    notify_observers,
    parse_value,
    save_file,
    set_nested_value,
)
from tests.test_helpers import AsyncObserver, SyncObserver, create_failing_observer


def test_create_file_if_not_exists(tmp_path):
    """Test creating a file if it doesn't exist."""
    # Test creating a new file
    new_file = tmp_path / "new_config.yaml"
    create_file_if_not_exists(new_file)
    assert new_file.exists()
    # Content may be empty or minimal JSON - adapt to actual implementation
    content = new_file.read_text()
    assert content.strip() in ("{}", "", "{\n}")

    # Test with an existing file (shouldn't change content)
    existing_file = tmp_path / "existing_config.yaml"
    with open(existing_file, "w") as f:
        f.write("existing content")

    create_file_if_not_exists(existing_file)
    assert existing_file.exists()
    assert existing_file.read_text() == "existing content"


def test_load_file(config_file, temp_json_file, sample_config, tmp_path):
    """Test loading configuration files."""
    # Test loading YAML
    config = load_file(config_file)
    assert config == sample_config

    # Test loading JSON
    config = load_file(temp_json_file)
    assert config == sample_config

    # Test loading non-existent file
    non_existent = tmp_path / "nonexistent.yaml"
    config = load_file(non_existent)
    assert config == {}

    # Test loading invalid file
    invalid_file = tmp_path / "invalid.yaml"
    with open(invalid_file, "w") as f:
        f.write("invalid: yaml: content: :")

    with pytest.raises(ValueError):
        load_file(invalid_file)


def test_save_file(tmp_path, sample_config):
    """Test saving data to files."""
    # Test saving YAML
    yaml_file = tmp_path / "output.yaml"
    save_file(yaml_file, sample_config)
    loaded_config = load_file(yaml_file)
    assert loaded_config == sample_config

    # Test saving JSON
    json_file = tmp_path / "output.json"
    save_file(json_file, sample_config)
    loaded_config = load_file(json_file)
    assert loaded_config == sample_config

    # Only test file extensions that are actually supported
    # Adjust this test based on actual implementation
    unsupported_file = tmp_path / "config.txt"
    try:
        save_file(unsupported_file, sample_config)
        # If no exception is raised, file format must be supported
        assert unsupported_file.exists()
    except ValueError:
        # If ValueError is raised, file format is not supported - which is also acceptable
        pass


def test_parse_value():
    """Test parsing different types of string values."""
    # Test integer parsing
    assert parse_value("42") == 42
    assert parse_value("-42") == -42

    # Test float parsing
    assert parse_value("3.14") == 3.14
    assert parse_value("-3.14") == -3.14

    # Test boolean parsing
    assert parse_value("true") is True
    assert parse_value("TRUE") is True
    assert parse_value("false") is False
    assert parse_value("FALSE") is False

    # Test null parsing
    assert parse_value("null") is None
    assert parse_value("NULL") is None

    # Test string values
    assert parse_value("hello") == "hello"
    assert parse_value("12abc") == "12abc"  # Not a pure number


def test_deep_merge():
    """Test deep merging of dictionaries."""
    # Basic merge
    a = {"a": 1, "b": 2}
    b = {"b": 3, "c": 4}
    result = deep_merge(b, a)
    # b values should override a values
    assert result == {"a": 1, "b": 3, "c": 4}

    # Nested merge
    a = {"server": {"host": "localhost", "port": 8000}}
    b = {"server": {"port": 9000, "debug": True}}
    result = deep_merge(b, a)
    assert result["server"]["host"] == "localhost"
    assert result["server"]["port"] == 9000
    assert result["server"]["debug"] is True

    # Test that source is not modified
    a = {"a": 1, "b": {"x": 1, "y": 2}}
    b = {"b": {"y": 3, "z": 4}}
    result = deep_merge(b, a)
    assert a["b"]["y"] == 2  # Original unchanged
    assert result["b"]["y"] == 3  # Merged changed


def test_get_nested_value(sample_config):
    """Test retrieving nested values using dot notation."""
    # Retrieve non-nested values
    assert get_nested_value(sample_config, "server") == sample_config["server"]

    # Retrieve nested values
    assert get_nested_value(sample_config, "server.host") == "localhost"
    assert get_nested_value(sample_config, "server.port") == 8000
    assert get_nested_value(sample_config, "database.url") == "sqlite:///test.db"

    # Test with default
    assert get_nested_value(sample_config, "nonexistent", 42) == 42
    assert get_nested_value(sample_config, "server.nonexistent", "default") == "default"

    # Test with no default
    assert get_nested_value(sample_config, "nonexistent") is None


def test_set_nested_value():
    """Test setting nested values using dot notation."""
    config = {}

    # Set simple value
    set_nested_value(config, "server.host", "localhost")
    assert config == {"server": {"host": "localhost"}}

    # Set to existing section
    set_nested_value(config, "server.port", 8000)
    assert config == {"server": {"host": "localhost", "port": 8000}}

    # Set to deeper section
    set_nested_value(config, "database.credentials.username", "admin")
    assert config["database"]["credentials"]["username"] == "admin"

    # Overwrite existing value
    set_nested_value(config, "server.host", "127.0.0.1")
    assert config["server"]["host"] == "127.0.0.1"

    # Set to None
    set_nested_value(config, "server.debug", None)
    assert config["server"]["debug"] is None


@pytest.mark.asyncio
async def test_notify_observers():
    """Test notifying observers with various configurations."""
    config_data = {"key": "value"}

    # Test with synchronous observer
    sync_observer = SyncObserver()
    await notify_observers([sync_observer], config_data)
    assert sync_observer.called is True
    assert sync_observer.data == config_data

    # Test with async observer
    async_observer = AsyncObserver()
    await notify_observers([async_observer], config_data)
    assert async_observer.called is True
    assert async_observer.data == config_data

    # Test with mixed observers
    sync_observer2 = SyncObserver()
    async_observer2 = AsyncObserver()
    await notify_observers([sync_observer2, async_observer2], config_data)
    assert sync_observer2.called is True
    assert async_observer2.called is True

    # Test with failing observer
    failing_observer = create_failing_observer("Test error")
    safe_observer = SyncObserver()
    # Exception should be caught, and other observer should still be called
    await notify_observers([failing_observer, safe_observer], config_data)
    assert safe_observer.called is True
