"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cdk8s = require("cdk8s");
const minimatch_1 = require("minimatch");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 *
 * @stability stable
 */
class ConfigMap extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id);
        this._binaryData = {};
        this._data = {};
        this.apiObject = new k8s.KubeConfigMap(this, 'Resource', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
        });
        for (const [k, v] of Object.entries((_b = props.data) !== null && _b !== void 0 ? _b : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_c = props.binaryData) !== null && _c !== void 0 ? _c : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     *
     * @param name The name of the config map to import.
     * @stability stable
     */
    static fromConfigMapName(name) {
        return { name };
    }
    /**
     * Adds a data entry to the config map.
     *
     * @param key The key.
     * @param value The value.
     * @stability stable
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     *
     * @stability stable
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map.
     *
     * BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     *
     * @param key The key.
     * @param value The value.
     * @stability stable
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     *
     * @stability stable
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     *
     * @param localFile The path to the local file.
     * @param key The ConfigMap key (default to the file name).
     * @stability stable
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = fs.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     *
     * @param localDir A path to a local directory.
     * @param options Options.
     * @stability stable
     */
    addDirectory(localDir, options = {}) {
        var _b, _c;
        const exclude = (_b = options.exclude) !== null && _b !== void 0 ? _b : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                const mm = new minimatch_1.Minimatch(pattern);
                if (mm.match(file)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_c = options.keyPrefix) !== null && _c !== void 0 ? _c : '';
        for (const file of fs.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (fs.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
_a = JSII_RTTI_SYMBOL_1;
ConfigMap[_a] = { fqn: "cdk8s-plus-22.ConfigMap", version: "1.0.0-beta.60" };
//# sourceMappingURL=data:application/json;base64,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