"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.LabelSelectorOptions = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? true;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-22.AbstractPod", version: "2.0.0-rc.15" };
/**
 * Options for `LabelSelector.of`.
 */
class LabelSelectorOptions {
}
exports.LabelSelectorOptions = LabelSelectorOptions;
_b = JSII_RTTI_SYMBOL_1;
LabelSelectorOptions[_b] = { fqn: "cdk8s-plus-22.LabelSelectorOptions", version: "2.0.0-rc.15" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_c = JSII_RTTI_SYMBOL_1;
LabelSelector[_c] = { fqn: "cdk8s-plus-22.LabelSelector", version: "2.0.0-rc.15" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_d = JSII_RTTI_SYMBOL_1;
Pod[_d] = { fqn: "cdk8s-plus-22.Pod", version: "2.0.0-rc.15" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_e = JSII_RTTI_SYMBOL_1;
PodDns[_e] = { fqn: "cdk8s-plus-22.PodDns", version: "2.0.0-rc.15" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_f = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_f] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "2.0.0-rc.15" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, 'DoesNotExist', undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, 'Gt', values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, 'Lt', values);
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_g = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_g] = { fqn: "cdk8s-plus-22.NodeLabelQuery", version: "2.0.0-rc.15" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_h = JSII_RTTI_SYMBOL_1;
LabelExpression[_h] = { fqn: "cdk8s-plus-22.LabelExpression", version: "2.0.0-rc.15" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery('Equal', key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery('Exists', key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery('Exists');
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_j = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_j] = { fqn: "cdk8s-plus-22.NodeTaintQuery", version: "2.0.0-rc.15" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_k = JSII_RTTI_SYMBOL_1;
Pods[_k] = { fqn: "cdk8s-plus-22.Pods", version: "2.0.0-rc.15" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_l = JSII_RTTI_SYMBOL_1;
LabeledNode[_l] = { fqn: "cdk8s-plus-22.LabeledNode", version: "2.0.0-rc.15" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_m = JSII_RTTI_SYMBOL_1;
TaintedNode[_m] = { fqn: "cdk8s-plus-22.TaintedNode", version: "2.0.0-rc.15" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_o = JSII_RTTI_SYMBOL_1;
NamedNode[_o] = { fqn: "cdk8s-plus-22.NamedNode", version: "2.0.0-rc.15" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_p = JSII_RTTI_SYMBOL_1;
Node[_p] = { fqn: "cdk8s-plus-22.Node", version: "2.0.0-rc.15" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_q = JSII_RTTI_SYMBOL_1;
Topology[_q] = { fqn: "cdk8s-plus-22.Topology", version: "2.0.0-rc.15" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push({
                key: query.key,
                value: query.value,
                effect: query.effect,
                operator: query.operator,
                tolerationSeconds: query.evictAfter?.toSeconds(),
            });
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => ({ key: s.key, operator: s.operator, values: s.values })) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_r = JSII_RTTI_SYMBOL_1;
PodScheduling[_r] = { fqn: "cdk8s-plus-22.PodScheduling", version: "2.0.0-rc.15" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n));
    }
}
exports.PodConnections = PodConnections;
_s = JSII_RTTI_SYMBOL_1;
PodConnections[_s] = { fqn: "cdk8s-plus-22.PodConnections", version: "2.0.0-rc.15" };
//# sourceMappingURL=data:application/json;base64,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