"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !exports.hasOwnProperty(p)) __createBinding(exports, m, p);
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_codepipeline_1 = require("aws-cdk-lib/aws-codepipeline");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const sonar_code_scanner_1 = require("./code_scanner/sonar-code-scanner");
__exportStar(require("./code_scanner/sonar-code-scanner"), exports);
const DEFAULT_BRANCH_NAME = "mainline";
/**
 * An extension to CodePipeline which configures sane defaults for a NX Monorepo
 * codebase. In addition to this, it also creates a CodeCommit repository with
 * automated PR builds and approvals.
 */
class PDKPipeline extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        const codeRepository = new aws_codecommit_1.Repository(this, "CodeRepository", {
            repositoryName: props.repositoryName,
        });
        codeRepository.applyRemovalPolicy(props.codeCommitRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN);
        const accessLogsBucket = new aws_s3_1.Bucket(this, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_ENFORCED,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        const artifactBucket = new aws_s3_1.Bucket(this, "ArtifactsBucket", {
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: props.crossAccountKeys
                ? aws_s3_1.BucketEncryption.KMS
                : aws_s3_1.BucketEncryption.S3_MANAGED,
            encryptionKey: props.crossAccountKeys
                ? new aws_kms_1.Key(this, "ArtifactKey", {
                    enableKeyRotation: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                })
                : undefined,
            objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_ENFORCED,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            serverAccessLogsPrefix: "access-logs",
            serverAccessLogsBucket: accessLogsBucket,
        });
        const codePipeline = new aws_codepipeline_1.Pipeline(this, "CodePipeline", {
            enableKeyRotation: props.crossAccountKeys,
            restartExecutionOnUpdate: true,
            crossAccountKeys: props.crossAccountKeys,
            artifactBucket,
        });
        // ignore input and primaryOutputDirectory
        const { input, primaryOutputDirectory, commands, ...synthShellStepPartialProps } = props.synthShellStepPartialProps || {};
        const synthShellStep = new pipelines_1.ShellStep("Synth", {
            input: pipelines_1.CodePipelineSource.codeCommit(codeRepository, props.defaultBranchName || DEFAULT_BRANCH_NAME),
            installCommands: [
                "npm install -g aws-cdk",
                "yarn install --frozen-lockfile || npx projen && yarn install --frozen-lockfile",
            ],
            commands: commands && commands.length > 0
                ? commands
                : ["npx nx run-many --target=build --all"],
            primaryOutputDirectory: props.primarySynthDirectory,
            ...(synthShellStepPartialProps || {}),
        });
        synthShellStep.addOutputDirectory(".");
        const codePipelineProps = {
            codePipeline,
            ...props,
            crossAccountKeys: undefined,
            synth: synthShellStep,
        };
        this.codePipeline = new pipelines_1.CodePipeline(this, id, codePipelineProps);
        this.codeRepository = codeRepository;
        this.sonarCodeScannerConfig = props.sonarCodeScannerConfig
            ? {
                cdkOutDir: props.primarySynthDirectory,
                ...props.sonarCodeScannerConfig,
            }
            : undefined;
        new aws_cdk_lib_1.CfnOutput(this, "CodeRepositoryGRCUrl", {
            value: this.codeRepository.repositoryCloneUrlGrc,
        });
    }
    /**
     * @inheritDoc
     */
    addStage(stage, options) {
        // Add any root Aspects to the stage level as currently this doesn't happen automatically
        aws_cdk_lib_1.Aspects.of(stage.node.root).all.forEach((aspect) => aws_cdk_lib_1.Aspects.of(stage).add(aspect));
        return this.codePipeline.addStage(stage, options);
    }
    buildPipeline() {
        this.codePipeline.buildPipeline();
        this.sonarCodeScannerConfig &&
            new sonar_code_scanner_1.SonarCodeScanner(this, "SonarCodeScanner", {
                artifactBucketArn: this.codePipeline.pipeline.artifactBucket.bucketArn,
                artifactBucketKeyArn: this.codePipeline.pipeline.artifactBucket.encryptionKey?.keyArn,
                synthBuildArn: this.codePipeline.synthProject.projectArn,
                ...this.sonarCodeScannerConfig,
            });
        this.suppressCDKViolations();
    }
    suppressCDKViolations() {
        this.suppressRules(["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"], "Wildcards are needed for dynamically created resources.");
        this.suppressRules([
            "AwsSolutions-CB4",
            "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
        ], "Encryption of Codebuild is not required.");
        this.suppressRules(["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"], "Access Log buckets should not have s3 bucket logging");
    }
    suppressRules(rules, reason) {
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, rules.map((r) => ({
            id: r,
            reason,
        })), true);
    }
}
exports.PDKPipeline = PDKPipeline;
_a = JSII_RTTI_SYMBOL_1;
PDKPipeline[_a] = { fqn: "aws-prototyping-sdk.pipeline.PDKPipeline", version: "0.17.5" };
//# sourceMappingURL=data:application/json;base64,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