"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const nx_monorepo_1 = require("./nx-monorepo");
/**
 * Component which manged the project specific NX Config and is added to all NXMonorepo subprojects.
 *
 */
class NxProject extends projen_1.Component {
    constructor(project) {
        super(project);
        /**
         * List of implicit dependencies.
         *
         * @internal
         * @private
         */
        this._implicitDependencies = [];
    }
    /**
     * Retrieves an instance of NXProject if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project) {
        return project.components.find((c) => c instanceof NxProject);
    }
    /**
     * Adds an implicit dependency between the dependant (this project) and dependee.
     *
     * @param dependee project to add the implicit dependency on.
     */
    addImplicitDependency(dependee) {
        this._implicitDependencies.push(dependee instanceof projen_1.Project ? dependee.name : dependee);
    }
    /**
     * Generate a nx block in the package.json if this is a NodeProject. Otherwise generate a project.json with
     * the relevant NX configuration.
     */
    synthesize() {
        if (this.project instanceof javascript_1.NodeProject) {
            this._implicitDependencies.length > 0 &&
                this.project
                    .tryFindObjectFile("package.json")
                    ?.addOverride("nx.implicitDependencies", this._implicitDependencies);
        }
        else {
            const projectJson = this.project.tryFindObjectFile("project.json") ||
                new projen_1.JsonFile(this.project, "project.json", {
                    readonly: true,
                    marker: true,
                    obj: {},
                });
            const projectPath = path.relative(this.project.root.outdir, this.project.outdir);
            projectJson.addOverride("name", this.project.name);
            projectJson.addOverride("root", projectPath);
            this._implicitDependencies.length > 0 &&
                projectJson.addOverride("implicitDependencies", this._implicitDependencies);
            projectJson.addOverride("targets", this.project.tasks.all.reduce((p, c) => ({
                [c.name]: {
                    executor: "nx:run-commands",
                    options: {
                        command: `${nx_monorepo_1.buildDownloadExecutableCommand(this.project.root.package.packageManager, `projen ${c.name}`)}`,
                        cwd: projectPath,
                    },
                },
                ...p,
            }), {}));
        }
        super.synthesize();
    }
}
exports.NxProject = NxProject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibngtcHJvamVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm54LXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUE7c0NBQ3NDO0FBQ3RDLDZCQUE2QjtBQUM3QixtQ0FBc0Q7QUFDdEQsc0RBQW9EO0FBQ3BELCtDQUErRDtBQUUvRDs7O0dBR0c7QUFDSCxNQUFhLFNBQVUsU0FBUSxrQkFBUztJQW9CdEMsWUFBWSxPQUFnQjtRQUMxQixLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7UUFUakI7Ozs7O1dBS0c7UUFDYywwQkFBcUIsR0FBYSxFQUFFLENBQUM7SUFJdEQsQ0FBQztJQXJCRDs7OztPQUlHO0lBQ0gsTUFBTSxDQUFDLEVBQUUsQ0FBQyxPQUFnQjtRQUN4QixPQUFPLE9BQU8sQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLFlBQVksU0FBUyxDQUUvQyxDQUFDO0lBQ2hCLENBQUM7SUFjRDs7OztPQUlHO0lBQ0kscUJBQXFCLENBQUMsUUFBMEI7UUFDckQsSUFBSSxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FDN0IsUUFBUSxZQUFZLGdCQUFPLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FDdkQsQ0FBQztJQUNKLENBQUM7SUFFRDs7O09BR0c7SUFDSCxVQUFVO1FBQ1IsSUFBSSxJQUFJLENBQUMsT0FBTyxZQUFZLHdCQUFXLEVBQUU7WUFDdkMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sR0FBRyxDQUFDO2dCQUNuQyxJQUFJLENBQUMsT0FBTztxQkFDVCxpQkFBaUIsQ0FBQyxjQUFjLENBQUM7b0JBQ2xDLEVBQUUsV0FBVyxDQUFDLHlCQUF5QixFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1NBQzFFO2FBQU07WUFDTCxNQUFNLFdBQVcsR0FDZixJQUFJLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLGNBQWMsQ0FBQztnQkFDOUMsSUFBSSxpQkFBUSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsY0FBYyxFQUFFO29CQUN6QyxRQUFRLEVBQUUsSUFBSTtvQkFDZCxNQUFNLEVBQUUsSUFBSTtvQkFDWixHQUFHLEVBQUUsRUFBRTtpQkFDUixDQUFDLENBQUM7WUFFTCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUMvQixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQ3hCLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUNwQixDQUFDO1lBQ0YsV0FBVyxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNuRCxXQUFXLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztZQUM3QyxJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxHQUFHLENBQUM7Z0JBQ25DLFdBQVcsQ0FBQyxXQUFXLENBQ3JCLHNCQUFzQixFQUN0QixJQUFJLENBQUMscUJBQXFCLENBQzNCLENBQUM7WUFDSixXQUFXLENBQUMsV0FBVyxDQUNyQixTQUFTLEVBQ1QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FDM0IsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUNULENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUNSLFFBQVEsRUFBRSxpQkFBaUI7b0JBQzNCLE9BQU8sRUFBRTt3QkFDUCxPQUFPLEVBQUUsR0FBRyw0Q0FBOEIsQ0FDdkMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFvQixDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQ3pELFVBQVUsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUNuQixFQUFFO3dCQUNILEdBQUcsRUFBRSxXQUFXO3FCQUNqQjtpQkFDRjtnQkFDRCxHQUFHLENBQUM7YUFDTCxDQUFDLEVBQ0YsRUFBRSxDQUNILENBQ0YsQ0FBQztTQUNIO1FBQ0QsS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ3JCLENBQUM7Q0FDRjtBQXZGRCw4QkF1RkMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IENvbXBvbmVudCwgSnNvbkZpbGUsIFByb2plY3QgfSBmcm9tIFwicHJvamVuXCI7XG5pbXBvcnQgeyBOb2RlUHJvamVjdCB9IGZyb20gXCJwcm9qZW4vbGliL2phdmFzY3JpcHRcIjtcbmltcG9ydCB7IGJ1aWxkRG93bmxvYWRFeGVjdXRhYmxlQ29tbWFuZCB9IGZyb20gXCIuL254LW1vbm9yZXBvXCI7XG5cbi8qKlxuICogQ29tcG9uZW50IHdoaWNoIG1hbmdlZCB0aGUgcHJvamVjdCBzcGVjaWZpYyBOWCBDb25maWcgYW5kIGlzIGFkZGVkIHRvIGFsbCBOWE1vbm9yZXBvIHN1YnByb2plY3RzLlxuICpcbiAqL1xuZXhwb3J0IGNsYXNzIE54UHJvamVjdCBleHRlbmRzIENvbXBvbmVudCB7XG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYW4gaW5zdGFuY2Ugb2YgTlhQcm9qZWN0IGlmIG9uZSBpcyBhc3NvY2lhdGVkIHRvIHRoZSBnaXZlbiBwcm9qZWN0LlxuICAgKlxuICAgKiBAcGFyYW0gcHJvamVjdCBwcm9qZWN0IGluc3RhbmNlLlxuICAgKi9cbiAgc3RhdGljIG9mKHByb2plY3Q6IFByb2plY3QpOiBOeFByb2plY3QgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiBwcm9qZWN0LmNvbXBvbmVudHMuZmluZCgoYykgPT4gYyBpbnN0YW5jZW9mIE54UHJvamVjdCkgYXNcbiAgICAgIHwgTnhQcm9qZWN0XG4gICAgICB8IHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGltcGxpY2l0IGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IF9pbXBsaWNpdERlcGVuZGVuY2llczogc3RyaW5nW10gPSBbXTtcblxuICBjb25zdHJ1Y3Rvcihwcm9qZWN0OiBQcm9qZWN0KSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBpbXBsaWNpdCBkZXBlbmRlbmN5IGJldHdlZW4gdGhlIGRlcGVuZGFudCAodGhpcyBwcm9qZWN0KSBhbmQgZGVwZW5kZWUuXG4gICAqXG4gICAqIEBwYXJhbSBkZXBlbmRlZSBwcm9qZWN0IHRvIGFkZCB0aGUgaW1wbGljaXQgZGVwZW5kZW5jeSBvbi5cbiAgICovXG4gIHB1YmxpYyBhZGRJbXBsaWNpdERlcGVuZGVuY3koZGVwZW5kZWU6IFByb2plY3QgfCBzdHJpbmcpIHtcbiAgICB0aGlzLl9pbXBsaWNpdERlcGVuZGVuY2llcy5wdXNoKFxuICAgICAgZGVwZW5kZWUgaW5zdGFuY2VvZiBQcm9qZWN0ID8gZGVwZW5kZWUubmFtZSA6IGRlcGVuZGVlXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSBhIG54IGJsb2NrIGluIHRoZSBwYWNrYWdlLmpzb24gaWYgdGhpcyBpcyBhIE5vZGVQcm9qZWN0LiBPdGhlcndpc2UgZ2VuZXJhdGUgYSBwcm9qZWN0Lmpzb24gd2l0aFxuICAgKiB0aGUgcmVsZXZhbnQgTlggY29uZmlndXJhdGlvbi5cbiAgICovXG4gIHN5bnRoZXNpemUoKSB7XG4gICAgaWYgKHRoaXMucHJvamVjdCBpbnN0YW5jZW9mIE5vZGVQcm9qZWN0KSB7XG4gICAgICB0aGlzLl9pbXBsaWNpdERlcGVuZGVuY2llcy5sZW5ndGggPiAwICYmXG4gICAgICAgIHRoaXMucHJvamVjdFxuICAgICAgICAgIC50cnlGaW5kT2JqZWN0RmlsZShcInBhY2thZ2UuanNvblwiKVxuICAgICAgICAgID8uYWRkT3ZlcnJpZGUoXCJueC5pbXBsaWNpdERlcGVuZGVuY2llc1wiLCB0aGlzLl9pbXBsaWNpdERlcGVuZGVuY2llcyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IHByb2plY3RKc29uID1cbiAgICAgICAgdGhpcy5wcm9qZWN0LnRyeUZpbmRPYmplY3RGaWxlKFwicHJvamVjdC5qc29uXCIpIHx8XG4gICAgICAgIG5ldyBKc29uRmlsZSh0aGlzLnByb2plY3QsIFwicHJvamVjdC5qc29uXCIsIHtcbiAgICAgICAgICByZWFkb25seTogdHJ1ZSxcbiAgICAgICAgICBtYXJrZXI6IHRydWUsXG4gICAgICAgICAgb2JqOiB7fSxcbiAgICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IHByb2plY3RQYXRoID0gcGF0aC5yZWxhdGl2ZShcbiAgICAgICAgdGhpcy5wcm9qZWN0LnJvb3Qub3V0ZGlyLFxuICAgICAgICB0aGlzLnByb2plY3Qub3V0ZGlyXG4gICAgICApO1xuICAgICAgcHJvamVjdEpzb24uYWRkT3ZlcnJpZGUoXCJuYW1lXCIsIHRoaXMucHJvamVjdC5uYW1lKTtcbiAgICAgIHByb2plY3RKc29uLmFkZE92ZXJyaWRlKFwicm9vdFwiLCBwcm9qZWN0UGF0aCk7XG4gICAgICB0aGlzLl9pbXBsaWNpdERlcGVuZGVuY2llcy5sZW5ndGggPiAwICYmXG4gICAgICAgIHByb2plY3RKc29uLmFkZE92ZXJyaWRlKFxuICAgICAgICAgIFwiaW1wbGljaXREZXBlbmRlbmNpZXNcIixcbiAgICAgICAgICB0aGlzLl9pbXBsaWNpdERlcGVuZGVuY2llc1xuICAgICAgICApO1xuICAgICAgcHJvamVjdEpzb24uYWRkT3ZlcnJpZGUoXG4gICAgICAgIFwidGFyZ2V0c1wiLFxuICAgICAgICB0aGlzLnByb2plY3QudGFza3MuYWxsLnJlZHVjZShcbiAgICAgICAgICAocCwgYykgPT4gKHtcbiAgICAgICAgICAgIFtjLm5hbWVdOiB7XG4gICAgICAgICAgICAgIGV4ZWN1dG9yOiBcIm54OnJ1bi1jb21tYW5kc1wiLFxuICAgICAgICAgICAgICBvcHRpb25zOiB7XG4gICAgICAgICAgICAgICAgY29tbWFuZDogYCR7YnVpbGREb3dubG9hZEV4ZWN1dGFibGVDb21tYW5kKFxuICAgICAgICAgICAgICAgICAgKHRoaXMucHJvamVjdC5yb290IGFzIE5vZGVQcm9qZWN0KS5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyLFxuICAgICAgICAgICAgICAgICAgYHByb2plbiAke2MubmFtZX1gXG4gICAgICAgICAgICAgICAgKX1gLFxuICAgICAgICAgICAgICAgIGN3ZDogcHJvamVjdFBhdGgsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgLi4ucCxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICB7fVxuICAgICAgICApXG4gICAgICApO1xuICAgIH1cbiAgICBzdXBlci5zeW50aGVzaXplKCk7XG4gIH1cbn1cbiJdfQ==