"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.buildExecutableCommand = exports.buildDownloadExecutableCommand = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const nx_project_1 = require("./nx-project");
const syncpack_options_1 = require("./syncpack-options");
/**
 * Execute download/execute command to run based on package manager configured.
 *
 * @param packageManager package manager being used.
 * @param args args to append.
 */
function buildDownloadExecutableCommand(packageManager, ...args) {
    if (args.length === 0) {
        throw new Error("args must be specified");
    }
    const argLiteral = args.join(" ");
    switch (packageManager) {
        case javascript_1.NodePackageManager.YARN2:
            return `yarn dlx ${argLiteral}`;
        case javascript_1.NodePackageManager.PNPM:
            return `pnpm dlx ${argLiteral}`;
        case javascript_1.NodePackageManager.YARN:
        default:
            return `npx ${argLiteral}`;
    }
}
exports.buildDownloadExecutableCommand = buildDownloadExecutableCommand;
/**
 * Execute command to run based on package manager configured.
 *
 * @param packageManager package manager being used.
 * @param args args to append.
 */
function buildExecutableCommand(packageManager, ...args) {
    if (args.length === 0) {
        throw new Error("args must be specified");
    }
    const argLiteral = args.join(" ");
    switch (packageManager) {
        case javascript_1.NodePackageManager.YARN:
        case javascript_1.NodePackageManager.YARN2:
            return `yarn ${argLiteral}`;
        case javascript_1.NodePackageManager.PNPM:
            return `pnpm exec ${argLiteral}`;
        default:
            return `npx ${argLiteral}`;
    }
}
exports.buildExecutableCommand = buildExecutableCommand;
function binCommand(packageManager) {
    switch (packageManager) {
        case javascript_1.NodePackageManager.YARN:
        case javascript_1.NodePackageManager.YARN2:
            return `yarn bin`;
        case javascript_1.NodePackageManager.PNPM:
            return `pnpm bin`;
        default:
            return `npm bin`;
    }
}
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            projenCommand: options.packageManager
                ? buildExecutableCommand(options.packageManager, "projen")
                : undefined,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
            gitignore: [".nx/cache", ...(options.gitignore ?? [])],
            eslintOptions: options.eslintOptions ?? {
                dirs: ["."],
                ignorePatterns: ["packages/**/*.*"],
            },
            tsconfig: options.tsconfig ?? {
                compilerOptions: {
                    rootDir: ".",
                },
                include: ["**/*.ts"],
            },
        });
        this._options = options;
        // engines
        this.package.addEngine("node", ">=16");
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.PNPM: {
                // https://pnpm.io/package_json
                // https://github.com/pnpm/pnpm/releases/tag/v8.0.0
                this.package.addEngine("pnpm", ">=8");
                break;
            }
            case javascript_1.NodePackageManager.YARN: {
                this.package.addEngine("yarn", ">=1 <2");
                break;
            }
            case javascript_1.NodePackageManager.YARN2: {
                this.package.addEngine("yarn", ">=2 <3");
                break;
            }
        }
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // Add alias task for "projen" to synthesize workspace
        this.addTask("synth-workspace", {
            exec: buildExecutableCommand(this.package.packageManager, "projen"),
            description: "Synthesize workspace",
        });
        this.addTask("run-many", {
            receiveArgs: true,
            exec: buildExecutableCommand(this.package.packageManager, "nx", "run-many"),
            env: {
                NX_NON_NATIVE_HASHER: "true",
            },
            description: "Run task against multiple workspace projects",
        });
        this.addTask("graph", {
            receiveArgs: true,
            exec: buildExecutableCommand(this.package.packageManager, "nx", "graph"),
            description: "Generate dependency graph for monorepo",
        });
        // Map tasks to nx run-many
        if (options.scripts == null || options.scripts.build == null) {
            this._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        }
        if (options.scripts == null || options.scripts["pre-compile"] == null) {
            this._overrideNxBuildTask(this.preCompileTask, { target: "pre-compile" });
        }
        if (options.scripts == null || options.scripts.compile == null) {
            this._overrideNxBuildTask(this.compileTask, { target: "compile" });
        }
        if (options.scripts == null || options.scripts["post-compile"] == null) {
            this._overrideNxBuildTask(this.postCompileTask, {
                target: "post-compile",
            });
        }
        if (options.scripts == null || options.scripts.test == null) {
            this._overrideNxBuildTask(this.testTask, { target: "test" });
        }
        if (options.scripts == null || options.scripts.eslint == null) {
            // The Projenrc component of TypeScriptProject resets the eslint task as part of preSynthesize which would undo
            // our changes, so we disable further resets.
            this._overrideNxBuildTask(this.eslint?.eslintTask, { target: "eslint" }, { disableReset: true });
        }
        if (options.scripts == null || options.scripts.package == null) {
            this._overrideNxBuildTask(this.packageTask, { target: "package" });
        }
        if (options.scripts == null || options.scripts.prepare == null) {
            this._overrideNxBuildTask("prepare", { target: "prepare" });
        }
        if (options.scripts == null || options.scripts.watch == null) {
            this._overrideNxBuildTask(this.watchTask, {
                target: "watch",
                noBail: false,
                ignoreCycles: true,
                skipCache: true,
                outputStyle: "stream",
            });
        }
        this.addDevDeps("nx", "@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        this.package.addPackageResolutions("@types/babel__traverse@7.18.2");
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec(buildExecutableCommand(this.package.packageManager, "npm-check-updates", "--deep", "--rejectVersion", "0.0.0", "-u"));
            upgradeDepsTask.exec(buildExecutableCommand(this.package.packageManager, "syncpack", "fix-mismatches"));
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec(buildExecutableCommand(this.package.packageManager, "projen"));
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: options.nxConfig?.cacheableOperations || [
                                "build",
                                "test",
                            ],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                namedInputs: {
                    // https://nx.dev/more-concepts/customizing-inputs#defaults
                    default: ["{projectRoot}/**/*"],
                    ...options.nxConfig?.namedInputs,
                },
                targetDefaults: options.nxConfig?.targetDefaults,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Ensure NXProject is added when adding subprojects to the monorepo.
     *
     * @internal
     * @param subproject project to add.
     */
    _addSubProject(subproject) {
        !nx_project_1.NxProject.of(subproject) &&
            subproject._addComponent(new nx_project_1.NxProject(subproject));
        super._addSubProject(subproject);
    }
    /**
     * Helper to format `npx nx run-many ...` style command.
     * @param options
     */
    formatNxRunManyCommand(options) {
        const args = [];
        if (options.configuration) {
            args.push(`--configuration=${options.configuration}`);
        }
        if (options.runner) {
            args.push(`--runner=${options.runner}`);
        }
        if (options.parallel) {
            args.push(`--parallel=${options.parallel}`);
        }
        if (options.skipCache) {
            args.push("--skip-nx-cache");
        }
        if (options.ignoreCycles) {
            args.push("--nx-ignore-cycles");
        }
        if (options.noBail !== true) {
            args.push("--nx-bail");
        }
        if (options.projects && options.projects.length) {
            args.push(`--projects=${options.projects.join(",")}`);
        }
        if (options.exclude) {
            args.push(`--exclude=${options.exclude}`);
        }
        if (options.verbose) {
            args.push("--verbose");
        }
        return buildExecutableCommand(this.package.packageManager, "nx", "run-many", `--target=${options.target}`, `--output-style=${options.outputStyle || "stream"}`, ...args);
    }
    /**
     * Overrides "build" related project tasks (build, compile, test, etc.) with `npx nx run-many` format.
     * @param task - The task or task name to override
     * @param options - Nx run-many options
     * @param overrideOptions - Options for overriding the task
     * @returns - The task that was overridden
     * @internal
     */
    _overrideNxBuildTask(task, options, overrideOptions) {
        if (typeof task === "string") {
            task = this.tasks.tryFind(task);
        }
        if (task == null) {
            return;
        }
        if (overrideOptions?.force) {
            // @ts-ignore - private property
            task._locked = false;
        }
        task.reset(this.formatNxRunManyCommand(options), {
            receiveArgs: true,
        });
        task.description += " for all affected projects";
        // Fix for https://github.com/nrwl/nx/pull/15071
        task.env("NX_NON_NATIVE_HASHER", "true");
        if (overrideOptions?.disableReset) {
            // Prevent any further resets of the task to force it to remain as the overridden nx build task
            task.reset = () => { };
        }
        return task;
    }
    /**
     * Add project task that executes `npx nx run-many ...` style command.
     */
    addNxRunManyTask(name, options) {
        return this.addTask(name, {
            receiveArgs: true,
            exec: this.formatNxRunManyCommand(options),
            env: {
                NX_NON_NATIVE_HASHER: "true",
            },
        });
    }
    /**
     * Create an implicit dependency between two Projects. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     * @throws error if this is called on a dependent which does not have a NXProject component attached.
     */
    addImplicitDependency(dependent, dependee) {
        const nxProject = nx_project_1.NxProject.of(dependent);
        if (!nxProject) {
            throw new Error(`${dependent.name} does not have an NXProject associated.`);
        }
        else {
            nxProject.addImplicitDependency(dependee);
        }
    }
    /**
     * Adds a dependency between two Java Projects in the monorepo.
     * @param dependent project you want to have the dependency
     * @param dependee project you wish to depend on
     */
    addJavaDependency(dependent, dependee) {
        // Add implicit dependency for build order
        this.addImplicitDependency(dependent, dependee);
        // Add dependency in pom.xml
        dependent.addDependency(`${dependee.pom.groupId}/${dependee.pom.artifactId}@${dependee.pom.version}`);
        // Add a repository so that the dependency in the pom can be resolved
        dependent.pom.addRepository({
            id: dependee.name,
            url: `file://${path.join(path.relative(dependent.outdir, dependee.outdir), dependee.packaging.distdir)}`,
        });
    }
    /**
     * Adds a dependency between two Python Projects in the monorepo. The dependent must have Poetry enabled.
     * @param dependent project you want to have the dependency (must be a Poetry Python Project)
     * @param dependee project you wish to depend on
     * @throws error if the dependent does not have Poetry enabled
     */
    addPythonPoetryDependency(dependent, dependee) {
        // Check we're adding the dependency to a poetry python project
        if (!(dependent.depsManager instanceof python_1.Poetry)) {
            throw new Error(`${dependent.name} must be a PythonProject with Poetry enabled to add this dependency`);
        }
        // Add implicit dependency for build order
        this.addImplicitDependency(dependent, dependee);
        // Add local path dependency
        dependent.addDependency(`${dependee.name}@{path="${path.relative(dependent.outdir, dependee.outdir)}", develop=true}`);
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return [...this.instantiationOrderSubProjects].sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * Create symbolic links to all local workspace bins. This enables the usage of bins the same
     * way as consumers of the packages have when installing from the registry.
     */
    linkLocalWorkspaceBins() {
        const bins = [];
        this.subProjects.forEach((subProject) => {
            if (subProject instanceof javascript_1.NodeProject) {
                const pkgBins = subProject.package.manifest.bin() || {};
                bins.push(...Object.entries(pkgBins).map(([cmd, bin]) => {
                    const resolvedBin = path.join("$PWD", path.relative(this.outdir, subProject.outdir), bin);
                    return [cmd, resolvedBin];
                }));
            }
        });
        const linkTask = this.addTask("workspace:bin:link", {
            steps: bins.map(([cmd, bin]) => ({
                exec: `ln -s ${bin} $(${binCommand(this.package.packageManager)})/${cmd} &>/dev/null; exit 0;`,
            })),
        });
        (this.tasks.tryFind("prepare") || this.addTask("prepare")).spawn(linkTask);
    }
    preSynthesize() {
        super.preSynthesize();
        if (this._options.workspaceConfig?.linkLocalWorkspaceBins === true) {
            this.linkLocalWorkspaceBins();
        }
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            // PNPM hoisting hides transitive bundled dependencies which results in
            // transitive dependencies being packed incorrectly.
            this.subProjects.forEach((subProject) => {
                if (isNodeProject(subProject) && getBundledDeps(subProject).length) {
                    const pkgFolder = path.relative(this.root.outdir, subProject.outdir);
                    // Create a symlink in the sub-project node_modules for all transitive deps
                    // before running "package" task
                    subProject.packageTask.prependExec(`pdk-pnpm-link-bundled-transitive-deps ${pkgFolder}`);
                }
            });
        }
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.installNonNodeDependencies();
        // Prevent sub NodeProject packages from `postSynthesis` which will cause individual/extraneous installs.
        // The workspace package install will handle all the sub NodeProject packages automatically.
        const subProjectPackages = [];
        this.subProjects.forEach((subProject) => {
            if (isNodeProject(subProject)) {
                const subNodeProject = subProject;
                subProjectPackages.push(subNodeProject.package);
                // @ts-ignore - `installDependencies` is private
                subNodeProject.package.installDependencies = () => { };
            }
        });
        super.synth();
        // Force workspace install deps if any node subproject package has changed, unless the workspace changed
        if (
        // @ts-ignore - `file` is private
        this.package.file.changed !== true &&
            // @ts-ignore - `file` is private
            subProjectPackages.find((pkg) => pkg.file.changed === true)) {
            // @ts-ignore - `installDependencies` is private
            this.package.installDependencies();
        }
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        let noHoist = this.workspaceConfig?.noHoist;
        // Automatically add all sub-project "bundledDependencies" to workspace "hohoist", otherwise they are not bundled in npm package
        if (this.workspaceConfig?.disableNoHoistBundled !== true) {
            const noHoistBundled = this.subProjects.flatMap((sub) => {
                if (sub instanceof javascript_1.NodeProject) {
                    return getBundledDeps(sub).flatMap((dep) => [
                        `${sub.name}/${dep.name}`,
                        `${sub.name}/${dep.name}/*`,
                    ]);
                }
                return [];
            });
            if (noHoistBundled.length) {
                noHoist = [...(noHoist || []), ...noHoistBundled];
            }
        }
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        this.package.addField("workspaces", {
            packages: this.workspacePackages,
            nohoist: noHoist,
        });
    }
    /**
     * Ensures the install task for non-node projects is executed postinstall.
     *
     * @private
     */
    installNonNodeDependencies() {
        const installProjects = this.subProjects.filter((project) => !(project instanceof javascript_1.NodeProject) && project.tasks.tryFind("install"));
        if (installProjects.length > 0) {
            const monorepoInstallTask = this.tasks.tryFind("postinstall") ?? this.addTask("postinstall");
            monorepoInstallTask.exec(buildExecutableCommand(this.package.packageManager, `nx run-many --target install --projects ${installProjects
                .map((project) => project.name)
                .join(",")} --parallel=1`));
            // Update the nx.json to ensure that install-py follows dependency order
            this.nxJson.addOverride("targetDependencies.install", [
                {
                    target: "install",
                    projects: "dependencies",
                },
            ]);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.17.5" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
/**
 * Gets bundled dependencies for a given project
 */
function getBundledDeps(project) {
    return project.deps.all.filter((dep) => dep.type === projen_1.DependencyType.BUNDLED);
}
//# sourceMappingURL=data:application/json;base64,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