The nx-monorepo package vends a NxMonorepoProject Projen construct that adds [NX](https://nx.dev/getting-started/intro) monorepo support and manages your yarn/npm/pnpm workspaces on your behalf. This construct enables polyglot builds (and inter language build dependencies), build caching, dependency visualization and much, much more.

The PDK itself uses the nx-monorepo project itself and is a good reference for seeing how a complex, polyglot monorepo can be set up.

> **BREAKING CHANGES** (pre-release)
> - v0.13.0: `WorkspaceConfig.nxConfig` type `NxConfig => Nx.WorkspaceConfig`, and `overrideProjectTargets` removed in favor of `PDKProject.nx` config to manage Nx project configurations. See [PR 231](https://github.com/aws/aws-prototyping-sdk/pull/231).

To get started simply run the following command in an empty directory:

```bash
npx projen new --from @aws-prototyping-sdk/nx-monorepo
```

This will bootstrap a new Projen monorepo and contain the following in the .projenrc.ts:

```ts
import { nx_monorepo } from "aws-prototyping-sdk";

const project = new nx_monorepo.NxMonorepoProject({
  defaultReleaseBranch: "main",
  deps: ["aws-cdk-lib", "constructs", "cdk-nag"],
  devDeps: ["aws-prototyping-sdk"],
  name: "my-package",
});

project.synth();
```

To add new packages to the monorepo, you can simply add them as a child to the monorepo. To demonstrate, lets add a PDK Pipeline TS Project as a child as follows:

```ts
import { nx_monorepo } from "aws-prototyping-sdk";

const project = new nx_monorepo.NxMonorepoProject({
  defaultReleaseBranch: "main",
  deps: ["aws-cdk-lib", "constructs", "cdk-nag"],
  devDeps: ["aws-prototyping-sdk"],
  name: "my-package",
});

new PDKPipelineTsProject({
  parent: project,
  outdir: "packages/cicd",
  defaultReleaseBranch: "mainline",
  name: "cicd",
  cdkVersion: "2.1.0"
});

project.synth();
```

Once added, run `npx projen` from the root directory. You will now notice a new TS package has been created under the packages/cicd path.

Now let's add a python project to the monorepo and add an inter-language build dependency.

```ts
import { nx_monorepo } from "aws-prototyping-sdk";
import { PDKPipelineTsProject } from "aws-prototyping-sdk/pipeline";
import { PythonProject } from "projen/lib/python";

const project = new nx_monorepo.NxMonorepoProject({
  defaultReleaseBranch: "main",
  deps: ["aws-cdk-lib", "constructs", "cdk-nag"],
  devDeps: ["aws-prototyping-sdk"],
  name: "test",
});

const pipelineProject = new PDKPipelineTsProject({
  parent: project,
  outdir: "packages/cicd",
  defaultReleaseBranch: "mainline",
  name: "cicd",
  cdkVersion: "2.1.0"
});

// Standard Projen projects also work here
const pythonlib = new PythonProject({
  parent: project,
  outdir: "packages/pythonlib",
  authorEmail: "",
  authorName: "",
  moduleName: "pythonlib",
  name: "pythonlib",
  version: "0.0.0"
});

// Pipeline project depends on pythonlib to build first
project.addImplicitDependency(pipelineProject, pythonlib);

project.synth();
```

Run `npx projen` from the root directory. You will now notice a new Python package has been created under packages/pythonlib.

To visualize our dependency graph, run the following command from the root directory: `npx nx graph`.

Now lets test building our project, from the root directory run `npx nx run-many --target=build --nx-bail`. As you can see, the pythonlib was built first followed by the cicd package.
> This is equivalent to running `yarn build`, `pnpm build`, or `npm run build` depending on your node package manager, and similarly `yarn build` also accepts the same [Nx Run-Many options](https://nx.dev/packages/nx/documents/run-many#options) (eg: `yarn build --projects=cicd`).

The NxMonorepoProject also manages your yarn/pnpm workspaces for you and synthesizes these into your package.json pnpm-workspace.yml respectively.

For more information on NX commands, refer to this [documentation](https://nx.dev/using-nx/nx-cli).

### Homogenous Dependencies

As well as adding implicit dependencies, you can add dependencies between projects of the same language in order to have one project consume another project's code.

#### Typescript

Since the `NxMonorepoProject` manages a yarn/npm/pnpm workspace, configuring dependencies between Typescript projects is as straightforward as referencing them in `deps`.

Note that dependencies cannot be added in the same project synthesis (`npx projen`) as when projects are created. This means a two-pass approach is recommended, first to create your new projects, and then to add the dependencies.

For example:

First add your new projects:

```ts
new TypeScriptProject({
  parent: monorepo,
  outdir: "packages/a",
  defaultReleaseBranch: "main",
  name: "project-a"
});

new TypeScriptProject({
  parent: monorepo,
  outdir: "packages/b",
  defaultReleaseBranch: "main",
  name: "project-b",
});
```

Synthesise, then you can set up your dependency:

```ts
const a = new TypeScriptProject({
  parent: monorepo,
  outdir: "packages/a",
  defaultReleaseBranch: "main",
  name: "project-a"
});

new TypeScriptProject({
  parent: monorepo,
  outdir: "packages/b",
  defaultReleaseBranch: "main",
  name: "project-b",
  // B depends on A
  deps: [a.package.packageName],
});
```

#### Python

##### Poetry (Recommended)

The recommended way to configure dependencies between python projects within your monorepo is to use Poetry. Poetry sets up separate virtual environments per project but also supports local file dependencies. You can use the monorepo's `addPythonPoetryDependency` method:

```ts
const a = new PythonProject({
  parent: monorepo,
  outdir: 'packages/a',
  moduleName: 'a',
  name: 'a',
  authorName: 'jack',
  authorEmail: 'me@example.com',
  version: '1.0.0',
  poetry: true,
});

const b = new PythonProject({
  parent: monorepo,
  outdir: 'packages/b',
  moduleName: 'b',
  name: 'b',
  authorName: 'jack',
  authorEmail: 'me@example.com',
  version: '1.0.0',
  poetry: true,
});

// b depends on a
monorepo.addPythonPoetryDependency(b, a);
```

##### Pip

If you are using pip for your python projects, you can set up a dependency using a single shared virtual environment. You can then install packages you wish to depend on into that environment using pip's [Editable Installs](https://pip.pypa.io/en/stable/topics/local-project-installs/#editable-installs).

You will also need to add an implicit dependency to tell the monorepo the correct build order for your packages.

For example:

```ts
const sharedEnv: VenvOptions = {
  envdir: '../../.env',
};

const a = new PythonProject({
  parent: monorepo,
  outdir: 'packages/a',
  moduleName: 'a',
  name: 'a',
  authorName: 'jack',
  authorEmail: 'me@example.com',
  version: '1.0.0',
  venvOptions: sharedEnv,
});

// Install A into the virtual env since it is consumed by B
a.tasks.tryFind('install')!.exec('pip install --editable .');

const b = new PythonProject({
  parent: monorepo,
  outdir: 'packages/b',
  moduleName: 'b',
  name: 'b',
  authorName: 'jack',
  authorEmail: 'me@example.com',
  version: '1.0.0',
  venvOptions: sharedEnv,
  // B depends on A
  deps: [a.moduleName],
});

// Add the implicit dependency so that the monorepo will build A before B
monorepo.addImplicitDependency(b, a);
```

#### Java

The recommended way to configure dependencies between java projects within your monorepo is to use shared maven repositories. The default java project build will already create a distributable in the correct format for a maven repository in its `dist/java` folder, so you can use this as a repository. You can use the monorepo's `addJavaDependency` method:

For example:

```ts
const a = new JavaProject({
  parent: monorepo,
  outdir: 'packages/a',
  groupId: 'com.mycompany',
  artifactId: 'a',
  name: 'a',
  version: '1.0.0',
});

const b = new JavaProject({
  parent: monorepo,
  outdir: 'packages/b',
  groupId: 'com.mycompany',
  artifactId: 'b',
  name: 'b',
  version: '1.0.0',
});

// b depends on a
monorepo.addJavaDependency(b, a);
```

# API Reference <a name="API Reference" id="api-reference"></a>


## Structs <a name="Structs" id="Structs"></a>

### MonorepoUpgradeDepsOptions <a name="MonorepoUpgradeDepsOptions" id="@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions"></a>

Configuration for Monorepo Upgrade Deps task.

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions.Initializer"></a>

```typescript
import { MonorepoUpgradeDepsOptions } from '@aws-prototyping-sdk/nx-monorepo'

const monorepoUpgradeDepsOptions: MonorepoUpgradeDepsOptions = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions.property.syncpackConfig">syncpackConfig</a></code> | <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig">SyncpackConfig</a></code> | Syncpack configuration. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions.property.taskName">taskName</a></code> | <code>string</code> | Name of the task to create. |

---

##### `syncpackConfig`<sup>Optional</sup> <a name="syncpackConfig" id="@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions.property.syncpackConfig"></a>

```typescript
public readonly syncpackConfig: SyncpackConfig;
```

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig">SyncpackConfig</a>
- *Default:* SyncpackConfig.DEFAULT_CONFIG

Syncpack configuration.

No merging is performed and as such a complete syncpackConfig is required if supplied.

---

##### `taskName`<sup>Optional</sup> <a name="taskName" id="@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions.property.taskName"></a>

```typescript
public readonly taskName: string;
```

- *Type:* string
- *Default:* upgrade-deps

Name of the task to create.

---

### NamedInputs <a name="NamedInputs" id="@aws-prototyping-sdk/nx-monorepo.Nx.NamedInputs"></a>

Named inputs config.

> [https://nx.dev/reference/nx-json#inputs-&-namedinputs](https://nx.dev/reference/nx-json#inputs-&-namedinputs)

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.Nx.NamedInputs.Initializer"></a>

```typescript
import { Nx } from '@aws-prototyping-sdk/nx-monorepo'

const namedInputs: Nx.NamedInputs = { ... }
```


### NxMonorepoProjectOptions <a name="NxMonorepoProjectOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions"></a>

Configuration options for the NxMonorepoProject.

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.Initializer"></a>

```typescript
import { NxMonorepoProjectOptions } from '@aws-prototyping-sdk/nx-monorepo'

const nxMonorepoProjectOptions: NxMonorepoProjectOptions = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.name">name</a></code> | <code>string</code> | This is the name of your project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.commitGenerated">commitGenerated</a></code> | <code>boolean</code> | Whether to commit the managed files by default. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitIgnoreOptions">gitIgnoreOptions</a></code> | <code>projen.IgnoreFileOptions</code> | Configuration options for .gitignore file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitOptions">gitOptions</a></code> | <code>projen.GitOptions</code> | Configuration options for git. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.logging">logging</a></code> | <code>projen.LoggerOptions</code> | Configure logging options such as verbosity. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.outdir">outdir</a></code> | <code>string</code> | The root directory of the project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.parent">parent</a></code> | <code>projen.Project</code> | The parent project, if this project is part of a bigger project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenCommand">projenCommand</a></code> | <code>string</code> | The shell command to use in order to run the projen CLI. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJson">projenrcJson</a></code> | <code>boolean</code> | Generate (once) .projenrc.json (in JSON). Set to `false` in order to disable .projenrc.json generation. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJsonOptions">projenrcJsonOptions</a></code> | <code>projen.ProjenrcJsonOptions</code> | Options for .projenrc.json. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.renovatebot">renovatebot</a></code> | <code>boolean</code> | Use renovatebot to handle dependency upgrades. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.renovatebotOptions">renovatebotOptions</a></code> | <code>projen.RenovatebotOptions</code> | Options for renovatebot. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoApproveOptions">autoApproveOptions</a></code> | <code>projen.github.AutoApproveOptions</code> | Enable and configure the 'auto approve' workflow. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoMerge">autoMerge</a></code> | <code>boolean</code> | Enable automatic merging on GitHub. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoMergeOptions">autoMergeOptions</a></code> | <code>projen.github.AutoMergeOptions</code> | Configure options for automatic merging on GitHub. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.clobber">clobber</a></code> | <code>boolean</code> | Add a `clobber` task which resets the repo to origin. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.devContainer">devContainer</a></code> | <code>boolean</code> | Add a VSCode development environment (used for GitHub Codespaces). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.github">github</a></code> | <code>boolean</code> | Enable GitHub integration. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.githubOptions">githubOptions</a></code> | <code>projen.github.GitHubOptions</code> | Options for GitHub integration. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitpod">gitpod</a></code> | <code>boolean</code> | Add a Gitpod development environment. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.mergify">mergify</a></code> | <code>boolean</code> | Whether mergify should be enabled on this repository or not. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.mergifyOptions">mergifyOptions</a></code> | <code>projen.github.MergifyOptions</code> | Options for mergify. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projectType">projectType</a></code> | <code>projen.ProjectType</code> | Which type of project this is (library/app). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenCredentials">projenCredentials</a></code> | <code>projen.github.GithubCredentials</code> | Choose a method of providing GitHub API access for projen workflows. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenTokenSecret">projenTokenSecret</a></code> | <code>string</code> | The name of a secret which includes a GitHub Personal Access Token to be used by projen workflows. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.readme">readme</a></code> | <code>projen.SampleReadmeProps</code> | The README setup. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.stale">stale</a></code> | <code>boolean</code> | Auto-close of stale issues and pull request. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.staleOptions">staleOptions</a></code> | <code>projen.github.StaleOptions</code> | Auto-close stale issues and pull requests. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.vscode">vscode</a></code> | <code>boolean</code> | Enable VSCode integration. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.allowLibraryDependencies">allowLibraryDependencies</a></code> | <code>boolean</code> | Allow the project to include `peerDependencies` and `bundledDependencies`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorEmail">authorEmail</a></code> | <code>string</code> | Author's e-mail. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorName">authorName</a></code> | <code>string</code> | Author's name. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorOrganization">authorOrganization</a></code> | <code>boolean</code> | Is the author an organization. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorUrl">authorUrl</a></code> | <code>string</code> | Author's URL / Website. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoDetectBin">autoDetectBin</a></code> | <code>boolean</code> | Automatically add all executables under the `bin` directory to your `package.json` file under the `bin` section. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bin">bin</a></code> | <code>{[ key: string ]: string}</code> | Binary programs vended with your module. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bugsEmail">bugsEmail</a></code> | <code>string</code> | The email address to which issues should be reported. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bugsUrl">bugsUrl</a></code> | <code>string</code> | The url to your project's issue tracker. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bundledDeps">bundledDeps</a></code> | <code>string[]</code> | List of dependencies to bundle into this module. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.codeArtifactOptions">codeArtifactOptions</a></code> | <code>projen.javascript.CodeArtifactOptions</code> | Options for npm packages using AWS CodeArtifact. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.deps">deps</a></code> | <code>string[]</code> | Runtime dependencies of this module. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.description">description</a></code> | <code>string</code> | The description is just a string that helps people understand the purpose of the package. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.devDeps">devDeps</a></code> | <code>string[]</code> | Build dependencies for this module. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.entrypoint">entrypoint</a></code> | <code>string</code> | Module entrypoint (`main` in `package.json`). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.homepage">homepage</a></code> | <code>string</code> | Package's Homepage / Website. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.keywords">keywords</a></code> | <code>string[]</code> | Keywords to include in `package.json`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.license">license</a></code> | <code>string</code> | License's SPDX identifier. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.licensed">licensed</a></code> | <code>boolean</code> | Indicates if a license should be added. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.maxNodeVersion">maxNodeVersion</a></code> | <code>string</code> | Minimum node.js version to require via `engines` (inclusive). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.minNodeVersion">minNodeVersion</a></code> | <code>string</code> | Minimum Node.js version to require via package.json `engines` (inclusive). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmAccess">npmAccess</a></code> | <code>projen.javascript.NpmAccess</code> | Access level of the npm package. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmRegistry">npmRegistry</a></code> | <code>string</code> | The host name of the npm registry to publish to. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmRegistryUrl">npmRegistryUrl</a></code> | <code>string</code> | The base URL of the npm package registry. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmTokenSecret">npmTokenSecret</a></code> | <code>string</code> | GitHub secret which contains the NPM token to use when publishing packages. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.packageManager">packageManager</a></code> | <code>projen.javascript.NodePackageManager</code> | The Node Package Manager used to execute scripts. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.packageName">packageName</a></code> | <code>string</code> | The "name" in package.json. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.peerDependencyOptions">peerDependencyOptions</a></code> | <code>projen.javascript.PeerDependencyOptions</code> | Options for `peerDeps`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.peerDeps">peerDeps</a></code> | <code>string[]</code> | Peer dependencies for this module. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.pnpmVersion">pnpmVersion</a></code> | <code>string</code> | The version of PNPM to use if using PNPM as a package manager. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.repository">repository</a></code> | <code>string</code> | The repository is the location where the actual code for your package lives. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.repositoryDirectory">repositoryDirectory</a></code> | <code>string</code> | If the package.json for your package is not in the root directory (for example if it is part of a monorepo), you can specify the directory in which it lives. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.scopedPackagesOptions">scopedPackagesOptions</a></code> | <code>projen.javascript.ScopedPackagesOptions[]</code> | Options for privately hosted scoped packages. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.scripts">scripts</a></code> | <code>{[ key: string ]: string}</code> | npm scripts to include. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.stability">stability</a></code> | <code>string</code> | Package's Stability. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.jsiiReleaseVersion">jsiiReleaseVersion</a></code> | <code>string</code> | Version requirement of `publib` which is used to publish modules to npm. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.majorVersion">majorVersion</a></code> | <code>number</code> | Major version to release from the default branch. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.minMajorVersion">minMajorVersion</a></code> | <code>number</code> | Minimal Major version to release. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmDistTag">npmDistTag</a></code> | <code>string</code> | The npmDistTag to use when publishing from the default branch. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.postBuildSteps">postBuildSteps</a></code> | <code>projen.github.workflows.JobStep[]</code> | Steps to execute after build as part of the release workflow. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.prerelease">prerelease</a></code> | <code>string</code> | Bump versions from the default branch as pre-releases (e.g. "beta", "alpha", "pre"). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.publishDryRun">publishDryRun</a></code> | <code>boolean</code> | Instead of actually publishing to package managers, just print the publishing command. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.publishTasks">publishTasks</a></code> | <code>boolean</code> | Define publishing tasks that can be executed manually as well as workflows. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseBranches">releaseBranches</a></code> | <code>{[ key: string ]: projen.release.BranchOptions}</code> | Defines additional release branches. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseEveryCommit">releaseEveryCommit</a></code> | <code>boolean</code> | Automatically release new versions every commit to one of branches in `releaseBranches`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseFailureIssue">releaseFailureIssue</a></code> | <code>boolean</code> | Create a github issue on every failed publishing task. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseFailureIssueLabel">releaseFailureIssueLabel</a></code> | <code>string</code> | The label to apply to issues indicating publish failures. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseSchedule">releaseSchedule</a></code> | <code>string</code> | CRON schedule to trigger new releases. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseTagPrefix">releaseTagPrefix</a></code> | <code>string</code> | Automatically add the given prefix to release tags. Useful if you are releasing on multiple branches with overlapping version numbers. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseTrigger">releaseTrigger</a></code> | <code>projen.release.ReleaseTrigger</code> | The release trigger to use. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseWorkflowName">releaseWorkflowName</a></code> | <code>string</code> | The name of the default release workflow. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseWorkflowSetupSteps">releaseWorkflowSetupSteps</a></code> | <code>projen.github.workflows.JobStep[]</code> | A set of workflow steps to execute in order to setup the workflow container. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.versionrcOptions">versionrcOptions</a></code> | <code>{[ key: string ]: any}</code> | Custom configuration used when creating changelog with standard-version package. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowContainerImage">workflowContainerImage</a></code> | <code>string</code> | Container image to use for GitHub workflows. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowRunsOn">workflowRunsOn</a></code> | <code>string[]</code> | Github Runner selection labels. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.defaultReleaseBranch">defaultReleaseBranch</a></code> | <code>string</code> | The name of the main release branch. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.artifactsDirectory">artifactsDirectory</a></code> | <code>string</code> | A directory which will contain build artifacts. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoApproveUpgrades">autoApproveUpgrades</a></code> | <code>boolean</code> | Automatically approve deps upgrade PRs, allowing them to be merged by mergify (if configued). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.buildWorkflow">buildWorkflow</a></code> | <code>boolean</code> | Define a GitHub workflow for building PRs. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.buildWorkflowTriggers">buildWorkflowTriggers</a></code> | <code>projen.github.workflows.Triggers</code> | Build workflow triggers. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bundlerOptions">bundlerOptions</a></code> | <code>projen.javascript.BundlerOptions</code> | Options for `Bundler`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.codeCov">codeCov</a></code> | <code>boolean</code> | Define a GitHub workflow step for sending code coverage metrics to https://codecov.io/ Uses codecov/codecov-action@v3 A secret is required for private repos. Configured with @codeCovTokenSecret. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.codeCovTokenSecret">codeCovTokenSecret</a></code> | <code>string</code> | Define the secret name for a specified https://codecov.io/ token A secret is required to send coverage for private repositories. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.copyrightOwner">copyrightOwner</a></code> | <code>string</code> | License copyright owner. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.copyrightPeriod">copyrightPeriod</a></code> | <code>string</code> | The copyright years to put in the LICENSE file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.dependabot">dependabot</a></code> | <code>boolean</code> | Use dependabot to handle dependency upgrades. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.dependabotOptions">dependabotOptions</a></code> | <code>projen.github.DependabotOptions</code> | Options for dependabot. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.depsUpgrade">depsUpgrade</a></code> | <code>boolean</code> | Use github workflows to handle dependency upgrades. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.depsUpgradeOptions">depsUpgradeOptions</a></code> | <code>projen.javascript.UpgradeDependenciesOptions</code> | Options for `UpgradeDependencies`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitignore">gitignore</a></code> | <code>string[]</code> | Additional entries to .gitignore. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.jest">jest</a></code> | <code>boolean</code> | Setup jest unit tests. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.jestOptions">jestOptions</a></code> | <code>projen.javascript.JestOptions</code> | Jest options. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.mutableBuild">mutableBuild</a></code> | <code>boolean</code> | Automatically update files modified during builds to pull-request branches. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmignore">npmignore</a></code> | <code>string[]</code> | Additional entries to .npmignore. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmignoreEnabled">npmignoreEnabled</a></code> | <code>boolean</code> | Defines an .npmignore file. Normally this is only needed for libraries that are packaged as tarballs. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmIgnoreOptions">npmIgnoreOptions</a></code> | <code>projen.IgnoreFileOptions</code> | Configuration options for .npmignore file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.package">package</a></code> | <code>boolean</code> | Defines a `package` task that will produce an npm tarball under the artifacts directory (e.g. `dist`). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.prettier">prettier</a></code> | <code>boolean</code> | Setup prettier. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.prettierOptions">prettierOptions</a></code> | <code>projen.javascript.PrettierOptions</code> | Prettier options. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenDevDependency">projenDevDependency</a></code> | <code>boolean</code> | Indicates of "projen" should be installed as a devDependency. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJs">projenrcJs</a></code> | <code>boolean</code> | Generate (once) .projenrc.js (in JavaScript). Set to `false` in order to disable .projenrc.js generation. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJsOptions">projenrcJsOptions</a></code> | <code>projen.javascript.ProjenrcOptions</code> | Options for .projenrc.js. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenVersion">projenVersion</a></code> | <code>string</code> | Version of projen to install. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.pullRequestTemplate">pullRequestTemplate</a></code> | <code>boolean</code> | Include a GitHub pull request template. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.pullRequestTemplateContents">pullRequestTemplateContents</a></code> | <code>string[]</code> | The contents of the pull request template. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.release">release</a></code> | <code>boolean</code> | Add release management to this project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseToNpm">releaseToNpm</a></code> | <code>boolean</code> | Automatically release to npm when new versions are introduced. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseWorkflow">releaseWorkflow</a></code> | <code>boolean</code> | DEPRECATED: renamed to `release`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowBootstrapSteps">workflowBootstrapSteps</a></code> | <code>projen.github.workflows.JobStep[]</code> | Workflow steps to use in order to bootstrap this repo. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowGitIdentity">workflowGitIdentity</a></code> | <code>projen.github.GitIdentity</code> | The git identity to use in workflows. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowNodeVersion">workflowNodeVersion</a></code> | <code>string</code> | The node version to use in GitHub workflows. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.disableTsconfig">disableTsconfig</a></code> | <code>boolean</code> | Do not generate a `tsconfig.json` file (used by jsii projects since tsconfig.json is generated by the jsii compiler). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.disableTsconfigDev">disableTsconfigDev</a></code> | <code>boolean</code> | Do not generate a `tsconfig.dev.json` file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.docgen">docgen</a></code> | <code>boolean</code> | Docgen by Typedoc. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.docsDirectory">docsDirectory</a></code> | <code>string</code> | Docs directory. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.entrypointTypes">entrypointTypes</a></code> | <code>string</code> | The .d.ts file that includes the type declarations for this module. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.eslint">eslint</a></code> | <code>boolean</code> | Setup eslint. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.eslintOptions">eslintOptions</a></code> | <code>projen.javascript.EslintOptions</code> | Eslint options. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.libdir">libdir</a></code> | <code>string</code> | Typescript  artifacts output directory. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcTs">projenrcTs</a></code> | <code>boolean</code> | Use TypeScript for your projenrc file (`.projenrc.ts`). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcTsOptions">projenrcTsOptions</a></code> | <code>projen.typescript.ProjenrcOptions</code> | Options for .projenrc.ts. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.sampleCode">sampleCode</a></code> | <code>boolean</code> | Generate one-time sample in `src/` and `test/` if there are no files there. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.srcdir">srcdir</a></code> | <code>string</code> | Typescript sources directory. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.testdir">testdir</a></code> | <code>string</code> | Jest tests directory. Tests files should be named `xxx.test.ts`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.tsconfig">tsconfig</a></code> | <code>projen.javascript.TypescriptConfigOptions</code> | Custom TSConfig. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.tsconfigDev">tsconfigDev</a></code> | <code>projen.javascript.TypescriptConfigOptions</code> | Custom tsconfig options for the development tsconfig.json file (used for testing). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.tsconfigDevFile">tsconfigDevFile</a></code> | <code>string</code> | The name of the development tsconfig.json file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.typescriptVersion">typescriptVersion</a></code> | <code>string</code> | TypeScript version to use. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.monorepoUpgradeDeps">monorepoUpgradeDeps</a></code> | <code>boolean</code> | Whether to include an upgrade-deps task at the root of the monorepo which will upgrade all dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.monorepoUpgradeDepsOptions">monorepoUpgradeDepsOptions</a></code> | <code><a href="#@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions">MonorepoUpgradeDepsOptions</a></code> | Monorepo Upgrade Deps options. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.nxConfig">nxConfig</a></code> | <code>@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig</code> | Configuration for NX. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workspaceConfig">workspaceConfig</a></code> | <code><a href="#@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig">WorkspaceConfig</a></code> | Configuration for workspace. |

---

##### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.name"></a>

```typescript
public readonly name: string;
```

- *Type:* string
- *Default:* $BASEDIR

This is the name of your project.

---

##### `commitGenerated`<sup>Optional</sup> <a name="commitGenerated" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.commitGenerated"></a>

```typescript
public readonly commitGenerated: boolean;
```

- *Type:* boolean
- *Default:* true

Whether to commit the managed files by default.

---

##### `gitIgnoreOptions`<sup>Optional</sup> <a name="gitIgnoreOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitIgnoreOptions"></a>

```typescript
public readonly gitIgnoreOptions: IgnoreFileOptions;
```

- *Type:* projen.IgnoreFileOptions

Configuration options for .gitignore file.

---

##### `gitOptions`<sup>Optional</sup> <a name="gitOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitOptions"></a>

```typescript
public readonly gitOptions: GitOptions;
```

- *Type:* projen.GitOptions

Configuration options for git.

---

##### `logging`<sup>Optional</sup> <a name="logging" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.logging"></a>

```typescript
public readonly logging: LoggerOptions;
```

- *Type:* projen.LoggerOptions
- *Default:* {}

Configure logging options such as verbosity.

---

##### `outdir`<sup>Optional</sup> <a name="outdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.outdir"></a>

```typescript
public readonly outdir: string;
```

- *Type:* string
- *Default:* "."

The root directory of the project.

Relative to this directory, all files are synthesized.

If this project has a parent, this directory is relative to the parent
directory and it cannot be the same as the parent or any of it's other
sub-projects.

---

##### `parent`<sup>Optional</sup> <a name="parent" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.parent"></a>

```typescript
public readonly parent: Project;
```

- *Type:* projen.Project

The parent project, if this project is part of a bigger project.

---

##### `projenCommand`<sup>Optional</sup> <a name="projenCommand" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenCommand"></a>

```typescript
public readonly projenCommand: string;
```

- *Type:* string
- *Default:* "npx projen"

The shell command to use in order to run the projen CLI.

Can be used to customize in special environments.

---

##### `projenrcJson`<sup>Optional</sup> <a name="projenrcJson" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJson"></a>

```typescript
public readonly projenrcJson: boolean;
```

- *Type:* boolean
- *Default:* false

Generate (once) .projenrc.json (in JSON). Set to `false` in order to disable .projenrc.json generation.

---

##### `projenrcJsonOptions`<sup>Optional</sup> <a name="projenrcJsonOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJsonOptions"></a>

```typescript
public readonly projenrcJsonOptions: ProjenrcJsonOptions;
```

- *Type:* projen.ProjenrcJsonOptions
- *Default:* default options

Options for .projenrc.json.

---

##### `renovatebot`<sup>Optional</sup> <a name="renovatebot" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.renovatebot"></a>

```typescript
public readonly renovatebot: boolean;
```

- *Type:* boolean
- *Default:* false

Use renovatebot to handle dependency upgrades.

---

##### `renovatebotOptions`<sup>Optional</sup> <a name="renovatebotOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.renovatebotOptions"></a>

```typescript
public readonly renovatebotOptions: RenovatebotOptions;
```

- *Type:* projen.RenovatebotOptions
- *Default:* default options

Options for renovatebot.

---

##### `autoApproveOptions`<sup>Optional</sup> <a name="autoApproveOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoApproveOptions"></a>

```typescript
public readonly autoApproveOptions: AutoApproveOptions;
```

- *Type:* projen.github.AutoApproveOptions
- *Default:* auto approve is disabled

Enable and configure the 'auto approve' workflow.

---

##### `autoMerge`<sup>Optional</sup> <a name="autoMerge" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoMerge"></a>

```typescript
public readonly autoMerge: boolean;
```

- *Type:* boolean
- *Default:* true

Enable automatic merging on GitHub.

Has no effect if `github.mergify`
is set to false.

---

##### `autoMergeOptions`<sup>Optional</sup> <a name="autoMergeOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoMergeOptions"></a>

```typescript
public readonly autoMergeOptions: AutoMergeOptions;
```

- *Type:* projen.github.AutoMergeOptions
- *Default:* see defaults in `AutoMergeOptions`

Configure options for automatic merging on GitHub.

Has no effect if
`github.mergify` or `autoMerge` is set to false.

---

##### `clobber`<sup>Optional</sup> <a name="clobber" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.clobber"></a>

```typescript
public readonly clobber: boolean;
```

- *Type:* boolean
- *Default:* true, but false for subprojects

Add a `clobber` task which resets the repo to origin.

---

##### `devContainer`<sup>Optional</sup> <a name="devContainer" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.devContainer"></a>

```typescript
public readonly devContainer: boolean;
```

- *Type:* boolean
- *Default:* false

Add a VSCode development environment (used for GitHub Codespaces).

---

##### `github`<sup>Optional</sup> <a name="github" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.github"></a>

```typescript
public readonly github: boolean;
```

- *Type:* boolean
- *Default:* true

Enable GitHub integration.

Enabled by default for root projects. Disabled for non-root projects.

---

##### `githubOptions`<sup>Optional</sup> <a name="githubOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.githubOptions"></a>

```typescript
public readonly githubOptions: GitHubOptions;
```

- *Type:* projen.github.GitHubOptions
- *Default:* see GitHubOptions

Options for GitHub integration.

---

##### `gitpod`<sup>Optional</sup> <a name="gitpod" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitpod"></a>

```typescript
public readonly gitpod: boolean;
```

- *Type:* boolean
- *Default:* false

Add a Gitpod development environment.

---

##### ~~`mergify`~~<sup>Optional</sup> <a name="mergify" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.mergify"></a>

- *Deprecated:* use `githubOptions.mergify` instead

```typescript
public readonly mergify: boolean;
```

- *Type:* boolean
- *Default:* true

Whether mergify should be enabled on this repository or not.

---

##### ~~`mergifyOptions`~~<sup>Optional</sup> <a name="mergifyOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.mergifyOptions"></a>

- *Deprecated:* use `githubOptions.mergifyOptions` instead

```typescript
public readonly mergifyOptions: MergifyOptions;
```

- *Type:* projen.github.MergifyOptions
- *Default:* default options

Options for mergify.

---

##### ~~`projectType`~~<sup>Optional</sup> <a name="projectType" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projectType"></a>

- *Deprecated:* no longer supported at the base project level

```typescript
public readonly projectType: ProjectType;
```

- *Type:* projen.ProjectType
- *Default:* ProjectType.UNKNOWN

Which type of project this is (library/app).

---

##### `projenCredentials`<sup>Optional</sup> <a name="projenCredentials" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenCredentials"></a>

```typescript
public readonly projenCredentials: GithubCredentials;
```

- *Type:* projen.github.GithubCredentials
- *Default:* use a personal access token named PROJEN_GITHUB_TOKEN

Choose a method of providing GitHub API access for projen workflows.

---

##### ~~`projenTokenSecret`~~<sup>Optional</sup> <a name="projenTokenSecret" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenTokenSecret"></a>

- *Deprecated:* use `projenCredentials`

```typescript
public readonly projenTokenSecret: string;
```

- *Type:* string
- *Default:* "PROJEN_GITHUB_TOKEN"

The name of a secret which includes a GitHub Personal Access Token to be used by projen workflows.

This token needs to have the `repo`, `workflows`
and `packages` scope.

---

##### `readme`<sup>Optional</sup> <a name="readme" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.readme"></a>

```typescript
public readonly readme: SampleReadmeProps;
```

- *Type:* projen.SampleReadmeProps
- *Default:* { filename: 'README.md', contents: '# replace this' }

The README setup.

---

*Example*

```typescript
"{ filename: 'readme.md', contents: '# title' }"
```


##### `stale`<sup>Optional</sup> <a name="stale" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.stale"></a>

```typescript
public readonly stale: boolean;
```

- *Type:* boolean
- *Default:* false

Auto-close of stale issues and pull request.

See `staleOptions` for options.

---

##### `staleOptions`<sup>Optional</sup> <a name="staleOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.staleOptions"></a>

```typescript
public readonly staleOptions: StaleOptions;
```

- *Type:* projen.github.StaleOptions
- *Default:* see defaults in `StaleOptions`

Auto-close stale issues and pull requests.

To disable set `stale` to `false`.

---

##### `vscode`<sup>Optional</sup> <a name="vscode" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.vscode"></a>

```typescript
public readonly vscode: boolean;
```

- *Type:* boolean
- *Default:* true

Enable VSCode integration.

Enabled by default for root projects. Disabled for non-root projects.

---

##### `allowLibraryDependencies`<sup>Optional</sup> <a name="allowLibraryDependencies" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.allowLibraryDependencies"></a>

```typescript
public readonly allowLibraryDependencies: boolean;
```

- *Type:* boolean
- *Default:* true

Allow the project to include `peerDependencies` and `bundledDependencies`.

This is normally only allowed for libraries. For apps, there's no meaning
for specifying these.

---

##### `authorEmail`<sup>Optional</sup> <a name="authorEmail" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorEmail"></a>

```typescript
public readonly authorEmail: string;
```

- *Type:* string

Author's e-mail.

---

##### `authorName`<sup>Optional</sup> <a name="authorName" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorName"></a>

```typescript
public readonly authorName: string;
```

- *Type:* string

Author's name.

---

##### `authorOrganization`<sup>Optional</sup> <a name="authorOrganization" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorOrganization"></a>

```typescript
public readonly authorOrganization: boolean;
```

- *Type:* boolean

Is the author an organization.

---

##### `authorUrl`<sup>Optional</sup> <a name="authorUrl" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.authorUrl"></a>

```typescript
public readonly authorUrl: string;
```

- *Type:* string

Author's URL / Website.

---

##### `autoDetectBin`<sup>Optional</sup> <a name="autoDetectBin" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoDetectBin"></a>

```typescript
public readonly autoDetectBin: boolean;
```

- *Type:* boolean
- *Default:* true

Automatically add all executables under the `bin` directory to your `package.json` file under the `bin` section.

---

##### `bin`<sup>Optional</sup> <a name="bin" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bin"></a>

```typescript
public readonly bin: {[ key: string ]: string};
```

- *Type:* {[ key: string ]: string}

Binary programs vended with your module.

You can use this option to add/customize how binaries are represented in
your `package.json`, but unless `autoDetectBin` is `false`, every
executable file under `bin` will automatically be added to this section.

---

##### `bugsEmail`<sup>Optional</sup> <a name="bugsEmail" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bugsEmail"></a>

```typescript
public readonly bugsEmail: string;
```

- *Type:* string

The email address to which issues should be reported.

---

##### `bugsUrl`<sup>Optional</sup> <a name="bugsUrl" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bugsUrl"></a>

```typescript
public readonly bugsUrl: string;
```

- *Type:* string

The url to your project's issue tracker.

---

##### `bundledDeps`<sup>Optional</sup> <a name="bundledDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bundledDeps"></a>

```typescript
public readonly bundledDeps: string[];
```

- *Type:* string[]

List of dependencies to bundle into this module.

These modules will be
added both to the `dependencies` section and `bundledDependencies` section of
your `package.json`.

The recommendation is to only specify the module name here (e.g.
`express`). This will behave similar to `yarn add` or `npm install` in the
sense that it will add the module as a dependency to your `package.json`
file with the latest version (`^`). You can specify semver requirements in
the same syntax passed to `npm i` or `yarn add` (e.g. `express@^2`) and
this will be what you `package.json` will eventually include.

---

##### `codeArtifactOptions`<sup>Optional</sup> <a name="codeArtifactOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.codeArtifactOptions"></a>

```typescript
public readonly codeArtifactOptions: CodeArtifactOptions;
```

- *Type:* projen.javascript.CodeArtifactOptions
- *Default:* undefined

Options for npm packages using AWS CodeArtifact.

This is required if publishing packages to, or installing scoped packages from AWS CodeArtifact

---

##### `deps`<sup>Optional</sup> <a name="deps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.deps"></a>

```typescript
public readonly deps: string[];
```

- *Type:* string[]
- *Default:* []

Runtime dependencies of this module.

The recommendation is to only specify the module name here (e.g.
`express`). This will behave similar to `yarn add` or `npm install` in the
sense that it will add the module as a dependency to your `package.json`
file with the latest version (`^`). You can specify semver requirements in
the same syntax passed to `npm i` or `yarn add` (e.g. `express@^2`) and
this will be what you `package.json` will eventually include.

---

*Example*

```typescript
[ 'express', 'lodash', 'foo@^2' ]
```


##### `description`<sup>Optional</sup> <a name="description" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.description"></a>

```typescript
public readonly description: string;
```

- *Type:* string

The description is just a string that helps people understand the purpose of the package.

It can be used when searching for packages in a package manager as well.
See https://classic.yarnpkg.com/en/docs/package-json/#toc-description

---

##### `devDeps`<sup>Optional</sup> <a name="devDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.devDeps"></a>

```typescript
public readonly devDeps: string[];
```

- *Type:* string[]
- *Default:* []

Build dependencies for this module.

These dependencies will only be
available in your build environment but will not be fetched when this
module is consumed.

The recommendation is to only specify the module name here (e.g.
`express`). This will behave similar to `yarn add` or `npm install` in the
sense that it will add the module as a dependency to your `package.json`
file with the latest version (`^`). You can specify semver requirements in
the same syntax passed to `npm i` or `yarn add` (e.g. `express@^2`) and
this will be what you `package.json` will eventually include.

---

*Example*

```typescript
[ 'typescript', '@types/express' ]
```


##### `entrypoint`<sup>Optional</sup> <a name="entrypoint" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.entrypoint"></a>

```typescript
public readonly entrypoint: string;
```

- *Type:* string
- *Default:* "lib/index.js"

Module entrypoint (`main` in `package.json`).

Set to an empty string to not include `main` in your package.json

---

##### `homepage`<sup>Optional</sup> <a name="homepage" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.homepage"></a>

```typescript
public readonly homepage: string;
```

- *Type:* string

Package's Homepage / Website.

---

##### `keywords`<sup>Optional</sup> <a name="keywords" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.keywords"></a>

```typescript
public readonly keywords: string[];
```

- *Type:* string[]

Keywords to include in `package.json`.

---

##### `license`<sup>Optional</sup> <a name="license" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.license"></a>

```typescript
public readonly license: string;
```

- *Type:* string
- *Default:* "Apache-2.0"

License's SPDX identifier.

See https://github.com/projen/projen/tree/main/license-text for a list of supported licenses.
Use the `licensed` option if you want to no license to be specified.

---

##### `licensed`<sup>Optional</sup> <a name="licensed" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.licensed"></a>

```typescript
public readonly licensed: boolean;
```

- *Type:* boolean
- *Default:* true

Indicates if a license should be added.

---

##### `maxNodeVersion`<sup>Optional</sup> <a name="maxNodeVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.maxNodeVersion"></a>

```typescript
public readonly maxNodeVersion: string;
```

- *Type:* string
- *Default:* no max

Minimum node.js version to require via `engines` (inclusive).

---

##### `minNodeVersion`<sup>Optional</sup> <a name="minNodeVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.minNodeVersion"></a>

```typescript
public readonly minNodeVersion: string;
```

- *Type:* string
- *Default:* no "engines" specified

Minimum Node.js version to require via package.json `engines` (inclusive).

---

##### `npmAccess`<sup>Optional</sup> <a name="npmAccess" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmAccess"></a>

```typescript
public readonly npmAccess: NpmAccess;
```

- *Type:* projen.javascript.NpmAccess
- *Default:* for scoped packages (e.g. `foo@bar`), the default is `NpmAccess.RESTRICTED`, for non-scoped packages, the default is `NpmAccess.PUBLIC`.

Access level of the npm package.

---

##### ~~`npmRegistry`~~<sup>Optional</sup> <a name="npmRegistry" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmRegistry"></a>

- *Deprecated:* use `npmRegistryUrl` instead

```typescript
public readonly npmRegistry: string;
```

- *Type:* string

The host name of the npm registry to publish to.

Cannot be set together with `npmRegistryUrl`.

---

##### `npmRegistryUrl`<sup>Optional</sup> <a name="npmRegistryUrl" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmRegistryUrl"></a>

```typescript
public readonly npmRegistryUrl: string;
```

- *Type:* string
- *Default:* "https://registry.npmjs.org"

The base URL of the npm package registry.

Must be a URL (e.g. start with "https://" or "http://")

---

##### `npmTokenSecret`<sup>Optional</sup> <a name="npmTokenSecret" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmTokenSecret"></a>

```typescript
public readonly npmTokenSecret: string;
```

- *Type:* string
- *Default:* "NPM_TOKEN"

GitHub secret which contains the NPM token to use when publishing packages.

---

##### `packageManager`<sup>Optional</sup> <a name="packageManager" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.packageManager"></a>

```typescript
public readonly packageManager: NodePackageManager;
```

- *Type:* projen.javascript.NodePackageManager
- *Default:* NodePackageManager.YARN

The Node Package Manager used to execute scripts.

---

##### `packageName`<sup>Optional</sup> <a name="packageName" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.packageName"></a>

```typescript
public readonly packageName: string;
```

- *Type:* string
- *Default:* defaults to project name

The "name" in package.json.

---

##### `peerDependencyOptions`<sup>Optional</sup> <a name="peerDependencyOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.peerDependencyOptions"></a>

```typescript
public readonly peerDependencyOptions: PeerDependencyOptions;
```

- *Type:* projen.javascript.PeerDependencyOptions

Options for `peerDeps`.

---

##### `peerDeps`<sup>Optional</sup> <a name="peerDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.peerDeps"></a>

```typescript
public readonly peerDeps: string[];
```

- *Type:* string[]
- *Default:* []

Peer dependencies for this module.

Dependencies listed here are required to
be installed (and satisfied) by the _consumer_ of this library. Using peer
dependencies allows you to ensure that only a single module of a certain
library exists in the `node_modules` tree of your consumers.

Note that prior to npm@7, peer dependencies are _not_ automatically
installed, which means that adding peer dependencies to a library will be a
breaking change for your customers.

Unless `peerDependencyOptions.pinnedDevDependency` is disabled (it is
enabled by default), projen will automatically add a dev dependency with a
pinned version for each peer dependency. This will ensure that you build &
test your module against the lowest peer version required.

---

##### `pnpmVersion`<sup>Optional</sup> <a name="pnpmVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.pnpmVersion"></a>

```typescript
public readonly pnpmVersion: string;
```

- *Type:* string
- *Default:* "7"

The version of PNPM to use if using PNPM as a package manager.

---

##### `repository`<sup>Optional</sup> <a name="repository" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.repository"></a>

```typescript
public readonly repository: string;
```

- *Type:* string

The repository is the location where the actual code for your package lives.

See https://classic.yarnpkg.com/en/docs/package-json/#toc-repository

---

##### `repositoryDirectory`<sup>Optional</sup> <a name="repositoryDirectory" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.repositoryDirectory"></a>

```typescript
public readonly repositoryDirectory: string;
```

- *Type:* string

If the package.json for your package is not in the root directory (for example if it is part of a monorepo), you can specify the directory in which it lives.

---

##### `scopedPackagesOptions`<sup>Optional</sup> <a name="scopedPackagesOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.scopedPackagesOptions"></a>

```typescript
public readonly scopedPackagesOptions: ScopedPackagesOptions[];
```

- *Type:* projen.javascript.ScopedPackagesOptions[]
- *Default:* fetch all scoped packages from the public npm registry

Options for privately hosted scoped packages.

---

##### ~~`scripts`~~<sup>Optional</sup> <a name="scripts" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.scripts"></a>

- *Deprecated:* use `project.addTask()` or `package.setScript()`

```typescript
public readonly scripts: {[ key: string ]: string};
```

- *Type:* {[ key: string ]: string}
- *Default:* {}

npm scripts to include.

If a script has the same name as a standard script,
the standard script will be overwritten.
Also adds the script as a task.

---

##### `stability`<sup>Optional</sup> <a name="stability" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.stability"></a>

```typescript
public readonly stability: string;
```

- *Type:* string

Package's Stability.

---

##### `jsiiReleaseVersion`<sup>Optional</sup> <a name="jsiiReleaseVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.jsiiReleaseVersion"></a>

```typescript
public readonly jsiiReleaseVersion: string;
```

- *Type:* string
- *Default:* "latest"

Version requirement of `publib` which is used to publish modules to npm.

---

##### `majorVersion`<sup>Optional</sup> <a name="majorVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.majorVersion"></a>

```typescript
public readonly majorVersion: number;
```

- *Type:* number
- *Default:* Major version is not enforced.

Major version to release from the default branch.

If this is specified, we bump the latest version of this major version line.
If not specified, we bump the global latest version.

---

##### `minMajorVersion`<sup>Optional</sup> <a name="minMajorVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.minMajorVersion"></a>

```typescript
public readonly minMajorVersion: number;
```

- *Type:* number
- *Default:* No minimum version is being enforced

Minimal Major version to release.

This can be useful to set to 1, as breaking changes before the 1.x major
release are not incrementing the major version number.

Can not be set together with `majorVersion`.

---

##### `npmDistTag`<sup>Optional</sup> <a name="npmDistTag" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmDistTag"></a>

```typescript
public readonly npmDistTag: string;
```

- *Type:* string
- *Default:* "latest"

The npmDistTag to use when publishing from the default branch.

To set the npm dist-tag for release branches, set the `npmDistTag` property
for each branch.

---

##### `postBuildSteps`<sup>Optional</sup> <a name="postBuildSteps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.postBuildSteps"></a>

```typescript
public readonly postBuildSteps: JobStep[];
```

- *Type:* projen.github.workflows.JobStep[]
- *Default:* []

Steps to execute after build as part of the release workflow.

---

##### `prerelease`<sup>Optional</sup> <a name="prerelease" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.prerelease"></a>

```typescript
public readonly prerelease: string;
```

- *Type:* string
- *Default:* normal semantic versions

Bump versions from the default branch as pre-releases (e.g. "beta", "alpha", "pre").

---

##### `publishDryRun`<sup>Optional</sup> <a name="publishDryRun" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.publishDryRun"></a>

```typescript
public readonly publishDryRun: boolean;
```

- *Type:* boolean
- *Default:* false

Instead of actually publishing to package managers, just print the publishing command.

---

##### `publishTasks`<sup>Optional</sup> <a name="publishTasks" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.publishTasks"></a>

```typescript
public readonly publishTasks: boolean;
```

- *Type:* boolean
- *Default:* false

Define publishing tasks that can be executed manually as well as workflows.

Normally, publishing only happens within automated workflows. Enable this
in order to create a publishing task for each publishing activity.

---

##### `releaseBranches`<sup>Optional</sup> <a name="releaseBranches" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseBranches"></a>

```typescript
public readonly releaseBranches: {[ key: string ]: BranchOptions};
```

- *Type:* {[ key: string ]: projen.release.BranchOptions}
- *Default:* no additional branches are used for release. you can use `addBranch()` to add additional branches.

Defines additional release branches.

A workflow will be created for each
release branch which will publish releases from commits in this branch.
Each release branch _must_ be assigned a major version number which is used
to enforce that versions published from that branch always use that major
version. If multiple branches are used, the `majorVersion` field must also
be provided for the default branch.

---

##### ~~`releaseEveryCommit`~~<sup>Optional</sup> <a name="releaseEveryCommit" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseEveryCommit"></a>

- *Deprecated:* Use `releaseTrigger: ReleaseTrigger.continuous()` instead

```typescript
public readonly releaseEveryCommit: boolean;
```

- *Type:* boolean
- *Default:* true

Automatically release new versions every commit to one of branches in `releaseBranches`.

---

##### `releaseFailureIssue`<sup>Optional</sup> <a name="releaseFailureIssue" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseFailureIssue"></a>

```typescript
public readonly releaseFailureIssue: boolean;
```

- *Type:* boolean
- *Default:* false

Create a github issue on every failed publishing task.

---

##### `releaseFailureIssueLabel`<sup>Optional</sup> <a name="releaseFailureIssueLabel" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseFailureIssueLabel"></a>

```typescript
public readonly releaseFailureIssueLabel: string;
```

- *Type:* string
- *Default:* "failed-release"

The label to apply to issues indicating publish failures.

Only applies if `releaseFailureIssue` is true.

---

##### ~~`releaseSchedule`~~<sup>Optional</sup> <a name="releaseSchedule" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseSchedule"></a>

- *Deprecated:* Use `releaseTrigger: ReleaseTrigger.scheduled()` instead

```typescript
public readonly releaseSchedule: string;
```

- *Type:* string
- *Default:* no scheduled releases

CRON schedule to trigger new releases.

---

##### `releaseTagPrefix`<sup>Optional</sup> <a name="releaseTagPrefix" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseTagPrefix"></a>

```typescript
public readonly releaseTagPrefix: string;
```

- *Type:* string
- *Default:* "v"

Automatically add the given prefix to release tags. Useful if you are releasing on multiple branches with overlapping version numbers.

Note: this prefix is used to detect the latest tagged version
when bumping, so if you change this on a project with an existing version
history, you may need to manually tag your latest release
with the new prefix.

---

##### `releaseTrigger`<sup>Optional</sup> <a name="releaseTrigger" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseTrigger"></a>

```typescript
public readonly releaseTrigger: ReleaseTrigger;
```

- *Type:* projen.release.ReleaseTrigger
- *Default:* Continuous releases (`ReleaseTrigger.continuous()`)

The release trigger to use.

---

##### `releaseWorkflowName`<sup>Optional</sup> <a name="releaseWorkflowName" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseWorkflowName"></a>

```typescript
public readonly releaseWorkflowName: string;
```

- *Type:* string
- *Default:* "Release"

The name of the default release workflow.

---

##### `releaseWorkflowSetupSteps`<sup>Optional</sup> <a name="releaseWorkflowSetupSteps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseWorkflowSetupSteps"></a>

```typescript
public readonly releaseWorkflowSetupSteps: JobStep[];
```

- *Type:* projen.github.workflows.JobStep[]

A set of workflow steps to execute in order to setup the workflow container.

---

##### `versionrcOptions`<sup>Optional</sup> <a name="versionrcOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.versionrcOptions"></a>

```typescript
public readonly versionrcOptions: {[ key: string ]: any};
```

- *Type:* {[ key: string ]: any}
- *Default:* standard configuration applicable for GitHub repositories

Custom configuration used when creating changelog with standard-version package.

Given values either append to default configuration or overwrite values in it.

---

##### `workflowContainerImage`<sup>Optional</sup> <a name="workflowContainerImage" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowContainerImage"></a>

```typescript
public readonly workflowContainerImage: string;
```

- *Type:* string
- *Default:* default image

Container image to use for GitHub workflows.

---

##### `workflowRunsOn`<sup>Optional</sup> <a name="workflowRunsOn" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowRunsOn"></a>

```typescript
public readonly workflowRunsOn: string[];
```

- *Type:* string[]
- *Default:* ["ubuntu-latest"]

Github Runner selection labels.

---

##### `defaultReleaseBranch`<sup>Required</sup> <a name="defaultReleaseBranch" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.defaultReleaseBranch"></a>

```typescript
public readonly defaultReleaseBranch: string;
```

- *Type:* string
- *Default:* "main"

The name of the main release branch.

---

##### `artifactsDirectory`<sup>Optional</sup> <a name="artifactsDirectory" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.artifactsDirectory"></a>

```typescript
public readonly artifactsDirectory: string;
```

- *Type:* string
- *Default:* "dist"

A directory which will contain build artifacts.

---

##### `autoApproveUpgrades`<sup>Optional</sup> <a name="autoApproveUpgrades" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.autoApproveUpgrades"></a>

```typescript
public readonly autoApproveUpgrades: boolean;
```

- *Type:* boolean
- *Default:* true

Automatically approve deps upgrade PRs, allowing them to be merged by mergify (if configued).

Throw if set to true but `autoApproveOptions` are not defined.

---

##### `buildWorkflow`<sup>Optional</sup> <a name="buildWorkflow" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.buildWorkflow"></a>

```typescript
public readonly buildWorkflow: boolean;
```

- *Type:* boolean
- *Default:* true if not a subproject

Define a GitHub workflow for building PRs.

---

##### `buildWorkflowTriggers`<sup>Optional</sup> <a name="buildWorkflowTriggers" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.buildWorkflowTriggers"></a>

```typescript
public readonly buildWorkflowTriggers: Triggers;
```

- *Type:* projen.github.workflows.Triggers
- *Default:* "{ pullRequest: {}, workflowDispatch: {} }"

Build workflow triggers.

---

##### `bundlerOptions`<sup>Optional</sup> <a name="bundlerOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.bundlerOptions"></a>

```typescript
public readonly bundlerOptions: BundlerOptions;
```

- *Type:* projen.javascript.BundlerOptions

Options for `Bundler`.

---

##### `codeCov`<sup>Optional</sup> <a name="codeCov" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.codeCov"></a>

```typescript
public readonly codeCov: boolean;
```

- *Type:* boolean
- *Default:* false

Define a GitHub workflow step for sending code coverage metrics to https://codecov.io/ Uses codecov/codecov-action@v3 A secret is required for private repos. Configured with @codeCovTokenSecret.

---

##### `codeCovTokenSecret`<sup>Optional</sup> <a name="codeCovTokenSecret" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.codeCovTokenSecret"></a>

```typescript
public readonly codeCovTokenSecret: string;
```

- *Type:* string
- *Default:* if this option is not specified, only public repositories are supported

Define the secret name for a specified https://codecov.io/ token A secret is required to send coverage for private repositories.

---

##### `copyrightOwner`<sup>Optional</sup> <a name="copyrightOwner" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.copyrightOwner"></a>

```typescript
public readonly copyrightOwner: string;
```

- *Type:* string
- *Default:* defaults to the value of authorName or "" if `authorName` is undefined.

License copyright owner.

---

##### `copyrightPeriod`<sup>Optional</sup> <a name="copyrightPeriod" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.copyrightPeriod"></a>

```typescript
public readonly copyrightPeriod: string;
```

- *Type:* string
- *Default:* current year

The copyright years to put in the LICENSE file.

---

##### `dependabot`<sup>Optional</sup> <a name="dependabot" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.dependabot"></a>

```typescript
public readonly dependabot: boolean;
```

- *Type:* boolean
- *Default:* false

Use dependabot to handle dependency upgrades.

Cannot be used in conjunction with `depsUpgrade`.

---

##### `dependabotOptions`<sup>Optional</sup> <a name="dependabotOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.dependabotOptions"></a>

```typescript
public readonly dependabotOptions: DependabotOptions;
```

- *Type:* projen.github.DependabotOptions
- *Default:* default options

Options for dependabot.

---

##### `depsUpgrade`<sup>Optional</sup> <a name="depsUpgrade" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.depsUpgrade"></a>

```typescript
public readonly depsUpgrade: boolean;
```

- *Type:* boolean
- *Default:* true

Use github workflows to handle dependency upgrades.

Cannot be used in conjunction with `dependabot`.

---

##### `depsUpgradeOptions`<sup>Optional</sup> <a name="depsUpgradeOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.depsUpgradeOptions"></a>

```typescript
public readonly depsUpgradeOptions: UpgradeDependenciesOptions;
```

- *Type:* projen.javascript.UpgradeDependenciesOptions
- *Default:* default options

Options for `UpgradeDependencies`.

---

##### `gitignore`<sup>Optional</sup> <a name="gitignore" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.gitignore"></a>

```typescript
public readonly gitignore: string[];
```

- *Type:* string[]

Additional entries to .gitignore.

---

##### `jest`<sup>Optional</sup> <a name="jest" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.jest"></a>

```typescript
public readonly jest: boolean;
```

- *Type:* boolean
- *Default:* true

Setup jest unit tests.

---

##### `jestOptions`<sup>Optional</sup> <a name="jestOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.jestOptions"></a>

```typescript
public readonly jestOptions: JestOptions;
```

- *Type:* projen.javascript.JestOptions
- *Default:* default options

Jest options.

---

##### `mutableBuild`<sup>Optional</sup> <a name="mutableBuild" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.mutableBuild"></a>

```typescript
public readonly mutableBuild: boolean;
```

- *Type:* boolean
- *Default:* true

Automatically update files modified during builds to pull-request branches.

This means
that any files synthesized by projen or e.g. test snapshots will always be up-to-date
before a PR is merged.

Implies that PR builds do not have anti-tamper checks.

---

##### ~~`npmignore`~~<sup>Optional</sup> <a name="npmignore" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmignore"></a>

- *Deprecated:* - use `project.addPackageIgnore`

```typescript
public readonly npmignore: string[];
```

- *Type:* string[]

Additional entries to .npmignore.

---

##### `npmignoreEnabled`<sup>Optional</sup> <a name="npmignoreEnabled" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmignoreEnabled"></a>

```typescript
public readonly npmignoreEnabled: boolean;
```

- *Type:* boolean
- *Default:* true

Defines an .npmignore file. Normally this is only needed for libraries that are packaged as tarballs.

---

##### `npmIgnoreOptions`<sup>Optional</sup> <a name="npmIgnoreOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.npmIgnoreOptions"></a>

```typescript
public readonly npmIgnoreOptions: IgnoreFileOptions;
```

- *Type:* projen.IgnoreFileOptions

Configuration options for .npmignore file.

---

##### `package`<sup>Optional</sup> <a name="package" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.package"></a>

```typescript
public readonly package: boolean;
```

- *Type:* boolean
- *Default:* true

Defines a `package` task that will produce an npm tarball under the artifacts directory (e.g. `dist`).

---

##### `prettier`<sup>Optional</sup> <a name="prettier" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.prettier"></a>

```typescript
public readonly prettier: boolean;
```

- *Type:* boolean
- *Default:* false

Setup prettier.

---

##### `prettierOptions`<sup>Optional</sup> <a name="prettierOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.prettierOptions"></a>

```typescript
public readonly prettierOptions: PrettierOptions;
```

- *Type:* projen.javascript.PrettierOptions
- *Default:* default options

Prettier options.

---

##### `projenDevDependency`<sup>Optional</sup> <a name="projenDevDependency" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenDevDependency"></a>

```typescript
public readonly projenDevDependency: boolean;
```

- *Type:* boolean
- *Default:* true

Indicates of "projen" should be installed as a devDependency.

---

##### `projenrcJs`<sup>Optional</sup> <a name="projenrcJs" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJs"></a>

```typescript
public readonly projenrcJs: boolean;
```

- *Type:* boolean
- *Default:* true if projenrcJson is false

Generate (once) .projenrc.js (in JavaScript). Set to `false` in order to disable .projenrc.js generation.

---

##### `projenrcJsOptions`<sup>Optional</sup> <a name="projenrcJsOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcJsOptions"></a>

```typescript
public readonly projenrcJsOptions: ProjenrcOptions;
```

- *Type:* projen.javascript.ProjenrcOptions
- *Default:* default options

Options for .projenrc.js.

---

##### `projenVersion`<sup>Optional</sup> <a name="projenVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenVersion"></a>

```typescript
public readonly projenVersion: string;
```

- *Type:* string
- *Default:* Defaults to the latest version.

Version of projen to install.

---

##### `pullRequestTemplate`<sup>Optional</sup> <a name="pullRequestTemplate" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.pullRequestTemplate"></a>

```typescript
public readonly pullRequestTemplate: boolean;
```

- *Type:* boolean
- *Default:* true

Include a GitHub pull request template.

---

##### `pullRequestTemplateContents`<sup>Optional</sup> <a name="pullRequestTemplateContents" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.pullRequestTemplateContents"></a>

```typescript
public readonly pullRequestTemplateContents: string[];
```

- *Type:* string[]
- *Default:* default content

The contents of the pull request template.

---

##### `release`<sup>Optional</sup> <a name="release" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.release"></a>

```typescript
public readonly release: boolean;
```

- *Type:* boolean
- *Default:* true (false for subprojects)

Add release management to this project.

---

##### `releaseToNpm`<sup>Optional</sup> <a name="releaseToNpm" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseToNpm"></a>

```typescript
public readonly releaseToNpm: boolean;
```

- *Type:* boolean
- *Default:* false

Automatically release to npm when new versions are introduced.

---

##### ~~`releaseWorkflow`~~<sup>Optional</sup> <a name="releaseWorkflow" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.releaseWorkflow"></a>

- *Deprecated:* see `release`.

```typescript
public readonly releaseWorkflow: boolean;
```

- *Type:* boolean
- *Default:* true if not a subproject

DEPRECATED: renamed to `release`.

---

##### `workflowBootstrapSteps`<sup>Optional</sup> <a name="workflowBootstrapSteps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowBootstrapSteps"></a>

```typescript
public readonly workflowBootstrapSteps: JobStep[];
```

- *Type:* projen.github.workflows.JobStep[]
- *Default:* "yarn install --frozen-lockfile && yarn projen"

Workflow steps to use in order to bootstrap this repo.

---

##### `workflowGitIdentity`<sup>Optional</sup> <a name="workflowGitIdentity" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowGitIdentity"></a>

```typescript
public readonly workflowGitIdentity: GitIdentity;
```

- *Type:* projen.github.GitIdentity
- *Default:* GitHub Actions

The git identity to use in workflows.

---

##### `workflowNodeVersion`<sup>Optional</sup> <a name="workflowNodeVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workflowNodeVersion"></a>

```typescript
public readonly workflowNodeVersion: string;
```

- *Type:* string
- *Default:* same as `minNodeVersion`

The node version to use in GitHub workflows.

---

##### `disableTsconfig`<sup>Optional</sup> <a name="disableTsconfig" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.disableTsconfig"></a>

```typescript
public readonly disableTsconfig: boolean;
```

- *Type:* boolean
- *Default:* false

Do not generate a `tsconfig.json` file (used by jsii projects since tsconfig.json is generated by the jsii compiler).

---

##### `disableTsconfigDev`<sup>Optional</sup> <a name="disableTsconfigDev" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.disableTsconfigDev"></a>

```typescript
public readonly disableTsconfigDev: boolean;
```

- *Type:* boolean
- *Default:* false

Do not generate a `tsconfig.dev.json` file.

---

##### `docgen`<sup>Optional</sup> <a name="docgen" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.docgen"></a>

```typescript
public readonly docgen: boolean;
```

- *Type:* boolean
- *Default:* false

Docgen by Typedoc.

---

##### `docsDirectory`<sup>Optional</sup> <a name="docsDirectory" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.docsDirectory"></a>

```typescript
public readonly docsDirectory: string;
```

- *Type:* string
- *Default:* "docs"

Docs directory.

---

##### `entrypointTypes`<sup>Optional</sup> <a name="entrypointTypes" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.entrypointTypes"></a>

```typescript
public readonly entrypointTypes: string;
```

- *Type:* string
- *Default:* .d.ts file derived from the project's entrypoint (usually lib/index.d.ts)

The .d.ts file that includes the type declarations for this module.

---

##### `eslint`<sup>Optional</sup> <a name="eslint" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.eslint"></a>

```typescript
public readonly eslint: boolean;
```

- *Type:* boolean
- *Default:* true

Setup eslint.

---

##### `eslintOptions`<sup>Optional</sup> <a name="eslintOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.eslintOptions"></a>

```typescript
public readonly eslintOptions: EslintOptions;
```

- *Type:* projen.javascript.EslintOptions
- *Default:* opinionated default options

Eslint options.

---

##### `libdir`<sup>Optional</sup> <a name="libdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.libdir"></a>

```typescript
public readonly libdir: string;
```

- *Type:* string
- *Default:* "lib"

Typescript  artifacts output directory.

---

##### `projenrcTs`<sup>Optional</sup> <a name="projenrcTs" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcTs"></a>

```typescript
public readonly projenrcTs: boolean;
```

- *Type:* boolean
- *Default:* false

Use TypeScript for your projenrc file (`.projenrc.ts`).

---

##### `projenrcTsOptions`<sup>Optional</sup> <a name="projenrcTsOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.projenrcTsOptions"></a>

```typescript
public readonly projenrcTsOptions: ProjenrcOptions;
```

- *Type:* projen.typescript.ProjenrcOptions

Options for .projenrc.ts.

---

##### `sampleCode`<sup>Optional</sup> <a name="sampleCode" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.sampleCode"></a>

```typescript
public readonly sampleCode: boolean;
```

- *Type:* boolean
- *Default:* true

Generate one-time sample in `src/` and `test/` if there are no files there.

---

##### `srcdir`<sup>Optional</sup> <a name="srcdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.srcdir"></a>

```typescript
public readonly srcdir: string;
```

- *Type:* string
- *Default:* "src"

Typescript sources directory.

---

##### `testdir`<sup>Optional</sup> <a name="testdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.testdir"></a>

```typescript
public readonly testdir: string;
```

- *Type:* string
- *Default:* "test"

Jest tests directory. Tests files should be named `xxx.test.ts`.

If this directory is under `srcdir` (e.g. `src/test`, `src/__tests__`),
then tests are going to be compiled into `lib/` and executed as javascript.
If the test directory is outside of `src`, then we configure jest to
compile the code in-memory.

---

##### `tsconfig`<sup>Optional</sup> <a name="tsconfig" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.tsconfig"></a>

```typescript
public readonly tsconfig: TypescriptConfigOptions;
```

- *Type:* projen.javascript.TypescriptConfigOptions
- *Default:* default options

Custom TSConfig.

---

##### `tsconfigDev`<sup>Optional</sup> <a name="tsconfigDev" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.tsconfigDev"></a>

```typescript
public readonly tsconfigDev: TypescriptConfigOptions;
```

- *Type:* projen.javascript.TypescriptConfigOptions
- *Default:* use the production tsconfig options

Custom tsconfig options for the development tsconfig.json file (used for testing).

---

##### `tsconfigDevFile`<sup>Optional</sup> <a name="tsconfigDevFile" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.tsconfigDevFile"></a>

```typescript
public readonly tsconfigDevFile: string;
```

- *Type:* string
- *Default:* "tsconfig.dev.json"

The name of the development tsconfig.json file.

---

##### `typescriptVersion`<sup>Optional</sup> <a name="typescriptVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.typescriptVersion"></a>

```typescript
public readonly typescriptVersion: string;
```

- *Type:* string
- *Default:* "latest"

TypeScript version to use.

NOTE: Typescript is not semantically versioned and should remain on the
same minor, so we recommend using a `~` dependency (e.g. `~1.2.3`).

---

##### `monorepoUpgradeDeps`<sup>Optional</sup> <a name="monorepoUpgradeDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.monorepoUpgradeDeps"></a>

```typescript
public readonly monorepoUpgradeDeps: boolean;
```

- *Type:* boolean
- *Default:* true

Whether to include an upgrade-deps task at the root of the monorepo which will upgrade all dependencies.

---

##### `monorepoUpgradeDepsOptions`<sup>Optional</sup> <a name="monorepoUpgradeDepsOptions" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.monorepoUpgradeDepsOptions"></a>

```typescript
public readonly monorepoUpgradeDepsOptions: MonorepoUpgradeDepsOptions;
```

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.MonorepoUpgradeDepsOptions">MonorepoUpgradeDepsOptions</a>
- *Default:* undefined

Monorepo Upgrade Deps options.

This is only used if monorepoUpgradeDeps is true.

---

##### `nxConfig`<sup>Optional</sup> <a name="nxConfig" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.nxConfig"></a>

```typescript
public readonly nxConfig: WorkspaceConfig;
```

- *Type:* @aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig

Configuration for NX.

---

##### `workspaceConfig`<sup>Optional</sup> <a name="workspaceConfig" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions.property.workspaceConfig"></a>

```typescript
public readonly workspaceConfig: WorkspaceConfig;
```

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig">WorkspaceConfig</a>

Configuration for workspace.

---

### NxRunManyOptions <a name="NxRunManyOptions" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions"></a>

> [https://nx.dev/packages/nx/documents/run-many#options](https://nx.dev/packages/nx/documents/run-many#options)

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.Initializer"></a>

```typescript
import { NxRunManyOptions } from '@aws-prototyping-sdk/nx-monorepo'

const nxRunManyOptions: NxRunManyOptions = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.target">target</a></code> | <code>string</code> | Task to run for affected projects. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.configuration">configuration</a></code> | <code>string</code> | This is the configuration to use when performing tasks on projects. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.exclude">exclude</a></code> | <code>string</code> | Exclude certain projects from being processed. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.ignoreCycles">ignoreCycles</a></code> | <code>boolean</code> | Ignore cycles in the task graph. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.noBail">noBail</a></code> | <code>boolean</code> | Do not stop command execution after the first failed task. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.outputStyle">outputStyle</a></code> | <code>string</code> | Defines how Nx emits outputs tasks logs. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.parallel">parallel</a></code> | <code>number</code> | Max number of parallel processes. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.projects">projects</a></code> | <code>string[]</code> | Project to run as list project names and/or patterns. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.runner">runner</a></code> | <code>string</code> | This is the name of the tasks runner configuration in nx.json. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.skipCache">skipCache</a></code> | <code>boolean</code> | Rerun the tasks even when the results are available in the cache. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.verbose">verbose</a></code> | <code>boolean</code> | Prints additional information about the commands (e.g. stack traces). |

---

##### `target`<sup>Required</sup> <a name="target" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.target"></a>

```typescript
public readonly target: string;
```

- *Type:* string

Task to run for affected projects.

---

##### `configuration`<sup>Optional</sup> <a name="configuration" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.configuration"></a>

```typescript
public readonly configuration: string;
```

- *Type:* string

This is the configuration to use when performing tasks on projects.

---

##### `exclude`<sup>Optional</sup> <a name="exclude" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.exclude"></a>

```typescript
public readonly exclude: string;
```

- *Type:* string

Exclude certain projects from being processed.

---

##### `ignoreCycles`<sup>Optional</sup> <a name="ignoreCycles" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.ignoreCycles"></a>

```typescript
public readonly ignoreCycles: boolean;
```

- *Type:* boolean

Ignore cycles in the task graph.

---

##### `noBail`<sup>Optional</sup> <a name="noBail" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.noBail"></a>

```typescript
public readonly noBail: boolean;
```

- *Type:* boolean

Do not stop command execution after the first failed task.

---

##### `outputStyle`<sup>Optional</sup> <a name="outputStyle" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.outputStyle"></a>

```typescript
public readonly outputStyle: string;
```

- *Type:* string
- *Default:* "stream"

Defines how Nx emits outputs tasks logs.

---

##### `parallel`<sup>Optional</sup> <a name="parallel" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.parallel"></a>

```typescript
public readonly parallel: number;
```

- *Type:* number
- *Default:* 3

Max number of parallel processes.

---

##### `projects`<sup>Optional</sup> <a name="projects" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.projects"></a>

```typescript
public readonly projects: string[];
```

- *Type:* string[]

Project to run as list project names and/or patterns.

---

##### `runner`<sup>Optional</sup> <a name="runner" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.runner"></a>

```typescript
public readonly runner: string;
```

- *Type:* string

This is the name of the tasks runner configuration in nx.json.

---

##### `skipCache`<sup>Optional</sup> <a name="skipCache" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.skipCache"></a>

```typescript
public readonly skipCache: boolean;
```

- *Type:* boolean

Rerun the tasks even when the results are available in the cache.

---

##### `verbose`<sup>Optional</sup> <a name="verbose" id="@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions.property.verbose"></a>

```typescript
public readonly verbose: boolean;
```

- *Type:* boolean

Prints additional information about the commands (e.g. stack traces).

---

### ProjectConfig <a name="ProjectConfig" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig"></a>

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.Initializer"></a>

```typescript
import { Nx } from '@aws-prototyping-sdk/nx-monorepo'

const projectConfig: Nx.ProjectConfig = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.implicitDependencies">implicitDependencies</a></code> | <code>string[]</code> | Implicit dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.includedScripts">includedScripts</a></code> | <code>string[]</code> | Explicit list of scripts for Nx to include. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.namedInputs">namedInputs</a></code> | <code>@aws-prototyping-sdk/nx-monorepo.Nx.NamedInputs</code> | Named inputs. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.tags">tags</a></code> | <code>string[]</code> | Project tag annotations. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.targets">targets</a></code> | <code>{[ key: string ]: @aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget}</code> | Targets configuration. |

---

##### `implicitDependencies`<sup>Optional</sup> <a name="implicitDependencies" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.implicitDependencies"></a>

```typescript
public readonly implicitDependencies: string[];
```

- *Type:* string[]

Implicit dependencies.

> [https://nx.dev/reference/project-configuration#implicitdependencies](https://nx.dev/reference/project-configuration#implicitdependencies)

---

##### `includedScripts`<sup>Optional</sup> <a name="includedScripts" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.includedScripts"></a>

```typescript
public readonly includedScripts: string[];
```

- *Type:* string[]

Explicit list of scripts for Nx to include.

> [https://nx.dev/reference/project-configuration#ignoring-package.json-scripts](https://nx.dev/reference/project-configuration#ignoring-package.json-scripts)

---

##### `namedInputs`<sup>Optional</sup> <a name="namedInputs" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.namedInputs"></a>

```typescript
public readonly namedInputs: NamedInputs;
```

- *Type:* @aws-prototyping-sdk/nx-monorepo.Nx.NamedInputs

Named inputs.

> [https://nx.dev/reference/nx-json#inputs-&-namedinputs](https://nx.dev/reference/nx-json#inputs-&-namedinputs)

---

##### `tags`<sup>Optional</sup> <a name="tags" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.tags"></a>

```typescript
public readonly tags: string[];
```

- *Type:* string[]

Project tag annotations.

> [https://nx.dev/reference/project-configuration#tags](https://nx.dev/reference/project-configuration#tags)

---

##### `targets`<sup>Optional</sup> <a name="targets" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectConfig.property.targets"></a>

```typescript
public readonly targets: {[ key: string ]: ProjectTarget};
```

- *Type:* {[ key: string ]: @aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget}

Targets configuration.

> [https://nx.dev/reference/project-configuration](https://nx.dev/reference/project-configuration)

---

### ProjectTarget <a name="ProjectTarget" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget"></a>

Project Target.

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget.Initializer"></a>

```typescript
import { Nx } from '@aws-prototyping-sdk/nx-monorepo'

const projectTarget: Nx.ProjectTarget = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget.property.dependsOn">dependsOn</a></code> | <code>@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency[]</code> | List of Target Dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget.property.inputs">inputs</a></code> | <code>string[]</code> | List of inputs to hash for cache key, relative to the root of the monorepo. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget.property.outputs">outputs</a></code> | <code>string[]</code> | List of outputs to cache, relative to the root of the monorepo. |

---

##### `dependsOn`<sup>Optional</sup> <a name="dependsOn" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget.property.dependsOn"></a>

```typescript
public readonly dependsOn: TargetDependency[];
```

- *Type:* @aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency[]

List of Target Dependencies.

---

##### `inputs`<sup>Optional</sup> <a name="inputs" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget.property.inputs"></a>

```typescript
public readonly inputs: string[];
```

- *Type:* string[]

List of inputs to hash for cache key, relative to the root of the monorepo.

note: must start with leading /

---

##### `outputs`<sup>Optional</sup> <a name="outputs" id="@aws-prototyping-sdk/nx-monorepo.Nx.ProjectTarget.property.outputs"></a>

```typescript
public readonly outputs: string[];
```

- *Type:* string[]

List of outputs to cache, relative to the root of the monorepo.

note: must start with leading /

---

### SemverGroup <a name="SemverGroup" id="@aws-prototyping-sdk/nx-monorepo.SemverGroup"></a>

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.SemverGroup.Initializer"></a>

```typescript
import { SemverGroup } from '@aws-prototyping-sdk/nx-monorepo'

const semverGroup: SemverGroup = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.dependencies">dependencies</a></code> | <code>string[]</code> | the names of the dependencies (eg. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.packages">packages</a></code> | <code>string[]</code> | the names of packages in your monorepo which belong to this group, taken from the "name" field in package.json, not the package directory name. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.range">range</a></code> | <code>string</code> | the semver range which dependencies in this group should use. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.dependencyTypes">dependencyTypes</a></code> | <code>string[]</code> | optionally only apply this group to dependencies of the provided types. |

---

##### `dependencies`<sup>Required</sup> <a name="dependencies" id="@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.dependencies"></a>

```typescript
public readonly dependencies: string[];
```

- *Type:* string[]

the names of the dependencies (eg.

"lodash") which belong to this group

---

##### `packages`<sup>Required</sup> <a name="packages" id="@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.packages"></a>

```typescript
public readonly packages: string[];
```

- *Type:* string[]

the names of packages in your monorepo which belong to this group, taken from the "name" field in package.json, not the package directory name.

---

##### `range`<sup>Required</sup> <a name="range" id="@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.range"></a>

```typescript
public readonly range: string;
```

- *Type:* string

the semver range which dependencies in this group should use.

---

##### `dependencyTypes`<sup>Optional</sup> <a name="dependencyTypes" id="@aws-prototyping-sdk/nx-monorepo.SemverGroup.property.dependencyTypes"></a>

```typescript
public readonly dependencyTypes: string[];
```

- *Type:* string[]

optionally only apply this group to dependencies of the provided types.

---

### SyncpackConfig <a name="SyncpackConfig" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig"></a>

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.Initializer"></a>

```typescript
import { SyncpackConfig } from '@aws-prototyping-sdk/nx-monorepo'

const syncpackConfig: SyncpackConfig = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.dependencyTypes">dependencyTypes</a></code> | <code>string[]</code> | which dependency properties to search within. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.dev">dev</a></code> | <code>boolean</code> | whether to search within devDependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.filter">filter</a></code> | <code>string</code> | a string which will be passed to `new RegExp()` to match against package names that should be included. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.indent">indent</a></code> | <code>string</code> | the character(s) to be used to indent your package.json files when writing to disk. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.overrides">overrides</a></code> | <code>boolean</code> | whether to search within npm overrides. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.peer">peer</a></code> | <code>boolean</code> | whether to search within peerDependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.pnpmOverrides">pnpmOverrides</a></code> | <code>boolean</code> | whether to search within pnpm overrides. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.prod">prod</a></code> | <code>boolean</code> | whether to search within dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.resolutions">resolutions</a></code> | <code>boolean</code> | whether to search within yarn resolutions. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.semverGroups">semverGroups</a></code> | <code><a href="#@aws-prototyping-sdk/nx-monorepo.SemverGroup">SemverGroup</a>[]</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.semverRange">semverRange</a></code> | <code>string</code> | defaults to `""` to ensure that exact dependency versions are used instead of loose ranges. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.sortAz">sortAz</a></code> | <code>string[]</code> | which fields within package.json files should be sorted alphabetically. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.sortFirst">sortFirst</a></code> | <code>string[]</code> | which fields within package.json files should appear at the top, and in what order. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.source">source</a></code> | <code>string[]</code> | glob patterns for package.json file locations. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.versionGroups">versionGroups</a></code> | <code><a href="#@aws-prototyping-sdk/nx-monorepo.VersionGroup">VersionGroup</a>[]</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.workspace">workspace</a></code> | <code>boolean</code> | whether to include the versions of the `--source` packages developed in your workspace/monorepo as part of the search for versions to sync. |

---

##### `dependencyTypes`<sup>Required</sup> <a name="dependencyTypes" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.dependencyTypes"></a>

```typescript
public readonly dependencyTypes: string[];
```

- *Type:* string[]

which dependency properties to search within.

---

##### `dev`<sup>Required</sup> <a name="dev" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.dev"></a>

```typescript
public readonly dev: boolean;
```

- *Type:* boolean

whether to search within devDependencies.

---

##### `filter`<sup>Required</sup> <a name="filter" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.filter"></a>

```typescript
public readonly filter: string;
```

- *Type:* string

a string which will be passed to `new RegExp()` to match against package names that should be included.

---

##### `indent`<sup>Required</sup> <a name="indent" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.indent"></a>

```typescript
public readonly indent: string;
```

- *Type:* string

the character(s) to be used to indent your package.json files when writing to disk.

---

##### `overrides`<sup>Required</sup> <a name="overrides" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.overrides"></a>

```typescript
public readonly overrides: boolean;
```

- *Type:* boolean

whether to search within npm overrides.

---

##### `peer`<sup>Required</sup> <a name="peer" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.peer"></a>

```typescript
public readonly peer: boolean;
```

- *Type:* boolean

whether to search within peerDependencies.

---

##### `pnpmOverrides`<sup>Required</sup> <a name="pnpmOverrides" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.pnpmOverrides"></a>

```typescript
public readonly pnpmOverrides: boolean;
```

- *Type:* boolean

whether to search within pnpm overrides.

---

##### `prod`<sup>Required</sup> <a name="prod" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.prod"></a>

```typescript
public readonly prod: boolean;
```

- *Type:* boolean

whether to search within dependencies.

---

##### `resolutions`<sup>Required</sup> <a name="resolutions" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.resolutions"></a>

```typescript
public readonly resolutions: boolean;
```

- *Type:* boolean

whether to search within yarn resolutions.

---

##### `semverGroups`<sup>Required</sup> <a name="semverGroups" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.semverGroups"></a>

```typescript
public readonly semverGroups: SemverGroup[];
```

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.SemverGroup">SemverGroup</a>[]

---

##### `semverRange`<sup>Required</sup> <a name="semverRange" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.semverRange"></a>

```typescript
public readonly semverRange: string;
```

- *Type:* string

defaults to `""` to ensure that exact dependency versions are used instead of loose ranges.

---

##### `sortAz`<sup>Required</sup> <a name="sortAz" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.sortAz"></a>

```typescript
public readonly sortAz: string[];
```

- *Type:* string[]

which fields within package.json files should be sorted alphabetically.

---

##### `sortFirst`<sup>Required</sup> <a name="sortFirst" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.sortFirst"></a>

```typescript
public readonly sortFirst: string[];
```

- *Type:* string[]

which fields within package.json files should appear at the top, and in what order.

---

##### `source`<sup>Required</sup> <a name="source" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.source"></a>

```typescript
public readonly source: string[];
```

- *Type:* string[]

glob patterns for package.json file locations.

---

##### `versionGroups`<sup>Required</sup> <a name="versionGroups" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.versionGroups"></a>

```typescript
public readonly versionGroups: VersionGroup[];
```

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.VersionGroup">VersionGroup</a>[]

---

##### `workspace`<sup>Required</sup> <a name="workspace" id="@aws-prototyping-sdk/nx-monorepo.SyncpackConfig.property.workspace"></a>

```typescript
public readonly workspace: boolean;
```

- *Type:* boolean

whether to include the versions of the `--source` packages developed in your workspace/monorepo as part of the search for versions to sync.

---

### TargetDefaults <a name="TargetDefaults" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDefaults"></a>

Target defaults config.

> [https://nx.dev/reference/nx-json#target-defaults](https://nx.dev/reference/nx-json#target-defaults)

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDefaults.Initializer"></a>

```typescript
import { Nx } from '@aws-prototyping-sdk/nx-monorepo'

const targetDefaults: Nx.TargetDefaults = { ... }
```


### TargetDependency <a name="TargetDependency" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency"></a>

Represents an NX Target Dependency.

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency.Initializer"></a>

```typescript
import { Nx } from '@aws-prototyping-sdk/nx-monorepo'

const targetDependency: Nx.TargetDependency = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency.property.projects">projects</a></code> | <code>@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependencyProject</code> | Target dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency.property.target">target</a></code> | <code>string</code> | Projen target i.e: build, test, etc. |

---

##### `projects`<sup>Required</sup> <a name="projects" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency.property.projects"></a>

```typescript
public readonly projects: TargetDependencyProject;
```

- *Type:* @aws-prototyping-sdk/nx-monorepo.Nx.TargetDependencyProject

Target dependencies.

---

##### `target`<sup>Required</sup> <a name="target" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency.property.target"></a>

```typescript
public readonly target: string;
```

- *Type:* string

Projen target i.e: build, test, etc.

---

### VersionGroup <a name="VersionGroup" id="@aws-prototyping-sdk/nx-monorepo.VersionGroup"></a>

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.VersionGroup.Initializer"></a>

```typescript
import { VersionGroup } from '@aws-prototyping-sdk/nx-monorepo'

const versionGroup: VersionGroup = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.dependencies">dependencies</a></code> | <code>string[]</code> | the names of the dependencies (eg. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.packages">packages</a></code> | <code>string[]</code> | the names of packages in your monorepo which belong to this group, taken from the "name" field in package.json, not the package directory name. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.dependencyTypes">dependencyTypes</a></code> | <code>string[]</code> | optionally only apply this group to dependencies of the provided types. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.isBanned">isBanned</a></code> | <code>boolean</code> | optionally force all dependencies in this group to be removed. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.pinVersion">pinVersion</a></code> | <code>string</code> | optionally force all dependencies in this group to have this version. |

---

##### `dependencies`<sup>Required</sup> <a name="dependencies" id="@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.dependencies"></a>

```typescript
public readonly dependencies: string[];
```

- *Type:* string[]

the names of the dependencies (eg.

"lodash") which belong to this group

---

##### `packages`<sup>Required</sup> <a name="packages" id="@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.packages"></a>

```typescript
public readonly packages: string[];
```

- *Type:* string[]

the names of packages in your monorepo which belong to this group, taken from the "name" field in package.json, not the package directory name.

---

##### `dependencyTypes`<sup>Optional</sup> <a name="dependencyTypes" id="@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.dependencyTypes"></a>

```typescript
public readonly dependencyTypes: string[];
```

- *Type:* string[]

optionally only apply this group to dependencies of the provided types.

---

##### `isBanned`<sup>Optional</sup> <a name="isBanned" id="@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.isBanned"></a>

```typescript
public readonly isBanned: boolean;
```

- *Type:* boolean

optionally force all dependencies in this group to be removed.

---

##### `pinVersion`<sup>Optional</sup> <a name="pinVersion" id="@aws-prototyping-sdk/nx-monorepo.VersionGroup.property.pinVersion"></a>

```typescript
public readonly pinVersion: string;
```

- *Type:* string

optionally force all dependencies in this group to have this version.

---

### WorkspaceConfig <a name="WorkspaceConfig" id="@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig"></a>

Workspace configurations.

> [https://classic.yarnpkg.com/lang/en/docs/workspaces/](https://classic.yarnpkg.com/lang/en/docs/workspaces/)

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.Initializer"></a>

```typescript
import { WorkspaceConfig } from '@aws-prototyping-sdk/nx-monorepo'

const workspaceConfig: WorkspaceConfig = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.additionalPackages">additionalPackages</a></code> | <code>string[]</code> | List of additional package globs to include in the workspace. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.disableNoHoistBundled">disableNoHoistBundled</a></code> | <code>boolean</code> | Disable automatically applying `noHoist` logic for all sub-project "bundledDependencies". |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.linkLocalWorkspaceBins">linkLocalWorkspaceBins</a></code> | <code>boolean</code> | Links all local workspace project bins so they can be used for local development. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.noHoist">noHoist</a></code> | <code>string[]</code> | List of package globs to exclude from hoisting in the workspace. |

---

##### `additionalPackages`<sup>Optional</sup> <a name="additionalPackages" id="@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.additionalPackages"></a>

```typescript
public readonly additionalPackages: string[];
```

- *Type:* string[]

List of additional package globs to include in the workspace.

All packages which are parented by the monorepo are automatically added to the workspace, but you can use this
property to specify any additional paths to packages which may not be managed by projen.

---

##### `disableNoHoistBundled`<sup>Optional</sup> <a name="disableNoHoistBundled" id="@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.disableNoHoistBundled"></a>

```typescript
public readonly disableNoHoistBundled: boolean;
```

- *Type:* boolean
- *Default:* false

Disable automatically applying `noHoist` logic for all sub-project "bundledDependencies".

---

##### `linkLocalWorkspaceBins`<sup>Optional</sup> <a name="linkLocalWorkspaceBins" id="@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.linkLocalWorkspaceBins"></a>

```typescript
public readonly linkLocalWorkspaceBins: boolean;
```

- *Type:* boolean

Links all local workspace project bins so they can be used for local development.

Package bins are only linked when installed from the registry, however it is very useful
for monorepo development to also utilize these bin scripts. When enabled, this flag will
recursively link all bins from packages.json files to the root node_modules/.bin.

---

##### `noHoist`<sup>Optional</sup> <a name="noHoist" id="@aws-prototyping-sdk/nx-monorepo.WorkspaceConfig.property.noHoist"></a>

```typescript
public readonly noHoist: string[];
```

- *Type:* string[]

List of package globs to exclude from hoisting in the workspace.

> [https://classic.yarnpkg.com/blog/2018/02/15/nohoist/](https://classic.yarnpkg.com/blog/2018/02/15/nohoist/)

---

### WorkspaceConfig <a name="WorkspaceConfig" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig"></a>

NX workspace configurations.

> [https://nx.dev/configuration/packagejson](https://nx.dev/configuration/packagejson)

#### Initializer <a name="Initializer" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.Initializer"></a>

```typescript
import { Nx } from '@aws-prototyping-sdk/nx-monorepo'

const workspaceConfig: Nx.WorkspaceConfig = { ... }
```

#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.affectedBranch">affectedBranch</a></code> | <code>string</code> | Affected branch. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.cacheableOperations">cacheableOperations</a></code> | <code>string[]</code> | Defines the list of targets/operations that are cached by Nx. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.namedInputs">namedInputs</a></code> | <code>@aws-prototyping-sdk/nx-monorepo.Nx.NamedInputs</code> | Named inputs. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.nxCloudReadOnlyAccessToken">nxCloudReadOnlyAccessToken</a></code> | <code>string</code> | Read only access token if enabling nx cloud. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.nxIgnore">nxIgnore</a></code> | <code>string[]</code> | List of patterns to include in the .nxignore file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.targetDefaults">targetDefaults</a></code> | <code>@aws-prototyping-sdk/nx-monorepo.Nx.TargetDefaults</code> | Target defaults. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.targetDependencies">targetDependencies</a></code> | <code>{[ key: string ]: @aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency[]}</code> | Configuration for TargetDependencies. |

---

##### `affectedBranch`<sup>Optional</sup> <a name="affectedBranch" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.affectedBranch"></a>

```typescript
public readonly affectedBranch: string;
```

- *Type:* string
- *Default:* mainline

Affected branch.

---

##### `cacheableOperations`<sup>Optional</sup> <a name="cacheableOperations" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.cacheableOperations"></a>

```typescript
public readonly cacheableOperations: string[];
```

- *Type:* string[]
- *Default:* ["build", "test"]

Defines the list of targets/operations that are cached by Nx.

> [https://nx.dev/reference/nx-json#tasks-runner-options](https://nx.dev/reference/nx-json#tasks-runner-options)

---

##### `namedInputs`<sup>Optional</sup> <a name="namedInputs" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.namedInputs"></a>

```typescript
public readonly namedInputs: NamedInputs;
```

- *Type:* @aws-prototyping-sdk/nx-monorepo.Nx.NamedInputs

Named inputs.

> [https://nx.dev/reference/nx-json#inputs-&-namedinputs](https://nx.dev/reference/nx-json#inputs-&-namedinputs)

---

##### `nxCloudReadOnlyAccessToken`<sup>Optional</sup> <a name="nxCloudReadOnlyAccessToken" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.nxCloudReadOnlyAccessToken"></a>

```typescript
public readonly nxCloudReadOnlyAccessToken: string;
```

- *Type:* string

Read only access token if enabling nx cloud.

---

##### `nxIgnore`<sup>Optional</sup> <a name="nxIgnore" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.nxIgnore"></a>

```typescript
public readonly nxIgnore: string[];
```

- *Type:* string[]

List of patterns to include in the .nxignore file.

> [https://nx.dev/configuration/packagejson#nxignore](https://nx.dev/configuration/packagejson#nxignore)

---

##### `targetDefaults`<sup>Optional</sup> <a name="targetDefaults" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.targetDefaults"></a>

```typescript
public readonly targetDefaults: TargetDefaults;
```

- *Type:* @aws-prototyping-sdk/nx-monorepo.Nx.TargetDefaults

Target defaults.

> [https://nx.dev/reference/nx-json#target-defaults](https://nx.dev/reference/nx-json#target-defaults)

---

##### `targetDependencies`<sup>Optional</sup> <a name="targetDependencies" id="@aws-prototyping-sdk/nx-monorepo.Nx.WorkspaceConfig.property.targetDependencies"></a>

```typescript
public readonly targetDependencies: {[ key: string ]: TargetDependency[]};
```

- *Type:* {[ key: string ]: @aws-prototyping-sdk/nx-monorepo.Nx.TargetDependency[]}

Configuration for TargetDependencies.

> [https://nx.dev/configuration/packagejson#target-dependencies](https://nx.dev/configuration/packagejson#target-dependencies)

---

## Classes <a name="Classes" id="Classes"></a>

### NxMonorepoProject <a name="NxMonorepoProject" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject"></a>

This project type will bootstrap a NX based monorepo with support for polygot builds, build caching, dependency graph visualization and much more.

#### Initializers <a name="Initializers" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.Initializer"></a>

```typescript
import { NxMonorepoProject } from '@aws-prototyping-sdk/nx-monorepo'

new NxMonorepoProject(options: NxMonorepoProjectOptions)
```

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.Initializer.parameter.options">options</a></code> | <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions">NxMonorepoProjectOptions</a></code> | *No description.* |

---

##### `options`<sup>Required</sup> <a name="options" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.Initializer.parameter.options"></a>

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProjectOptions">NxMonorepoProjectOptions</a>

---

#### Methods <a name="Methods" id="Methods"></a>

| **Name** | **Description** |
| --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addExcludeFromCleanup">addExcludeFromCleanup</a></code> | Exclude the matching files from pre-synth cleanup. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addGitIgnore">addGitIgnore</a></code> | Adds a .gitignore pattern. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPackageIgnore">addPackageIgnore</a></code> | Exclude these files from the bundled package. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTask">addTask</a></code> | Adds a new task to this project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTip">addTip</a></code> | Prints a "tip" message during synthesis. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.annotateGenerated">annotateGenerated</a></code> | Marks the provided file(s) as being generated. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.postSynthesize">postSynthesize</a></code> | Called after all components are synthesized. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.preSynthesize">preSynthesize</a></code> | Called before all components are synthesized. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.removeTask">removeTask</a></code> | Removes a task from a project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.runTaskCommand">runTaskCommand</a></code> | Returns the shell command to execute in order to run a task. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.synth">synth</a></code> | Synthesize all project files into `outdir`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindFile">tryFindFile</a></code> | Finds a file at the specified relative path within this project and all its subprojects. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindJsonFile">tryFindJsonFile</a></code> | Finds a json file by name. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindObjectFile">tryFindObjectFile</a></code> | Finds an object file (like JsonFile, YamlFile, etc.) by name. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryRemoveFile">tryRemoveFile</a></code> | Finds a file at the specified relative path within this project and removes it. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addBins">addBins</a></code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addBundledDeps">addBundledDeps</a></code> | Defines bundled dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addCompileCommand">addCompileCommand</a></code> | DEPRECATED. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addDeps">addDeps</a></code> | Defines normal dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addDevDeps">addDevDeps</a></code> | Defines development/test dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addFields">addFields</a></code> | Directly set fields in `package.json`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addKeywords">addKeywords</a></code> | Adds keywords to package.json (deduplicated). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPeerDeps">addPeerDeps</a></code> | Defines peer dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addScripts">addScripts</a></code> | Replaces the contents of multiple npm package.json scripts. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTestCommand">addTestCommand</a></code> | DEPRECATED. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.hasScript">hasScript</a></code> | Indicates if a script by the name name is defined. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.removeScript">removeScript</a></code> | Removes the npm script (always successful). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.renderWorkflowSetup">renderWorkflowSetup</a></code> | Returns the set of workflow steps which should be executed to bootstrap a workflow. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.setScript">setScript</a></code> | Replaces the contents of an npm package.json script. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addImplicitDependency">addImplicitDependency</a></code> | Create an implicit dependency between two Projects. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addJavaDependency">addJavaDependency</a></code> | Adds a dependency between two Java Projects in the monorepo. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addNxRunManyTask">addNxRunManyTask</a></code> | Add project task that executes `npx nx run-many ...` style command. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPythonPoetryDependency">addPythonPoetryDependency</a></code> | Adds a dependency between two Python Projects in the monorepo. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addWorkspacePackages">addWorkspacePackages</a></code> | Add one or more additional package globs to the workspace. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.formatNxRunManyCommand">formatNxRunManyCommand</a></code> | Helper to format `npx nx run-many ...` style command. |

---

##### `addExcludeFromCleanup` <a name="addExcludeFromCleanup" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addExcludeFromCleanup"></a>

```typescript
public addExcludeFromCleanup(globs: string): void
```

Exclude the matching files from pre-synth cleanup.

Can be used when, for example, some
source files include the projen marker and we don't want them to be erased during synth.

###### `globs`<sup>Required</sup> <a name="globs" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addExcludeFromCleanup.parameter.globs"></a>

- *Type:* string

The glob patterns to match.

---

##### `addGitIgnore` <a name="addGitIgnore" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addGitIgnore"></a>

```typescript
public addGitIgnore(pattern: string): void
```

Adds a .gitignore pattern.

###### `pattern`<sup>Required</sup> <a name="pattern" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addGitIgnore.parameter.pattern"></a>

- *Type:* string

The glob pattern to ignore.

---

##### `addPackageIgnore` <a name="addPackageIgnore" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPackageIgnore"></a>

```typescript
public addPackageIgnore(pattern: string): void
```

Exclude these files from the bundled package.

Implemented by project types based on the
packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.

###### `pattern`<sup>Required</sup> <a name="pattern" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPackageIgnore.parameter.pattern"></a>

- *Type:* string

---

##### `addTask` <a name="addTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTask"></a>

```typescript
public addTask(name: string, props?: TaskOptions): Task
```

Adds a new task to this project.

This will fail if the project already has
a task with this name.

###### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTask.parameter.name"></a>

- *Type:* string

The task name to add.

---

###### `props`<sup>Optional</sup> <a name="props" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTask.parameter.props"></a>

- *Type:* projen.TaskOptions

Task properties.

---

##### ~~`addTip`~~ <a name="addTip" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTip"></a>

```typescript
public addTip(message: string): void
```

Prints a "tip" message during synthesis.

###### `message`<sup>Required</sup> <a name="message" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTip.parameter.message"></a>

- *Type:* string

The message.

---

##### `annotateGenerated` <a name="annotateGenerated" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.annotateGenerated"></a>

```typescript
public annotateGenerated(glob: string): void
```

Marks the provided file(s) as being generated.

This is achieved using the
github-linguist attributes. Generated files do not count against the
repository statistics and language breakdown.

> [https://github.com/github/linguist/blob/master/docs/overrides.md](https://github.com/github/linguist/blob/master/docs/overrides.md)

###### `glob`<sup>Required</sup> <a name="glob" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.annotateGenerated.parameter.glob"></a>

- *Type:* string

the glob pattern to match (could be a file path).

---

##### `postSynthesize` <a name="postSynthesize" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.postSynthesize"></a>

```typescript
public postSynthesize(): void
```

Called after all components are synthesized.

Order is *not* guaranteed.

##### `preSynthesize` <a name="preSynthesize" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.preSynthesize"></a>

```typescript
public preSynthesize(): void
```

Called before all components are synthesized.

##### `removeTask` <a name="removeTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.removeTask"></a>

```typescript
public removeTask(name: string): Task
```

Removes a task from a project.

###### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.removeTask.parameter.name"></a>

- *Type:* string

The name of the task to remove.

---

##### `runTaskCommand` <a name="runTaskCommand" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.runTaskCommand"></a>

```typescript
public runTaskCommand(task: Task): string
```

Returns the shell command to execute in order to run a task.

This will
typically be `npx projen TASK`.

###### `task`<sup>Required</sup> <a name="task" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.runTaskCommand.parameter.task"></a>

- *Type:* projen.Task

The task for which the command is required.

---

##### `synth` <a name="synth" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.synth"></a>

```typescript
public synth(): void
```

Synthesize all project files into `outdir`.

1. Call "this.preSynthesize()"
2. Delete all generated files
3. Synthesize all sub-projects
4. Synthesize all components of this project
5. Call "postSynthesize()" for all components of this project
6. Call "this.postSynthesize()"

##### `tryFindFile` <a name="tryFindFile" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindFile"></a>

```typescript
public tryFindFile(filePath: string): FileBase
```

Finds a file at the specified relative path within this project and all its subprojects.

###### `filePath`<sup>Required</sup> <a name="filePath" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindFile.parameter.filePath"></a>

- *Type:* string

The file path.

If this path is relative, it will be resolved
from the root of _this_ project.

---

##### ~~`tryFindJsonFile`~~ <a name="tryFindJsonFile" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindJsonFile"></a>

```typescript
public tryFindJsonFile(filePath: string): JsonFile
```

Finds a json file by name.

###### `filePath`<sup>Required</sup> <a name="filePath" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindJsonFile.parameter.filePath"></a>

- *Type:* string

The file path.

---

##### `tryFindObjectFile` <a name="tryFindObjectFile" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindObjectFile"></a>

```typescript
public tryFindObjectFile(filePath: string): ObjectFile
```

Finds an object file (like JsonFile, YamlFile, etc.) by name.

###### `filePath`<sup>Required</sup> <a name="filePath" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryFindObjectFile.parameter.filePath"></a>

- *Type:* string

The file path.

---

##### `tryRemoveFile` <a name="tryRemoveFile" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryRemoveFile"></a>

```typescript
public tryRemoveFile(filePath: string): FileBase
```

Finds a file at the specified relative path within this project and removes it.

###### `filePath`<sup>Required</sup> <a name="filePath" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.tryRemoveFile.parameter.filePath"></a>

- *Type:* string

The file path.

If this path is relative, it will be
resolved from the root of _this_ project.

---

##### `addBins` <a name="addBins" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addBins"></a>

```typescript
public addBins(bins: {[ key: string ]: string}): void
```

###### `bins`<sup>Required</sup> <a name="bins" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addBins.parameter.bins"></a>

- *Type:* {[ key: string ]: string}

---

##### `addBundledDeps` <a name="addBundledDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addBundledDeps"></a>

```typescript
public addBundledDeps(deps: string): void
```

Defines bundled dependencies.

Bundled dependencies will be added as normal dependencies as well as to the
`bundledDependencies` section of your `package.json`.

###### `deps`<sup>Required</sup> <a name="deps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addBundledDeps.parameter.deps"></a>

- *Type:* string

Names modules to install.

By default, the the dependency will
be installed in the next `npx projen` run and the version will be recorded
in your `package.json` file. You can upgrade manually or using `yarn
add/upgrade`. If you wish to specify a version range use this syntax:
`module@^7`.

---

##### ~~`addCompileCommand`~~ <a name="addCompileCommand" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addCompileCommand"></a>

```typescript
public addCompileCommand(commands: string): void
```

DEPRECATED.

###### `commands`<sup>Required</sup> <a name="commands" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addCompileCommand.parameter.commands"></a>

- *Type:* string

---

##### `addDeps` <a name="addDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addDeps"></a>

```typescript
public addDeps(deps: string): void
```

Defines normal dependencies.

###### `deps`<sup>Required</sup> <a name="deps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addDeps.parameter.deps"></a>

- *Type:* string

Names modules to install.

By default, the the dependency will
be installed in the next `npx projen` run and the version will be recorded
in your `package.json` file. You can upgrade manually or using `yarn
add/upgrade`. If you wish to specify a version range use this syntax:
`module@^7`.

---

##### `addDevDeps` <a name="addDevDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addDevDeps"></a>

```typescript
public addDevDeps(deps: string): void
```

Defines development/test dependencies.

###### `deps`<sup>Required</sup> <a name="deps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addDevDeps.parameter.deps"></a>

- *Type:* string

Names modules to install.

By default, the the dependency will
be installed in the next `npx projen` run and the version will be recorded
in your `package.json` file. You can upgrade manually or using `yarn
add/upgrade`. If you wish to specify a version range use this syntax:
`module@^7`.

---

##### `addFields` <a name="addFields" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addFields"></a>

```typescript
public addFields(fields: {[ key: string ]: any}): void
```

Directly set fields in `package.json`.

###### `fields`<sup>Required</sup> <a name="fields" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addFields.parameter.fields"></a>

- *Type:* {[ key: string ]: any}

The fields to set.

---

##### `addKeywords` <a name="addKeywords" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addKeywords"></a>

```typescript
public addKeywords(keywords: string): void
```

Adds keywords to package.json (deduplicated).

###### `keywords`<sup>Required</sup> <a name="keywords" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addKeywords.parameter.keywords"></a>

- *Type:* string

The keywords to add.

---

##### `addPeerDeps` <a name="addPeerDeps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPeerDeps"></a>

```typescript
public addPeerDeps(deps: string): void
```

Defines peer dependencies.

When adding peer dependencies, a devDependency will also be added on the
pinned version of the declared peer. This will ensure that you are testing
your code against the minimum version required from your consumers.

###### `deps`<sup>Required</sup> <a name="deps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPeerDeps.parameter.deps"></a>

- *Type:* string

Names modules to install.

By default, the the dependency will
be installed in the next `npx projen` run and the version will be recorded
in your `package.json` file. You can upgrade manually or using `yarn
add/upgrade`. If you wish to specify a version range use this syntax:
`module@^7`.

---

##### `addScripts` <a name="addScripts" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addScripts"></a>

```typescript
public addScripts(scripts: {[ key: string ]: string}): void
```

Replaces the contents of multiple npm package.json scripts.

###### `scripts`<sup>Required</sup> <a name="scripts" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addScripts.parameter.scripts"></a>

- *Type:* {[ key: string ]: string}

The scripts to set.

---

##### ~~`addTestCommand`~~ <a name="addTestCommand" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTestCommand"></a>

```typescript
public addTestCommand(commands: string): void
```

DEPRECATED.

###### `commands`<sup>Required</sup> <a name="commands" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addTestCommand.parameter.commands"></a>

- *Type:* string

---

##### ~~`hasScript`~~ <a name="hasScript" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.hasScript"></a>

```typescript
public hasScript(name: string): boolean
```

Indicates if a script by the name name is defined.

###### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.hasScript.parameter.name"></a>

- *Type:* string

The name of the script.

---

##### `removeScript` <a name="removeScript" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.removeScript"></a>

```typescript
public removeScript(name: string): void
```

Removes the npm script (always successful).

###### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.removeScript.parameter.name"></a>

- *Type:* string

The name of the script.

---

##### `renderWorkflowSetup` <a name="renderWorkflowSetup" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.renderWorkflowSetup"></a>

```typescript
public renderWorkflowSetup(options?: RenderWorkflowSetupOptions): JobStep[]
```

Returns the set of workflow steps which should be executed to bootstrap a workflow.

###### `options`<sup>Optional</sup> <a name="options" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.renderWorkflowSetup.parameter.options"></a>

- *Type:* projen.javascript.RenderWorkflowSetupOptions

Options.

---

##### `setScript` <a name="setScript" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.setScript"></a>

```typescript
public setScript(name: string, command: string): void
```

Replaces the contents of an npm package.json script.

###### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.setScript.parameter.name"></a>

- *Type:* string

The script name.

---

###### `command`<sup>Required</sup> <a name="command" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.setScript.parameter.command"></a>

- *Type:* string

The command to execute.

---

##### `addImplicitDependency` <a name="addImplicitDependency" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addImplicitDependency"></a>

```typescript
public addImplicitDependency(dependent: Project, dependee: string | Project): void
```

Create an implicit dependency between two Projects.

This is typically
used in polygot repos where a Typescript project wants a build dependency
on a Python project as an example.

###### `dependent`<sup>Required</sup> <a name="dependent" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addImplicitDependency.parameter.dependent"></a>

- *Type:* projen.Project

project you want to have the dependency.

---

###### `dependee`<sup>Required</sup> <a name="dependee" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addImplicitDependency.parameter.dependee"></a>

- *Type:* string | projen.Project

project you wish to depend on.

---

##### `addJavaDependency` <a name="addJavaDependency" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addJavaDependency"></a>

```typescript
public addJavaDependency(dependent: JavaProject, dependee: JavaProject): void
```

Adds a dependency between two Java Projects in the monorepo.

###### `dependent`<sup>Required</sup> <a name="dependent" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addJavaDependency.parameter.dependent"></a>

- *Type:* projen.java.JavaProject

project you want to have the dependency.

---

###### `dependee`<sup>Required</sup> <a name="dependee" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addJavaDependency.parameter.dependee"></a>

- *Type:* projen.java.JavaProject

project you wish to depend on.

---

##### `addNxRunManyTask` <a name="addNxRunManyTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addNxRunManyTask"></a>

```typescript
public addNxRunManyTask(name: string, options: NxRunManyOptions): Task
```

Add project task that executes `npx nx run-many ...` style command.

###### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addNxRunManyTask.parameter.name"></a>

- *Type:* string

---

###### `options`<sup>Required</sup> <a name="options" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addNxRunManyTask.parameter.options"></a>

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions">NxRunManyOptions</a>

---

##### `addPythonPoetryDependency` <a name="addPythonPoetryDependency" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPythonPoetryDependency"></a>

```typescript
public addPythonPoetryDependency(dependent: PythonProject, dependee: PythonProject): void
```

Adds a dependency between two Python Projects in the monorepo.

The dependent must have Poetry enabled.

###### `dependent`<sup>Required</sup> <a name="dependent" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPythonPoetryDependency.parameter.dependent"></a>

- *Type:* projen.python.PythonProject

project you want to have the dependency (must be a Poetry Python Project).

---

###### `dependee`<sup>Required</sup> <a name="dependee" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addPythonPoetryDependency.parameter.dependee"></a>

- *Type:* projen.python.PythonProject

project you wish to depend on.

---

##### `addWorkspacePackages` <a name="addWorkspacePackages" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addWorkspacePackages"></a>

```typescript
public addWorkspacePackages(packageGlobs: string): void
```

Add one or more additional package globs to the workspace.

###### `packageGlobs`<sup>Required</sup> <a name="packageGlobs" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.addWorkspacePackages.parameter.packageGlobs"></a>

- *Type:* string

paths to the package to include in the workspace (for example packages/my-package).

---

##### `formatNxRunManyCommand` <a name="formatNxRunManyCommand" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.formatNxRunManyCommand"></a>

```typescript
public formatNxRunManyCommand(options: NxRunManyOptions): string
```

Helper to format `npx nx run-many ...` style command.

###### `options`<sup>Required</sup> <a name="options" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.formatNxRunManyCommand.parameter.options"></a>

- *Type:* <a href="#@aws-prototyping-sdk/nx-monorepo.NxRunManyOptions">NxRunManyOptions</a>

---


#### Properties <a name="Properties" id="Properties"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.buildTask">buildTask</a></code> | <code>projen.Task</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.commitGenerated">commitGenerated</a></code> | <code>boolean</code> | Whether to commit the managed files by default. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.compileTask">compileTask</a></code> | <code>projen.Task</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.components">components</a></code> | <code>projen.Component[]</code> | Returns all the components within this project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.deps">deps</a></code> | <code>projen.Dependencies</code> | Project dependencies. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.ejected">ejected</a></code> | <code>boolean</code> | Whether or not the project is being ejected. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.files">files</a></code> | <code>projen.FileBase[]</code> | All files in this project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.gitattributes">gitattributes</a></code> | <code>projen.GitAttributesFile</code> | The .gitattributes file for this repository. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.gitignore">gitignore</a></code> | <code>projen.IgnoreFile</code> | .gitignore. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.logger">logger</a></code> | <code>projen.Logger</code> | Logging utilities. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.name">name</a></code> | <code>string</code> | Project name. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.outdir">outdir</a></code> | <code>string</code> | Absolute output directory of this project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.packageTask">packageTask</a></code> | <code>projen.Task</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.postCompileTask">postCompileTask</a></code> | <code>projen.Task</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.preCompileTask">preCompileTask</a></code> | <code>projen.Task</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.projectBuild">projectBuild</a></code> | <code>projen.ProjectBuild</code> | Manages the build process of the project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.projenCommand">projenCommand</a></code> | <code>string</code> | The command to use in order to run the projen CLI. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.root">root</a></code> | <code>projen.Project</code> | The root project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tasks">tasks</a></code> | <code>projen.Tasks</code> | Project tasks. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.testTask">testTask</a></code> | <code>projen.Task</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.defaultTask">defaultTask</a></code> | <code>projen.Task</code> | This is the "default" task, the one that executes "projen". |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.initProject">initProject</a></code> | <code>projen.InitProject</code> | The options used when this project is bootstrapped via `projen new`. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.parent">parent</a></code> | <code>projen.Project</code> | A parent project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.projectType">projectType</a></code> | <code>projen.ProjectType</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.autoApprove">autoApprove</a></code> | <code>projen.github.AutoApprove</code> | Auto approve set up for this project. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.devContainer">devContainer</a></code> | <code>projen.vscode.DevContainer</code> | Access for .devcontainer.json (used for GitHub Codespaces). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.github">github</a></code> | <code>projen.github.GitHub</code> | Access all github components. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.gitpod">gitpod</a></code> | <code>projen.Gitpod</code> | Access for Gitpod. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.vscode">vscode</a></code> | <code>projen.vscode.VsCode</code> | Access all VSCode components. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.allowLibraryDependencies">allowLibraryDependencies</a></code> | <code>boolean</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.artifactsDirectory">artifactsDirectory</a></code> | <code>string</code> | The build output directory. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.artifactsJavascriptDirectory">artifactsJavascriptDirectory</a></code> | <code>string</code> | The location of the npm tarball after build (`${artifactsDirectory}/js`). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.bundler">bundler</a></code> | <code>projen.javascript.Bundler</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.entrypoint">entrypoint</a></code> | <code>string</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.manifest">manifest</a></code> | <code>any</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.npmrc">npmrc</a></code> | <code>projen.javascript.NpmConfig</code> | The .npmrc file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.package">package</a></code> | <code>projen.javascript.NodePackage</code> | API for managing the node package. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.packageManager">packageManager</a></code> | <code>projen.javascript.NodePackageManager</code> | The package manager to use. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.runScriptCommand">runScriptCommand</a></code> | <code>string</code> | The command to use to run scripts (e.g. `yarn run` or `npm run` depends on the package manager). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.autoMerge">autoMerge</a></code> | <code>projen.github.AutoMerge</code> | Component that sets up mergify for merging approved pull requests. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.buildWorkflow">buildWorkflow</a></code> | <code>projen.build.BuildWorkflow</code> | The PR build GitHub workflow. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.buildWorkflowJobId">buildWorkflowJobId</a></code> | <code>string</code> | The job ID of the build workflow. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.jest">jest</a></code> | <code>projen.javascript.Jest</code> | The Jest configuration (if enabled). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.maxNodeVersion">maxNodeVersion</a></code> | <code>string</code> | Maximum node version required by this package. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.minNodeVersion">minNodeVersion</a></code> | <code>string</code> | Minimum node.js version required by this package. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.npmignore">npmignore</a></code> | <code>projen.IgnoreFile</code> | The .npmignore file. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.prettier">prettier</a></code> | <code>projen.javascript.Prettier</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.publisher">publisher</a></code> | <code>projen.release.Publisher</code> | Package publisher. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.release">release</a></code> | <code>projen.release.Release</code> | Release management. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.upgradeWorkflow">upgradeWorkflow</a></code> | <code>projen.javascript.UpgradeDependencies</code> | The upgrade workflow. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.docsDirectory">docsDirectory</a></code> | <code>string</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.libdir">libdir</a></code> | <code>string</code> | The directory in which compiled .js files reside. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.srcdir">srcdir</a></code> | <code>string</code> | The directory in which the .ts sources reside. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.testdir">testdir</a></code> | <code>string</code> | The directory in which tests reside. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tsconfigDev">tsconfigDev</a></code> | <code>projen.javascript.TypescriptConfig</code> | A typescript configuration file which covers all files (sources, tests, projen). |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.watchTask">watchTask</a></code> | <code>projen.Task</code> | The "watch" task. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.docgen">docgen</a></code> | <code>boolean</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.eslint">eslint</a></code> | <code>projen.javascript.Eslint</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tsconfig">tsconfig</a></code> | <code>projen.javascript.TypescriptConfig</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tsconfigEslint">tsconfigEslint</a></code> | <code>projen.javascript.TypescriptConfig</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.nxJson">nxJson</a></code> | <code>projen.JsonFile</code> | *No description.* |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.subProjects">subProjects</a></code> | <code>projen.Project[]</code> | *No description.* |

---

##### `buildTask`<sup>Required</sup> <a name="buildTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.buildTask"></a>

```typescript
public readonly buildTask: Task;
```

- *Type:* projen.Task

---

##### `commitGenerated`<sup>Required</sup> <a name="commitGenerated" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.commitGenerated"></a>

```typescript
public readonly commitGenerated: boolean;
```

- *Type:* boolean

Whether to commit the managed files by default.

---

##### `compileTask`<sup>Required</sup> <a name="compileTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.compileTask"></a>

```typescript
public readonly compileTask: Task;
```

- *Type:* projen.Task

---

##### `components`<sup>Required</sup> <a name="components" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.components"></a>

```typescript
public readonly components: Component[];
```

- *Type:* projen.Component[]

Returns all the components within this project.

---

##### `deps`<sup>Required</sup> <a name="deps" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.deps"></a>

```typescript
public readonly deps: Dependencies;
```

- *Type:* projen.Dependencies

Project dependencies.

---

##### `ejected`<sup>Required</sup> <a name="ejected" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.ejected"></a>

```typescript
public readonly ejected: boolean;
```

- *Type:* boolean

Whether or not the project is being ejected.

---

##### `files`<sup>Required</sup> <a name="files" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.files"></a>

```typescript
public readonly files: FileBase[];
```

- *Type:* projen.FileBase[]

All files in this project.

---

##### `gitattributes`<sup>Required</sup> <a name="gitattributes" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.gitattributes"></a>

```typescript
public readonly gitattributes: GitAttributesFile;
```

- *Type:* projen.GitAttributesFile

The .gitattributes file for this repository.

---

##### `gitignore`<sup>Required</sup> <a name="gitignore" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.gitignore"></a>

```typescript
public readonly gitignore: IgnoreFile;
```

- *Type:* projen.IgnoreFile

.gitignore.

---

##### `logger`<sup>Required</sup> <a name="logger" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.logger"></a>

```typescript
public readonly logger: Logger;
```

- *Type:* projen.Logger

Logging utilities.

---

##### `name`<sup>Required</sup> <a name="name" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.name"></a>

```typescript
public readonly name: string;
```

- *Type:* string

Project name.

---

##### `outdir`<sup>Required</sup> <a name="outdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.outdir"></a>

```typescript
public readonly outdir: string;
```

- *Type:* string

Absolute output directory of this project.

---

##### `packageTask`<sup>Required</sup> <a name="packageTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.packageTask"></a>

```typescript
public readonly packageTask: Task;
```

- *Type:* projen.Task

---

##### `postCompileTask`<sup>Required</sup> <a name="postCompileTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.postCompileTask"></a>

```typescript
public readonly postCompileTask: Task;
```

- *Type:* projen.Task

---

##### `preCompileTask`<sup>Required</sup> <a name="preCompileTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.preCompileTask"></a>

```typescript
public readonly preCompileTask: Task;
```

- *Type:* projen.Task

---

##### `projectBuild`<sup>Required</sup> <a name="projectBuild" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.projectBuild"></a>

```typescript
public readonly projectBuild: ProjectBuild;
```

- *Type:* projen.ProjectBuild

Manages the build process of the project.

---

##### `projenCommand`<sup>Required</sup> <a name="projenCommand" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.projenCommand"></a>

```typescript
public readonly projenCommand: string;
```

- *Type:* string

The command to use in order to run the projen CLI.

---

##### `root`<sup>Required</sup> <a name="root" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.root"></a>

```typescript
public readonly root: Project;
```

- *Type:* projen.Project

The root project.

---

##### `tasks`<sup>Required</sup> <a name="tasks" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tasks"></a>

```typescript
public readonly tasks: Tasks;
```

- *Type:* projen.Tasks

Project tasks.

---

##### `testTask`<sup>Required</sup> <a name="testTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.testTask"></a>

```typescript
public readonly testTask: Task;
```

- *Type:* projen.Task

---

##### `defaultTask`<sup>Optional</sup> <a name="defaultTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.defaultTask"></a>

```typescript
public readonly defaultTask: Task;
```

- *Type:* projen.Task

This is the "default" task, the one that executes "projen".

Undefined if
the project is being ejected.

---

##### `initProject`<sup>Optional</sup> <a name="initProject" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.initProject"></a>

```typescript
public readonly initProject: InitProject;
```

- *Type:* projen.InitProject

The options used when this project is bootstrapped via `projen new`.

It
includes the original set of options passed to the CLI and also the JSII
FQN of the project type.

---

##### `parent`<sup>Optional</sup> <a name="parent" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.parent"></a>

```typescript
public readonly parent: Project;
```

- *Type:* projen.Project

A parent project.

If undefined, this is the root project.

---

##### `projectType`<sup>Required</sup> <a name="projectType" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.projectType"></a>

```typescript
public readonly projectType: ProjectType;
```

- *Type:* projen.ProjectType

---

##### `autoApprove`<sup>Optional</sup> <a name="autoApprove" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.autoApprove"></a>

```typescript
public readonly autoApprove: AutoApprove;
```

- *Type:* projen.github.AutoApprove

Auto approve set up for this project.

---

##### `devContainer`<sup>Optional</sup> <a name="devContainer" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.devContainer"></a>

```typescript
public readonly devContainer: DevContainer;
```

- *Type:* projen.vscode.DevContainer

Access for .devcontainer.json (used for GitHub Codespaces).

This will be `undefined` if devContainer boolean is false

---

##### `github`<sup>Optional</sup> <a name="github" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.github"></a>

```typescript
public readonly github: GitHub;
```

- *Type:* projen.github.GitHub

Access all github components.

This will be `undefined` for subprojects.

---

##### `gitpod`<sup>Optional</sup> <a name="gitpod" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.gitpod"></a>

```typescript
public readonly gitpod: Gitpod;
```

- *Type:* projen.Gitpod

Access for Gitpod.

This will be `undefined` if gitpod boolean is false

---

##### `vscode`<sup>Optional</sup> <a name="vscode" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.vscode"></a>

```typescript
public readonly vscode: VsCode;
```

- *Type:* projen.vscode.VsCode

Access all VSCode components.

This will be `undefined` for subprojects.

---

##### ~~`allowLibraryDependencies`~~<sup>Required</sup> <a name="allowLibraryDependencies" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.allowLibraryDependencies"></a>

- *Deprecated:* use `package.allowLibraryDependencies`

```typescript
public readonly allowLibraryDependencies: boolean;
```

- *Type:* boolean

---

##### `artifactsDirectory`<sup>Required</sup> <a name="artifactsDirectory" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.artifactsDirectory"></a>

```typescript
public readonly artifactsDirectory: string;
```

- *Type:* string

The build output directory.

An npm tarball will be created under the `js`
subdirectory. For example, if this is set to `dist` (the default), the npm
tarball will be placed under `dist/js/boom-boom-1.2.3.tg`.

---

##### `artifactsJavascriptDirectory`<sup>Required</sup> <a name="artifactsJavascriptDirectory" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.artifactsJavascriptDirectory"></a>

```typescript
public readonly artifactsJavascriptDirectory: string;
```

- *Type:* string

The location of the npm tarball after build (`${artifactsDirectory}/js`).

---

##### `bundler`<sup>Required</sup> <a name="bundler" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.bundler"></a>

```typescript
public readonly bundler: Bundler;
```

- *Type:* projen.javascript.Bundler

---

##### ~~`entrypoint`~~<sup>Required</sup> <a name="entrypoint" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.entrypoint"></a>

- *Deprecated:* use `package.entrypoint`

```typescript
public readonly entrypoint: string;
```

- *Type:* string

---

##### ~~`manifest`~~<sup>Required</sup> <a name="manifest" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.manifest"></a>

- *Deprecated:* use `package.addField(x, y)`

```typescript
public readonly manifest: any;
```

- *Type:* any

---

##### `npmrc`<sup>Required</sup> <a name="npmrc" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.npmrc"></a>

```typescript
public readonly npmrc: NpmConfig;
```

- *Type:* projen.javascript.NpmConfig

The .npmrc file.

---

##### `package`<sup>Required</sup> <a name="package" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.package"></a>

```typescript
public readonly package: NodePackage;
```

- *Type:* projen.javascript.NodePackage

API for managing the node package.

---

##### ~~`packageManager`~~<sup>Required</sup> <a name="packageManager" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.packageManager"></a>

- *Deprecated:* use `package.packageManager`

```typescript
public readonly packageManager: NodePackageManager;
```

- *Type:* projen.javascript.NodePackageManager

The package manager to use.

---

##### `runScriptCommand`<sup>Required</sup> <a name="runScriptCommand" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.runScriptCommand"></a>

```typescript
public readonly runScriptCommand: string;
```

- *Type:* string

The command to use to run scripts (e.g. `yarn run` or `npm run` depends on the package manager).

---

##### `autoMerge`<sup>Optional</sup> <a name="autoMerge" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.autoMerge"></a>

```typescript
public readonly autoMerge: AutoMerge;
```

- *Type:* projen.github.AutoMerge

Component that sets up mergify for merging approved pull requests.

---

##### `buildWorkflow`<sup>Optional</sup> <a name="buildWorkflow" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.buildWorkflow"></a>

```typescript
public readonly buildWorkflow: BuildWorkflow;
```

- *Type:* projen.build.BuildWorkflow

The PR build GitHub workflow.

`undefined` if `buildWorkflow` is disabled.

---

##### `buildWorkflowJobId`<sup>Optional</sup> <a name="buildWorkflowJobId" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.buildWorkflowJobId"></a>

```typescript
public readonly buildWorkflowJobId: string;
```

- *Type:* string

The job ID of the build workflow.

---

##### `jest`<sup>Optional</sup> <a name="jest" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.jest"></a>

```typescript
public readonly jest: Jest;
```

- *Type:* projen.javascript.Jest

The Jest configuration (if enabled).

---

##### `maxNodeVersion`<sup>Optional</sup> <a name="maxNodeVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.maxNodeVersion"></a>

```typescript
public readonly maxNodeVersion: string;
```

- *Type:* string

Maximum node version required by this package.

---

##### `minNodeVersion`<sup>Optional</sup> <a name="minNodeVersion" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.minNodeVersion"></a>

```typescript
public readonly minNodeVersion: string;
```

- *Type:* string

Minimum node.js version required by this package.

---

##### `npmignore`<sup>Optional</sup> <a name="npmignore" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.npmignore"></a>

```typescript
public readonly npmignore: IgnoreFile;
```

- *Type:* projen.IgnoreFile

The .npmignore file.

---

##### `prettier`<sup>Optional</sup> <a name="prettier" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.prettier"></a>

```typescript
public readonly prettier: Prettier;
```

- *Type:* projen.javascript.Prettier

---

##### ~~`publisher`~~<sup>Optional</sup> <a name="publisher" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.publisher"></a>

- *Deprecated:* use `release.publisher`.

```typescript
public readonly publisher: Publisher;
```

- *Type:* projen.release.Publisher

Package publisher.

This will be `undefined` if the project does not have a
release workflow.

---

##### `release`<sup>Optional</sup> <a name="release" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.release"></a>

```typescript
public readonly release: Release;
```

- *Type:* projen.release.Release

Release management.

---

##### `upgradeWorkflow`<sup>Optional</sup> <a name="upgradeWorkflow" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.upgradeWorkflow"></a>

```typescript
public readonly upgradeWorkflow: UpgradeDependencies;
```

- *Type:* projen.javascript.UpgradeDependencies

The upgrade workflow.

---

##### `docsDirectory`<sup>Required</sup> <a name="docsDirectory" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.docsDirectory"></a>

```typescript
public readonly docsDirectory: string;
```

- *Type:* string

---

##### `libdir`<sup>Required</sup> <a name="libdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.libdir"></a>

```typescript
public readonly libdir: string;
```

- *Type:* string

The directory in which compiled .js files reside.

---

##### `srcdir`<sup>Required</sup> <a name="srcdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.srcdir"></a>

```typescript
public readonly srcdir: string;
```

- *Type:* string

The directory in which the .ts sources reside.

---

##### `testdir`<sup>Required</sup> <a name="testdir" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.testdir"></a>

```typescript
public readonly testdir: string;
```

- *Type:* string

The directory in which tests reside.

---

##### `tsconfigDev`<sup>Required</sup> <a name="tsconfigDev" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tsconfigDev"></a>

```typescript
public readonly tsconfigDev: TypescriptConfig;
```

- *Type:* projen.javascript.TypescriptConfig

A typescript configuration file which covers all files (sources, tests, projen).

---

##### `watchTask`<sup>Required</sup> <a name="watchTask" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.watchTask"></a>

```typescript
public readonly watchTask: Task;
```

- *Type:* projen.Task

The "watch" task.

---

##### `docgen`<sup>Optional</sup> <a name="docgen" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.docgen"></a>

```typescript
public readonly docgen: boolean;
```

- *Type:* boolean

---

##### `eslint`<sup>Optional</sup> <a name="eslint" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.eslint"></a>

```typescript
public readonly eslint: Eslint;
```

- *Type:* projen.javascript.Eslint

---

##### `tsconfig`<sup>Optional</sup> <a name="tsconfig" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tsconfig"></a>

```typescript
public readonly tsconfig: TypescriptConfig;
```

- *Type:* projen.javascript.TypescriptConfig

---

##### `tsconfigEslint`<sup>Optional</sup> <a name="tsconfigEslint" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.tsconfigEslint"></a>

```typescript
public readonly tsconfigEslint: TypescriptConfig;
```

- *Type:* projen.javascript.TypescriptConfig

---

##### `nxJson`<sup>Required</sup> <a name="nxJson" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.nxJson"></a>

```typescript
public readonly nxJson: JsonFile;
```

- *Type:* projen.JsonFile

---

##### `subProjects`<sup>Required</sup> <a name="subProjects" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.subProjects"></a>

```typescript
public readonly subProjects: Project[];
```

- *Type:* projen.Project[]

---

#### Constants <a name="Constants" id="Constants"></a>

| **Name** | **Type** | **Description** |
| --- | --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.DEFAULT_TASK">DEFAULT_TASK</a></code> | <code>string</code> | The name of the default task (the task executed when `projen` is run without arguments). |

---

##### `DEFAULT_TASK`<sup>Required</sup> <a name="DEFAULT_TASK" id="@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject.property.DEFAULT_TASK"></a>

```typescript
public readonly DEFAULT_TASK: string;
```

- *Type:* string

The name of the default task (the task executed when `projen` is run without arguments).

Normally
this task should synthesize the project files.

---


## Enums <a name="Enums" id="Enums"></a>

### TargetDependencyProject <a name="TargetDependencyProject" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependencyProject"></a>

Supported enums for a TargetDependency.

#### Members <a name="Members" id="Members"></a>

| **Name** | **Description** |
| --- | --- |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependencyProject.SELF">SELF</a></code> | Only rely on the package where the target is called. |
| <code><a href="#@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependencyProject.DEPENDENCIES">DEPENDENCIES</a></code> | Target relies on executing the target against all dependencies first. |

---

##### `SELF` <a name="SELF" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependencyProject.SELF"></a>

Only rely on the package where the target is called.

This is usually done for test like targets where you only want to run unit
tests on the target packages without testing all dependent packages.

---


##### `DEPENDENCIES` <a name="DEPENDENCIES" id="@aws-prototyping-sdk/nx-monorepo.Nx.TargetDependencyProject.DEPENDENCIES"></a>

Target relies on executing the target against all dependencies first.

This is usually done for build like targets where you want to build all
dependant projects first.

---

