"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NX_VERSION = exports.ensurePackage = exports.removeDependenciesFromPackageJson = exports.addDependenciesToPackageJson = void 0;
const child_process_1 = require("child_process");
const module_1 = require("module");
const semver_1 = require("semver");
const install_packages_task_1 = require("../tasks/install-packages-task");
const nx_1 = require("../../nx");
const tmp_1 = require("tmp");
const path_1 = require("path");
const { readJson, updateJson, getPackageManagerCommand, workspaceRoot } = (0, nx_1.requireNx)();
const UNIDENTIFIED_VERSION = 'UNIDENTIFIED_VERSION';
const NON_SEMVER_TAGS = {
    '*': 2,
    [UNIDENTIFIED_VERSION]: 2,
    next: 1,
    latest: 0,
    previous: -1,
    legacy: -2,
};
function filterExistingDependencies(dependencies, existingAltDependencies) {
    if (!existingAltDependencies) {
        return dependencies;
    }
    return Object.keys(dependencies !== null && dependencies !== void 0 ? dependencies : {})
        .filter((d) => !existingAltDependencies[d])
        .reduce((acc, d) => (Object.assign(Object.assign({}, acc), { [d]: dependencies[d] })), {});
}
function cleanSemver(version) {
    var _a;
    return (_a = (0, semver_1.clean)(version)) !== null && _a !== void 0 ? _a : (0, semver_1.coerce)(version);
}
function isIncomingVersionGreater(incomingVersion, existingVersion) {
    var _a, _b, _c, _d;
    // if version is in the format of "latest", "next" or similar - keep it, otherwise try to parse it
    const incomingVersionCompareBy = incomingVersion in NON_SEMVER_TAGS
        ? incomingVersion
        : (_b = (_a = cleanSemver(incomingVersion)) === null || _a === void 0 ? void 0 : _a.toString()) !== null && _b !== void 0 ? _b : UNIDENTIFIED_VERSION;
    const existingVersionCompareBy = existingVersion in NON_SEMVER_TAGS
        ? existingVersion
        : (_d = (_c = cleanSemver(existingVersion)) === null || _c === void 0 ? void 0 : _c.toString()) !== null && _d !== void 0 ? _d : UNIDENTIFIED_VERSION;
    if (incomingVersionCompareBy in NON_SEMVER_TAGS &&
        existingVersionCompareBy in NON_SEMVER_TAGS) {
        return (NON_SEMVER_TAGS[incomingVersionCompareBy] >
            NON_SEMVER_TAGS[existingVersionCompareBy]);
    }
    if (incomingVersionCompareBy in NON_SEMVER_TAGS ||
        existingVersionCompareBy in NON_SEMVER_TAGS) {
        return true;
    }
    return (0, semver_1.gt)(cleanSemver(incomingVersion), cleanSemver(existingVersion));
}
function updateExistingAltDependenciesVersion(dependencies, existingAltDependencies) {
    return Object.keys(existingAltDependencies || {})
        .filter((d) => {
        if (!dependencies[d]) {
            return false;
        }
        const incomingVersion = dependencies[d];
        const existingVersion = existingAltDependencies[d];
        return isIncomingVersionGreater(incomingVersion, existingVersion);
    })
        .reduce((acc, d) => (Object.assign(Object.assign({}, acc), { [d]: dependencies[d] })), {});
}
function updateExistingDependenciesVersion(dependencies, existingDependencies = {}) {
    return Object.keys(dependencies)
        .filter((d) => {
        if (!existingDependencies[d]) {
            return true;
        }
        const incomingVersion = dependencies[d];
        const existingVersion = existingDependencies[d];
        return isIncomingVersionGreater(incomingVersion, existingVersion);
    })
        .reduce((acc, d) => (Object.assign(Object.assign({}, acc), { [d]: dependencies[d] })), {});
}
/**
 * Add Dependencies and Dev Dependencies to package.json
 *
 * For example:
 * ```typescript
 * addDependenciesToPackageJson(tree, { react: 'latest' }, { jest: 'latest' })
 * ```
 * This will **add** `react` and `jest` to the dependencies and devDependencies sections of package.json respectively.
 *
 * @param tree Tree representing file system to modify
 * @param dependencies Dependencies to be added to the dependencies section of package.json
 * @param devDependencies Dependencies to be added to the devDependencies section of package.json
 * @param packageJsonPath Path to package.json
 * @returns Callback to install dependencies only if necessary, no-op otherwise
 */
function addDependenciesToPackageJson(tree, dependencies, devDependencies, packageJsonPath = 'package.json') {
    const currentPackageJson = readJson(tree, packageJsonPath);
    /** Dependencies to install that are not met in dev dependencies */
    let filteredDependencies = filterExistingDependencies(dependencies, currentPackageJson.devDependencies);
    /** Dev dependencies to install that are not met in dependencies */
    let filteredDevDependencies = filterExistingDependencies(devDependencies, currentPackageJson.dependencies);
    // filtered dependencies should consist of:
    // - dependencies of the same type that are not present
    // - dependencies of the same type that have greater version
    // - specified dependencies of the other type that have greater version and are already installed as current type
    filteredDependencies = Object.assign(Object.assign({}, updateExistingDependenciesVersion(filteredDependencies, currentPackageJson.dependencies)), updateExistingAltDependenciesVersion(devDependencies, currentPackageJson.dependencies));
    filteredDevDependencies = Object.assign(Object.assign({}, updateExistingDependenciesVersion(filteredDevDependencies, currentPackageJson.devDependencies)), updateExistingAltDependenciesVersion(dependencies, currentPackageJson.devDependencies));
    filteredDependencies = removeLowerVersions(filteredDependencies, currentPackageJson.dependencies);
    filteredDevDependencies = removeLowerVersions(filteredDevDependencies, currentPackageJson.devDependencies);
    if (requiresAddingOfPackages(currentPackageJson, filteredDependencies, filteredDevDependencies)) {
        updateJson(tree, packageJsonPath, (json) => {
            json.dependencies = Object.assign(Object.assign({}, (json.dependencies || {})), filteredDependencies);
            json.devDependencies = Object.assign(Object.assign({}, (json.devDependencies || {})), filteredDevDependencies);
            json.dependencies = sortObjectByKeys(json.dependencies);
            json.devDependencies = sortObjectByKeys(json.devDependencies);
            return json;
        });
        return () => {
            (0, install_packages_task_1.installPackagesTask)(tree);
        };
    }
    return () => { };
}
exports.addDependenciesToPackageJson = addDependenciesToPackageJson;
/**
 * @returns The the incoming dependencies that are higher than the existing verions
 **/
function removeLowerVersions(incomingDeps, existingDeps) {
    return Object.keys(incomingDeps).reduce((acc, d) => {
        if ((existingDeps === null || existingDeps === void 0 ? void 0 : existingDeps[d]) &&
            !isIncomingVersionGreater(incomingDeps[d], existingDeps[d])) {
            return acc;
        }
        return Object.assign(Object.assign({}, acc), { [d]: incomingDeps[d] });
    }, {});
}
/**
 * Remove Dependencies and Dev Dependencies from package.json
 *
 * For example:
 * ```typescript
 * removeDependenciesFromPackageJson(tree, ['react'], ['jest'])
 * ```
 * This will **remove** `react` and `jest` from the dependencies and devDependencies sections of package.json respectively.
 *
 * @param dependencies Dependencies to be removed from the dependencies section of package.json
 * @param devDependencies Dependencies to be removed from the devDependencies section of package.json
 * @returns Callback to uninstall dependencies only if necessary. undefined is returned if changes are not necessary.
 */
function removeDependenciesFromPackageJson(tree, dependencies, devDependencies, packageJsonPath = 'package.json') {
    const currentPackageJson = readJson(tree, packageJsonPath);
    if (requiresRemovingOfPackages(currentPackageJson, dependencies, devDependencies)) {
        updateJson(tree, packageJsonPath, (json) => {
            for (const dep of dependencies) {
                delete json.dependencies[dep];
            }
            for (const devDep of devDependencies) {
                delete json.devDependencies[devDep];
            }
            json.dependencies = sortObjectByKeys(json.dependencies);
            json.devDependencies = sortObjectByKeys(json.devDependencies);
            return json;
        });
    }
    return () => {
        (0, install_packages_task_1.installPackagesTask)(tree);
    };
}
exports.removeDependenciesFromPackageJson = removeDependenciesFromPackageJson;
function sortObjectByKeys(obj) {
    if (!obj || typeof obj !== 'object' || Array.isArray(obj)) {
        return obj;
    }
    return Object.keys(obj)
        .sort()
        .reduce((result, key) => {
        return Object.assign(Object.assign({}, result), { [key]: obj[key] });
    }, {});
}
/**
 * Verifies whether the given packageJson dependencies require an update
 * given the deps & devDeps passed in
 */
function requiresAddingOfPackages(packageJsonFile, deps, devDeps) {
    let needsDepsUpdate = false;
    let needsDevDepsUpdate = false;
    packageJsonFile.dependencies = packageJsonFile.dependencies || {};
    packageJsonFile.devDependencies = packageJsonFile.devDependencies || {};
    if (Object.keys(deps).length > 0) {
        needsDepsUpdate = Object.keys(deps).some((entry) => {
            const incomingVersion = deps[entry];
            if (packageJsonFile.dependencies[entry]) {
                const existingVersion = packageJsonFile.dependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            if (packageJsonFile.devDependencies[entry]) {
                const existingVersion = packageJsonFile.devDependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            return true;
        });
    }
    if (Object.keys(devDeps).length > 0) {
        needsDevDepsUpdate = Object.keys(devDeps).some((entry) => {
            const incomingVersion = devDeps[entry];
            if (packageJsonFile.devDependencies[entry]) {
                const existingVersion = packageJsonFile.devDependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            if (packageJsonFile.dependencies[entry]) {
                const existingVersion = packageJsonFile.dependencies[entry];
                return isIncomingVersionGreater(incomingVersion, existingVersion);
            }
            return true;
        });
    }
    return needsDepsUpdate || needsDevDepsUpdate;
}
/**
 * Verifies whether the given packageJson dependencies require an update
 * given the deps & devDeps passed in
 */
function requiresRemovingOfPackages(packageJsonFile, deps, devDeps) {
    let needsDepsUpdate = false;
    let needsDevDepsUpdate = false;
    packageJsonFile.dependencies = packageJsonFile.dependencies || {};
    packageJsonFile.devDependencies = packageJsonFile.devDependencies || {};
    if (deps.length > 0) {
        needsDepsUpdate = deps.some((entry) => packageJsonFile.dependencies[entry]);
    }
    if (devDeps.length > 0) {
        needsDevDepsUpdate = devDeps.some((entry) => packageJsonFile.devDependencies[entry]);
    }
    return needsDepsUpdate || needsDevDepsUpdate;
}
const packageMapCache = new Map();
function ensurePackage(pkgOrTree, requiredVersionOrPackage, maybeRequiredVersion, _) {
    let pkg;
    let requiredVersion;
    if (typeof pkgOrTree === 'string') {
        pkg = pkgOrTree;
        requiredVersion = requiredVersionOrPackage;
    }
    else {
        // Old Signature
        pkg = requiredVersionOrPackage;
        requiredVersion = maybeRequiredVersion;
    }
    if (packageMapCache.has(pkg)) {
        return packageMapCache.get(pkg);
    }
    try {
        return require(pkg);
    }
    catch (e) {
        if (e.code !== 'MODULE_NOT_FOUND') {
            throw e;
        }
    }
    if (process.env.NX_DRY_RUN && process.env.NX_DRY_RUN !== 'false') {
        throw new Error('NOTE: This generator does not support --dry-run. If you are running this in Nx Console, it should execute fine once you hit the "Run" button.\n');
    }
    const tempDir = (0, tmp_1.dirSync)().name;
    console.log(`Fetching ${pkg}...`);
    (0, child_process_1.execSync)(`${getPackageManagerCommand().addDev} ${pkg}@${requiredVersion}`, {
        cwd: tempDir,
        stdio: 'ignore',
    });
    addToNodePath((0, path_1.join)(workspaceRoot, 'node_modules'));
    addToNodePath((0, path_1.join)(tempDir, 'node_modules'));
    // Re-initialize the added paths into require
    module_1.Module._initPaths();
    const result = require(require.resolve(pkg, {
        paths: [tempDir],
    }));
    packageMapCache.set(pkg, result);
    return result;
}
exports.ensurePackage = ensurePackage;
function addToNodePath(dir) {
    // NODE_PATH is a delimited list of paths.
    // The delimiter is different for windows.
    const delimiter = require('os').platform() === 'win32' ? ';' : ':';
    const paths = process.env.NODE_PATH
        ? process.env.NODE_PATH.split(delimiter)
        : [];
    // The path is already in the node path
    if (paths.includes(dir)) {
        return;
    }
    // Add the tmp path
    paths.push(dir);
    // Update the env variable.
    process.env.NODE_PATH = paths.join(delimiter);
}
function getPackageVersion(pkg) {
    return require((0, path_1.join)(pkg, 'package.json')).version;
}
/**
 * @description The version of Nx used by the workspace. Returns null if no version is found.
 */
exports.NX_VERSION = getPackageVersion('nx');
//# sourceMappingURL=package-json.js.map