"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const serverless = require("@aws-cdk/aws-sam");
const core_1 = require("@aws-cdk/core");
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser ', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.3', {
    isMultiUser: true
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        const rotationFunctionName = this.node.uniqueId;
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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