"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeKind = exports.EmptyDirMedium = exports.Volume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const k8s = require("./imports/k8s");
/**
 * Volume represents a named volume in a pod that may be accessed by any
 * container in the pod.
 *
 * Docker also has a concept of volumes, though it is somewhat looser and less
 * managed. In Docker, a volume is simply a directory on disk or in another
 * Container. Lifetimes are not managed and until very recently there were only
 * local-disk-backed volumes. Docker now provides volume drivers, but the
 * functionality is very limited for now (e.g. as of Docker 1.7 only one volume
 * driver is allowed per Container and there is no way to pass parameters to
 * volumes).
 *
 * A Kubernetes volume, on the other hand, has an explicit lifetime - the same
 * as the Pod that encloses it. Consequently, a volume outlives any Containers
 * that run within the Pod, and data is preserved across Container restarts. Of
 * course, when a Pod ceases to exist, the volume will cease to exist, too.
 * Perhaps more importantly than this, Kubernetes supports many types of
 * volumes, and a Pod can use any number of them simultaneously.
 *
 * At its core, a volume is just a directory, possibly with some data in it,
 * which is accessible to the Containers in a Pod. How that directory comes to
 * be, the medium that backs it, and the contents of it are determined by the
 * particular volume type used.
 *
 * To use a volume, a Pod specifies what volumes to provide for the Pod (the
 * .spec.volumes field) and where to mount those into Containers (the
 * .spec.containers[*].volumeMounts field).
 *
 * A process in a container sees a filesystem view composed from their Docker
 * image and volumes. The Docker image is at the root of the filesystem
 * hierarchy, and any volumes are mounted at the specified paths within the
 * image. Volumes can not mount onto other volumes
 */
class Volume {
    constructor(name, config) {
        this.name = name;
        this.config = config;
    }
    /**
     * Mounts an Amazon Web Services (AWS) EBS volume into your pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of an EBS volume are
     * persisted and the volume is unmounted. This means that an EBS volume can be pre-populated with data,
     * and that data can be shared between pods.
     *
     * There are some restrictions when using an awsElasticBlockStore volume:
     *
     * - the nodes on which pods are running must be AWS EC2 instances.
     * - those instances need to be in the same region and availability zone as the EBS volume.
     * - EBS only supports a single EC2 instance mounting a volume.
     */
    static fromAwsElasticBlockStore(volumeId, options = {}) {
        var _b, _c, _d;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `ebs-${volumeId}`, {
            awsElasticBlockStore: {
                volumeId,
                fsType: (_c = options.fsType) !== null && _c !== void 0 ? _c : 'ext4',
                partition: options.partition,
                readOnly: (_d = options.readOnly) !== null && _d !== void 0 ? _d : false,
            },
        });
    }
    /**
     * Mounts a Microsoft Azure Data Disk into a pod.
     */
    static fromAzureDisk(diskName, diskUri, options = {}) {
        var _b, _c, _d, _e, _f;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `azuredisk-${diskName}`, {
            azureDisk: {
                diskName,
                diskUri,
                cachingMode: (_c = options.cachingMode) !== null && _c !== void 0 ? _c : AzureDiskPersistentVolumeCachingMode.NONE,
                fsType: (_d = options.fsType) !== null && _d !== void 0 ? _d : 'ext4',
                kind: (_e = options.kind) !== null && _e !== void 0 ? _e : AzureDiskPersistentVolumeKind.SHARED,
                readOnly: (_f = options.readOnly) !== null && _f !== void 0 ? _f : false,
            },
        });
    }
    /**
     * Mounts a Google Compute Engine (GCE) persistent disk (PD) into your Pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of a PD are
     * preserved and the volume is merely unmounted. This means that a PD can be pre-populated
     * with data, and that data can be shared between pods.
     *
     * There are some restrictions when using a gcePersistentDisk:
     *
     * - the nodes on which Pods are running must be GCE VMs
     * - those VMs need to be in the same GCE project and zone as the persistent disk
     */
    static fromGcePersistentDisk(pdName, options = {}) {
        var _b, _c, _d;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `gcedisk-${pdName}`, {
            gcePersistentDisk: {
                pdName,
                fsType: (_c = options.fsType) !== null && _c !== void 0 ? _c : 'ext4',
                partition: options.partition,
                readOnly: (_d = options.readOnly) !== null && _d !== void 0 ? _d : false,
            },
        });
    }
    /**
     * Populate the volume from a ConfigMap.
     *
     * The configMap resource provides a way to inject configuration data into
     * Pods. The data stored in a ConfigMap object can be referenced in a volume
     * of type configMap and then consumed by containerized applications running
     * in a Pod.
     *
     * When referencing a configMap object, you can simply provide its name in the
     * volume to reference it. You can also customize the path to use for a
     * specific entry in the ConfigMap.
     *
     * @param configMap The config map to use to populate the volume.
     * @param options Options
     */
    static fromConfigMap(configMap, options = {}) {
        var _b;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `configmap-${configMap.name}`, {
            configMap: {
                name: configMap.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * An emptyDir volume is first created when a Pod is assigned to a Node, and
     * exists as long as that Pod is running on that node. As the name says, it is
     * initially empty. Containers in the Pod can all read and write the same
     * files in the emptyDir volume, though that volume can be mounted at the same
     * or different paths in each Container. When a Pod is removed from a node for
     * any reason, the data in the emptyDir is deleted forever.
     *
     * @see http://kubernetes.io/docs/user-guide/volumes#emptydir
     *
     * @param options - Additional options.
     */
    static fromEmptyDir(name, options = {}) {
        return new Volume(name, {
            emptyDir: {
                medium: options.medium,
                sizeLimit: options.sizeLimit
                    ? k8s.Quantity.fromString(`${options.sizeLimit.toMebibytes()}Mi`)
                    : undefined,
            },
        });
    }
    /**
     * Populate the volume from a Secret.
     *
     * A secret volume is used to pass sensitive information, such as passwords, to Pods.
     * You can store secrets in the Kubernetes API and mount them as files for use by pods
     * without coupling to Kubernetes directly.
     *
     * secret volumes are backed by tmpfs (a RAM-backed filesystem)
     * so they are never written to non-volatile storage.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#secret
     *
     * @param secr The secret to use to populate the volume.
     * @param options Options
     */
    static fromSecret(secr, options = {}) {
        var _b;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `secret-${secr.name}`, {
            secret: {
                secretName: secr.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * Used to mount a PersistentVolume into a Pod.
     * PersistentVolumeClaims are a way for users to "claim" durable storage (such as a GCE PersistentDisk or an iSCSI volume)
     * without knowing the details of the particular cloud environment.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/
     */
    static fromPersistentVolumeClaim(claim, options = {}) {
        var _b, _c;
        return new Volume((_b = options.name) !== null && _b !== void 0 ? _b : `pvc-${claim.name}`, {
            persistentVolumeClaim: {
                claimName: claim.name,
                readOnly: (_c = options.readOnly) !== null && _c !== void 0 ? _c : false,
            },
        });
    }
    asVolume() {
        return this;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            name: this.name,
            ...this.config,
        };
    }
}
exports.Volume = Volume;
_a = JSII_RTTI_SYMBOL_1;
Volume[_a] = { fqn: "cdk8s-plus-22.Volume", version: "2.0.0-beta.1" };
/**
  * @internal
 */
Volume.renderItems = (items) => {
    if (!items) {
        return undefined;
    }
    const result = new Array();
    for (const key of Object.keys(items).sort()) {
        result.push({
            key,
            path: items[key].path,
            mode: items[key].mode,
        });
    }
    return result;
};
/**
 * The medium on which to store the volume.
 */
var EmptyDirMedium;
(function (EmptyDirMedium) {
    /**
     * The default volume of the backing node.
     */
    EmptyDirMedium["DEFAULT"] = "";
    /**
     * Mount a tmpfs (RAM-backed filesystem) for you instead. While tmpfs is very
     * fast, be aware that unlike disks, tmpfs is cleared on node reboot and any
     * files you write will count against your Container's memory limit.
     */
    EmptyDirMedium["MEMORY"] = "Memory";
})(EmptyDirMedium = exports.EmptyDirMedium || (exports.EmptyDirMedium = {}));
/**
 * Azure Disk kinds.
 */
var AzureDiskPersistentVolumeKind;
(function (AzureDiskPersistentVolumeKind) {
    /**
     * Multiple blob disks per storage account.
     */
    AzureDiskPersistentVolumeKind["SHARED"] = "Shared";
    /**
     * Single blob disk per storage account.
     */
    AzureDiskPersistentVolumeKind["DEDICATED"] = "Dedicated";
    /**
     * Azure managed data disk.
     */
    AzureDiskPersistentVolumeKind["MANAGED"] = "Managed";
})(AzureDiskPersistentVolumeKind = exports.AzureDiskPersistentVolumeKind || (exports.AzureDiskPersistentVolumeKind = {}));
/**
 * Azure disk caching modes.
 */
var AzureDiskPersistentVolumeCachingMode;
(function (AzureDiskPersistentVolumeCachingMode) {
    /**
     * None.
     */
    AzureDiskPersistentVolumeCachingMode["NONE"] = "None";
    /**
     * ReadOnly.
     */
    AzureDiskPersistentVolumeCachingMode["READ_ONLY"] = "ReadOnly";
    /**
     * ReadWrite.
     */
    AzureDiskPersistentVolumeCachingMode["READ_WRITE"] = "ReadWrite";
})(AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeCachingMode || (exports.AzureDiskPersistentVolumeCachingMode = {}));
//# sourceMappingURL=data:application/json;base64,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