import { ApiObject } from 'cdk8s';
import { Construct } from 'constructs';
import * as base from './base';
import * as rb from './role-binding';
import * as secret from './secret';
export interface IServiceAccount extends base.IResource {
}
/**
 * Properties for initialization of `ServiceAccount`.
 */
export interface ServiceAccountProps extends base.ResourceProps {
    /**
     * List of secrets allowed to be used by pods running using this
     * ServiceAccount.
     *
     * @see https://kubernetes.io/docs/concepts/configuration/secret
     */
    readonly secrets?: secret.ISecret[];
    /**
     * Indicates whether pods running as this service account
     * should have an API token automatically mounted. Can be overridden at the pod level.
     *
     * @default true
     * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/#use-the-default-service-account-to-access-the-api-server
     */
    readonly automountToken?: boolean;
}
/**
 * A service account provides an identity for processes that run in a Pod.
 *
 * When you (a human) access the cluster (for example, using kubectl), you are
 * authenticated by the apiserver as a particular User Account (currently this
 * is usually admin, unless your cluster administrator has customized your
 * cluster). Processes in containers inside pods can also contact the apiserver.
 * When they do, they are authenticated as a particular Service Account (for
 * example, default).
 *
 * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account
 */
export declare class ServiceAccount extends base.Resource implements IServiceAccount, rb.ISubject {
    /**
     * Imports a service account from the cluster as a reference.
     * @param name The name of the service account resource.
     */
    static fromServiceAccountName(name: string): IServiceAccount;
    /**
     * @see base.Resource.apiObject
     */
    protected readonly apiObject: ApiObject;
    readonly resourceType = "serviceaccounts";
    private readonly _secrets;
    /**
     * Whether or not a token is automatically mounted for this
     * service account.
     */
    readonly automountToken: boolean;
    constructor(scope: Construct, id: string, props?: ServiceAccountProps);
    /**
     * Allow a secret to be accessed by pods using this service account.
     * @param secr The secret
     */
    addSecret(secr: secret.ISecret): void;
    /**
     * List of secrets allowed to be used by pods running using this service
     * account.
     *
     * Returns a copy. To add a secret, use `addSecret()`.
     */
    get secrets(): secret.ISecret[];
}
