"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterRole = exports.Role = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const rb = require("./role-binding");
/**
 * Role is a namespaced, logical grouping of PolicyRules that can be referenced
 * as a unit by a RoleBinding.
 */
class Role extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'roles';
        this._rules = [];
        this.apiObject = new k8s.KubeRole(this, 'Resource', {
            metadata: props.metadata,
            rules: cdk8s_1.Lazy.any({ produce: () => this.synthesizeRules() }),
        });
    }
    /**
     * Imports a role from the cluster as a reference.
     * @param name The name of the role resource.
     */
    static fromRoleName(name) {
        return {
            apiGroup: 'rbac.authorization.k8s.io',
            name,
            ...k8s.KubeRole.GVK,
        };
    }
    /**
     * Rules associaated with this Role.
     * Returns a copy, use `allow` to add rules.
     */
    get rules() {
        return [...this._rules];
    }
    /**
     * Add permission to perform a list of HTTP verbs on a collection of
     * resources.
     *
     * @param resources The resource(s) to apply to
     * @see https://kubernetes.io/docs/reference/access-authn-authz/authorization/#determine-the-request-verb
     */
    allow(verbs, ...resources) {
        this._rules.push({ verbs, resources });
    }
    /**
     * Add "create" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowCreate(...resources) {
        this.allow(['create'], ...resources);
    }
    /**
     * Add "get" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowGet(...resources) {
        this.allow(['get'], ...resources);
    }
    /**
     * Add "list" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowList(...resources) {
        this.allow(['list'], ...resources);
    }
    /**
     * Add "watch" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowWatch(...resources) {
        this.allow(['watch'], ...resources);
    }
    /**
     * Add "update" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowUpdate(...resources) {
        this.allow(['update'], ...resources);
    }
    /**
     * Add "patch" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowPatch(...resources) {
        this.allow(['patch'], ...resources);
    }
    /**
     * Add "delete" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowDelete(...resources) {
        this.allow(['delete'], ...resources);
    }
    /**
     * Add "deletecollection" permission for the resources.
     * @param resources The resource(s) to apply to
     */
    allowDeleteCollection(...resources) {
        this.allow(['deletecollection'], ...resources);
    }
    /**
     * Add "get", "list", and "watch" permissions for the resources.
     * @param resources The resource(s) to apply to
     */
    allowRead(...resources) {
        this.allow(['get', 'list', 'watch'], ...resources);
    }
    /**
     * Add "get", "list", "watch", "create", "update", "patch", "delete", and
     * "deletecollection" permissions for the resources.
     *
     * @param resources The resource(s) to apply to
     */
    allowReadWrite(...resources) {
        this.allow(['get', 'list', 'watch', 'create', 'update', 'patch', 'delete', 'deletecollection'], ...resources);
    }
    /**
     * Create a RoleBinding that binds the permissions in this Role
     * to a list of subjects, that will only apply this role's namespace.
     * @param subjects a list of subjects to bind to
     */
    bind(...subjects) {
        const binding = new rb.RoleBinding(this, 'RoleBinding', {
            metadata: {
                namespace: this.metadata.namespace,
            },
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    synthesizeRules() {
        const rules = [];
        for (const rule of this._rules) {
            for (const resource of rule.resources) {
                rules.push({
                    verbs: rule.verbs,
                    apiGroups: [resource.apiGroup === 'core' ? '' : resource.apiGroup],
                    resourceNames: resource.resourceName ? [resource.resourceName] : undefined,
                    resources: resource.resourceType ? [resource.resourceType] : undefined,
                });
            }
        }
        return rules;
    }
}
exports.Role = Role;
_a = JSII_RTTI_SYMBOL_1;
Role[_a] = { fqn: "cdk8s-plus-22.Role", version: "2.0.0-beta.1" };
/**
 * ClusterRole is a cluster level, logical grouping of PolicyRules that can be
 * referenced as a unit by a RoleBinding or ClusterRoleBinding.
 */
class ClusterRole extends base.Resource {
    constructor(scope, id, props = {}) {
        var _c, _d;
        super(scope, id);
        this.resourceType = 'clusterroles';
        this._labelSelector = {};
        this._rules = [];
        this.apiObject = new k8s.KubeClusterRole(this, 'Resource', {
            metadata: props.metadata,
            rules: cdk8s_1.Lazy.any({ produce: () => this.synthesizeRules() }),
            aggregationRule: cdk8s_1.Lazy.any({ produce: () => this.synthesizeAggregationRules() }),
        });
        for (const rule of (_c = props.rules) !== null && _c !== void 0 ? _c : []) {
            this.allow(rule.verbs, ...rule.endpoints);
        }
        for (const [key, value] of Object.entries((_d = props.aggregationLabels) !== null && _d !== void 0 ? _d : {})) {
            this.aggregate(key, value);
        }
    }
    /**
     * Imports a role from the cluster as a reference.
     * @param name The name of the role resource.
     */
    static fromClusterRoleName(name) {
        return {
            apiGroup: 'rbac.authorization.k8s.io',
            name,
            ...k8s.KubeClusterRole.GVK,
        };
    }
    /**
     * Rules associaated with this Role.
     * Returns a copy, use `allow` to add rules.
     */
    get rules() {
        return [...this._rules];
    }
    /**
     * Add permission to perform a list of HTTP verbs on a collection of
     * resources.
     *
     * @param endpoints The endpoints(s) to apply to
     * @see https://kubernetes.io/docs/reference/access-authn-authz/authorization/#determine-the-request-verb
     */
    allow(verbs, ...endpoints) {
        this._rules.push({ verbs, endpoints });
    }
    /**
     * Add "create" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowCreate(...endpoints) {
        this.allow(['create'], ...endpoints);
    }
    /**
     * Add "get" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowGet(...endpoints) {
        this.allow(['get'], ...endpoints);
    }
    /**
     * Add "list" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowList(...endpoints) {
        this.allow(['list'], ...endpoints);
    }
    /**
     * Add "watch" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowWatch(...endpoints) {
        this.allow(['watch'], ...endpoints);
    }
    /**
     * Add "update" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowUpdate(...endpoints) {
        this.allow(['update'], ...endpoints);
    }
    /**
     * Add "patch" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowPatch(...endpoints) {
        this.allow(['patch'], ...endpoints);
    }
    /**
     * Add "delete" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowDelete(...endpoints) {
        this.allow(['delete'], ...endpoints);
    }
    /**
     * Add "deletecollection" permission for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowDeleteCollection(...endpoints) {
        this.allow(['deletecollection'], ...endpoints);
    }
    /**
     * Add "get", "list", and "watch" permissions for the resources.
     * @param endpoints The resource(s) to apply to
     */
    allowRead(...endpoints) {
        this.allow(['get', 'list', 'watch'], ...endpoints);
    }
    /**
     * Add "get", "list", "watch", "create", "update", "patch", "delete", and
     * "deletecollection" permissions for the resources.
     *
     * @param endpoints The resource(s) to apply to
     */
    allowReadWrite(...endpoints) {
        this.allow(['get', 'list', 'watch', 'create', 'update', 'patch', 'delete', 'deletecollection'], ...endpoints);
    }
    /**
     * Aggregate rules from roles matching this label selector.
     */
    aggregate(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Combines the rules of the argument ClusterRole into this ClusterRole
     * using aggregation labels.
     * @param rol
     */
    combine(rol) {
        const key = `cdk8s.cluster-role/aggregate-to-${cdk8s_1.Names.toLabelValue(this)}`;
        const value = 'true';
        rol.metadata.addLabel(key, value);
        this.aggregate(key, value);
    }
    /**
     * Create a RoleBinding that binds the permissions in this ClusterRole
     * to a list of subjects, that will only apply to the given namespace.
     * @param namespace the namespace to limit permissions to.
     * @param subjects a list of subjects to bind to
     */
    bindInNamespace(namespace, ...subjects) {
        const binding = new rb.RoleBinding(this, `RoleBinding-${namespace}`, {
            metadata: {
                namespace,
            },
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    /**
     * Create a ClusterRoleBinding that binds the permissions in this
     * ClusterRole to a list of subjects, without namespace restrictions.
     * @param subjects a list of subjects to bind to
     */
    bind(...subjects) {
        const binding = new rb.ClusterRoleBinding(this, 'ClusterRoleBinding', {
            role: this,
        });
        binding.addSubjects(...subjects);
        return binding;
    }
    synthesizeRules() {
        const rules = [];
        for (const rule of this._rules) {
            for (const endpoint of rule.endpoints) {
                const resource = endpoint.asApiResource();
                const nonResource = endpoint.asNonApiResource();
                if (resource && nonResource) {
                    throw new Error('Endpoint must be either resource or non resource. not both.');
                }
                if (!resource && !nonResource) {
                    throw new Error('Endpoint must be either resource or non resource. not neither.');
                }
                if (resource) {
                    rules.push({
                        apiGroups: [resource.apiGroup === 'core' ? '' : resource.apiGroup],
                        resources: [resource.resourceType],
                        resourceNames: resource.resourceName ? [resource.resourceName] : [],
                        verbs: rule.verbs,
                    });
                }
                if (nonResource) {
                    rules.push({ verbs: rule.verbs, nonResourceUrLs: [nonResource] });
                }
            }
        }
        return rules;
    }
    synthesizeAggregationRules() {
        if (Object.keys(this._labelSelector).length === 0) {
            return undefined;
        }
        return { clusterRoleSelectors: [{ matchLabels: this._labelSelector }] };
    }
}
exports.ClusterRole = ClusterRole;
_b = JSII_RTTI_SYMBOL_1;
ClusterRole[_b] = { fqn: "cdk8s-plus-22.ClusterRole", version: "2.0.0-beta.1" };
//# sourceMappingURL=data:application/json;base64,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