import { ApiObject, ApiObjectMetadataDefinition } from 'cdk8s';
import { Construct } from 'constructs';
import { Resource, ResourceProps } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { Ingress } from './ingress';
import { RestartPolicy, IPodTemplate, PodTemplateProps } from './pod';
import { ExposeServiceViaIngressOptions, Protocol, Service, ServiceType } from './service';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Properties for initialization of `Deployment`.
 *
 * @stability stable
 */
export interface DeploymentProps extends ResourceProps, PodTemplateProps {
    /**
     * Number of desired pods.
     *
     * @default 1
     * @stability stable
     */
    readonly replicas?: number;
    /**
     * Automatically allocates a pod selector for this deployment.
     *
     * If this is set to `false` you must define your selector through
     * `deployment.podMetadata.addLabel()` and `deployment.selectByLabel()`.
     *
     * @default true
     * @stability stable
     */
    readonly defaultSelector?: boolean;
}
/**
 * Options for exposing a deployment via a service.
 *
 * @stability stable
 */
export interface ExposeDeploymentViaServiceOptions {
    /**
     * The port that the service should serve on.
     *
     * @default - Copied from the container of the deployment. If a port could not be determined, throws an error.
     * @stability stable
     */
    readonly port?: number;
    /**
     * The type of the exposed service.
     *
     * @default - ClusterIP.
     * @stability stable
     */
    readonly serviceType?: ServiceType;
    /**
     * The name of the service to expose.
     *
     * This will be set on the Service.metadata and must be a DNS_LABEL
     *
     * @default undefined Uses the system generated name.
     * @stability stable
     */
    readonly name?: string;
    /**
     * The IP protocol for this port.
     *
     * Supports "TCP", "UDP", and "SCTP". Default is TCP.
     *
     * @default Protocol.TCP
     * @stability stable
     */
    readonly protocol?: Protocol;
    /**
     * The port number the service will redirect to.
     *
     * @default - The port of the first container in the deployment (ie. containers[0].port)
     * @stability stable
     */
    readonly targetPort?: number;
}
/**
 * Options for exposing a deployment via an ingress.
 *
 * @stability stable
 */
export interface ExposeDeploymentViaIngressOptions extends ExposeDeploymentViaServiceOptions, ExposeServiceViaIngressOptions {
}
/**
 * A Deployment provides declarative updates for Pods and ReplicaSets.
 *
 * You describe a desired state in a Deployment, and the Deployment Controller changes the actual
 * state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
 * existing Deployments and adopt all their resources with new Deployments.
 *
 * > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
 *
 * Use Case
 * ---------
 *
 * The following are typical use cases for Deployments:
 *
 * - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
 *    Check the status of the rollout to see if it succeeds or not.
 * - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
 *    A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
 *    Each new ReplicaSet updates the revision of the Deployment.
 * - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
 *    Each rollback updates the revision of the Deployment.
 * - Scale up the Deployment to facilitate more load.
 * - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
 * - Use the status of the Deployment as an indicator that a rollout has stuck.
 * - Clean up older ReplicaSets that you don't need anymore.
 *
 * @stability stable
 */
export declare class Deployment extends Resource implements IPodTemplate {
    /**
     * Number of desired pods.
     *
     * @stability stable
     */
    readonly replicas: number;
    /**
     * The underlying cdk8s API object.
     *
     * @see base.Resource.apiObject
     * @stability stable
     */
    protected readonly apiObject: ApiObject;
    private readonly _podTemplate;
    private readonly _labelSelector;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: DeploymentProps);
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     *
     * @stability stable
     */
    get podMetadata(): ApiObjectMetadataDefinition;
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     *
     * @stability stable
     */
    get labelSelector(): Record<string, string>;
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    get containers(): Container[];
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    get volumes(): Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    get restartPolicy(): RestartPolicy | undefined;
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    get serviceAccount(): IServiceAccount | undefined;
    /**
     * Configure a label selector to this deployment.
     *
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     * @stability stable
     */
    selectByLabel(key: string, value: string): void;
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     * @stability stable
     */
    exposeViaService(options?: ExposeDeploymentViaServiceOptions): Service;
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     * @stability stable
     */
    exposeViaIngress(path: string, options?: ExposeDeploymentViaIngressOptions): Ingress;
    /**
     * Add a container to the pod.
     *
     * @stability stable
     */
    addContainer(container: ContainerProps): Container;
    /**
     * Add a volume to the pod.
     *
     * @stability stable
     */
    addVolume(volume: Volume): void;
    /**
     * @internal
     */
    _toKube(): k8s.DeploymentSpec;
}
