"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const API = require("./api");
/* eslint-disable no-console */
async function handler(event) {
    switch (event.RequestType) {
        case 'Create':
        case 'Update':
            return updateFlowTime(event);
        case 'Delete':
            return;
        default:
            throw new Error('Unknown signal');
    }
}
exports.handler = handler;
;
async function updateFlowTime(event) {
    let expressionString = event.ResourceProperties[API.PROP_SCHEDULE];
    const propStartTime = event.ResourceProperties[API.PROP_STARTTIME];
    let startTime = propStartTime
        ? new Date(Date.parse(propStartTime))
        : undefined;
    const propEndTime = event.ResourceProperties[API.PROP_ENDTIME];
    let endTime = propEndTime
        ? new Date(Date.parse(propEndTime))
        : undefined;
    if (/^rate/.test(expressionString)) {
        ({ expressionString, startTime } = buildForRateBasedSchedule(expressionString, startTime));
    }
    else if (/^cron/.test(expressionString)) {
        // do nothing, as we don't change the expressionString
        // TODO: do we have to set the startTime? I think we do
    }
    else {
        throw new Error(`Unrecognized expression ${expressionString}`);
    }
    return {
        Data: {
            [API.ATTR_SCHEDULE]: expressionString,
            [API.ATTR_STARTTIME]: startTime && startTime.getTime() / 1000,
            [API.ATTR_ENDTIME]: endTime && endTime.getTime() / 1000,
        },
    };
}
function buildForRateBasedSchedule(expressionString, startTime) {
    // TODO: the below statement is from a customer. I need to check it more thoroughly
    // Rebuilding expression string as Flows require always plural, so: hour -> hours, etc.
    const matches = /\((\d*)(.*)\)$/.exec(expressionString);
    if (!matches) {
        throw new Error(`Expression ${expressionString} is not a rate-based expression`);
    }
    const rate = parseFloat(matches[1]);
    let rateUnit = matches[2].trim();
    if (!rateUnit.endsWith('s')) {
        rateUnit = `${rateUnit}s`;
    }
    expressionString = `rate(${rate} ${rateUnit})`;
    if (startTime) {
        const nowMillis = Date.now();
        const startMillis = startTime.getTime();
        if (nowMillis > startMillis) {
            let denominator = 1;
            switch (rateUnit) {
                case 'minutes':
                    denominator = 60 * 1000; // minutes in millis
                    break;
                case 'hours':
                    denominator = 60 * 60 * 1000; // hours in millis
                    break;
                case 'days':
                    denominator = 24 * 60 * 60 * 1000; // days in millis
                    break;
                default:
                    throw new Error(`Unable to use schedule "${expressionString}"`);
            }
            const cyclesBetween = Math.ceil((nowMillis - startMillis) / denominator / rate);
            startTime = new Date((startMillis / denominator + cyclesBetween * rate) * denominator);
        }
    }
    return { expressionString, startTime };
}
//# sourceMappingURL=data:application/json;base64,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