"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowBase = exports.DataPullMode = exports.FlowStatus = exports.FlowType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appflow_1 = require("aws-cdk-lib/aws-appflow");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const flow_time_updater_1 = require("./flow-time-updater");
var FlowType;
(function (FlowType) {
    FlowType["EVENT"] = "Event";
    FlowType["ON_DEMAND"] = "OnDemand";
    FlowType["SCHEDULED"] = "Scheduled";
})(FlowType || (exports.FlowType = FlowType = {}));
var FlowStatus;
(function (FlowStatus) {
    FlowStatus["ACTIVE"] = "Active";
    FlowStatus["SUSPENDED"] = "Suspended";
})(FlowStatus || (exports.FlowStatus = FlowStatus = {}));
var DataPullMode;
(function (DataPullMode) {
    DataPullMode["COMPLETE"] = "Complete";
    DataPullMode["INCREMENTAL"] = "Incremental";
})(DataPullMode || (exports.DataPullMode = DataPullMode = {}));
class FlowBase extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.mappings = [];
        this.validations = [];
        this.transforms = [];
        this.filters = [];
        this.type = props.type;
        this._projectionFilter = this.initProjectionFilter(props.source.connectorType);
        this.name = props.name || id;
        const resource = new aws_appflow_1.CfnFlow(this, id, {
            flowName: this.name,
            flowStatus: props.status,
            triggerConfig: {
                triggerType: props.type,
                triggerProperties: props.triggerConfig
                    && props.triggerConfig.properties
                    && this.buildTriggerProperties(scope, id, props.triggerConfig.properties),
            },
            kmsArn: props.key?.keyArn,
            metadataCatalogConfig: aws_cdk_lib_1.Lazy.any({ produce: () => this._catalogMetadata }),
            description: props.description,
            sourceFlowConfig: {
                ...props.source.bind(this),
                incrementalPullConfig: this.buildIncrementalPullConfig(props.triggerConfig),
            },
            // NOTE: currently only a single destination is allowed with AppFlow
            //       it might require a change of approach in the future.
            destinationFlowConfigList: [props.destination.bind(this)],
            tasks: aws_cdk_lib_1.Lazy.any({
                produce: () => {
                    const tasks = [
                        ...this.mappings,
                        ...this.transforms,
                        ...this.filters,
                        ...this.validations,
                    ];
                    // TODO: make sure that this.filters doesn't already have a projection definition
                    if (this._projectionFilter.sourceFields.length > 0) {
                        tasks.unshift(this._projectionFilter);
                    }
                    return tasks;
                },
            }),
        });
        this.arn = resource.attrFlowArn;
        this.source = props.source;
        props.mappings.forEach(m => this._addMapping(m));
        if (props.validations) {
            props.validations.forEach(v => this._addValidation(v));
        }
        if (props.transforms) {
            props.transforms.forEach(t => this._addTransform(t));
        }
        if (props.filters) {
            props.filters.forEach(f => this._addFilter(f));
        }
        this.node.addValidation({
            validate: () => this.mappings.length === 0 ? ['A Flow must have at least one mapping'] : [],
        });
    }
    metric(metricName, options) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AppFlow',
            metricName,
            dimensionsMap: {
                FlowName: this.name,
            },
            ...options,
        });
    }
    metricFlowExecutionsStarted(options) {
        return this.metric('FlowExecutionsStarted', options);
    }
    metricFlowExecutionsFailed(options) {
        return this.metric('FlowExecutionsFailed', options);
    }
    metricFlowExecutionsSucceeded(options) {
        return this.metric('FlowExecutionsSucceeded', options);
    }
    metricFlowExecutionTime(options) {
        return this.metric('FlowExecutionTime', options);
    }
    metricFlowExecutionRecordsProcessed(options) {
        return this.metric('FlowExecutionRecordsProcessed', options);
    }
    buildTriggerProperties(scope, id, props) {
        const updater = new flow_time_updater_1.FlowTimeUpdater(scope, `${id}Updater`, {
            schedule: props.schedule,
            startTime: props.properties?.startTime,
            endTime: props.properties?.endTime,
        });
        this.node.addDependency(updater);
        return {
            dataPullMode: props.dataPullConfig.mode,
            flowErrorDeactivationThreshold: props.flowErrorDeactivationThreshold,
            scheduleExpression: updater.scheduleExpression,
            firstExecutionFrom: props.properties?.firstExecutionFrom &&
                Math.floor(props.properties.firstExecutionFrom.getTime() / 1000),
            scheduleStartTime: props.properties?.startTime && updater.startTime,
            scheduleEndTime: props.properties?.endTime && updater.endTime,
        };
    }
    initProjectionFilter(sourceType) {
        const filterConnectorOperator = {};
        filterConnectorOperator[sourceType.asTaskConnectorOperatorOrigin] = 'PROJECTION';
        return {
            taskType: 'Filter',
            connectorOperator: filterConnectorOperator,
            sourceFields: [],
        };
    }
    /**
       * Set the catalog definitionfor the flow
       *
       * @internal
       */
    _addCatalog(metadata) {
        this._catalogMetadata = metadata;
    }
    /**
     *
     * @param validation
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addValidation(validation) {
        this.validations.push(...validation.bind(this, this.source));
        return this;
    }
    /**
     *
     * @param mapping
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addMapping(mapping) {
        const boundMappingTasks = mapping.bind(this, this.source);
        this.addProjectionField(boundMappingTasks);
        this.mappings.push(...boundMappingTasks);
        return this;
    }
    /**
     *
     * @param filter
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addFilter(filter) {
        const boundFilterTasks = filter.bind(this, this.source);
        this.addProjectionField(boundFilterTasks);
        this.filters.push(...boundFilterTasks);
        return this;
    }
    /**
     *
     * @param transform
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addTransform(transform) {
        this.transforms.push(...transform.bind(this, this.source));
        return this;
    }
    addProjectionField(boundMappingTasks) {
        // TODO: test if this satisfies all the requirements.
        boundMappingTasks.forEach(boundMapping => {
            if (['Map', 'Filter'].includes(boundMapping.taskType)) {
                boundMapping.sourceFields.forEach(field => {
                    if (!this._projectionFilter.sourceFields.includes(field)) {
                        this._projectionFilter.sourceFields.push(field);
                    }
                });
            }
        });
    }
    // see: https://docs.aws.amazon.com/appflow/latest/userguide/flow-notifications.html
    onEvent(id, options = {}) {
        const rule = new aws_events_1.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.appflow'],
            resources: [this.arn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    onRunStarted(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['AppFlow Start Flow Run Report'],
        });
        return rule;
    }
    onRunCompleted(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['AppFlow End Flow Run Report'],
        });
        return rule;
    }
    buildIncrementalPullConfig(triggerConfig) {
        return triggerConfig && triggerConfig.properties && triggerConfig.properties.dataPullConfig
            && triggerConfig.properties.dataPullConfig.timestampField
            ? {
                datetimeTypeFieldName: triggerConfig.properties.dataPullConfig.timestampField,
            }
            : undefined;
    }
}
exports.FlowBase = FlowBase;
_a = JSII_RTTI_SYMBOL_1;
FlowBase[_a] = { fqn: "@cdklabs/cdk-appflow.FlowBase", version: "0.0.36" };
//# sourceMappingURL=data:application/json;base64,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