"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppFlowPermissionsManager = void 0;
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const glob = global;
/**
 * see: https://docs.aws.amazon.com/appflow/latest/userguide/salesforce.html#salesforce-setup
 * @internal
 */
class AppFlowPermissionsManager {
    /**
     * Singleton instance of the AppFlowPermissionsManager
     */
    static instance() {
        if (!glob.__cdkAppFlowPermissionsManager) {
            glob.__cdkAppFlowPermissionsManager = new AppFlowPermissionsManager();
        }
        return glob.__cdkAppFlowPermissionsManager;
    }
    constructor() {
        this.principal = new aws_iam_1.ServicePrincipal('appflow.amazonaws.com');
        this.tokenPermissions = new Map();
        this.writeS3BucketActions = ['s3:PutObject', 's3:AbortMultipartUpload',
            's3:ListMultipartUploadParts', 's3:ListBucketMultipartUploads',
            's3:GetBucketAcl', 's3:PutObjectAcl'];
        this.readS3BucketActions = ['s3:GetObject', 's3:ListBucket'];
        this.encryptKMSKeyActions = ['kms:Encrypt', 'kms:GenerateDataKey'];
        this.decryptKMSKeyActions = ['kms:Decrypt'];
        this.readSMSecretActions = ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'];
    }
    getActionsFor(arn) {
        let actions;
        let isNew;
        if (this.tokenPermissions.has(arn)) {
            actions = this.tokenPermissions.get(arn);
            isNew = false;
        }
        else {
            actions = new Set();
            this.tokenPermissions.set(arn, actions);
            isNew = true;
        }
        return { actions, isNew: isNew };
    }
    grantBucketWrite(bucket) {
        if (bucket) {
            const { actions, isNew } = this.getActionsFor(bucket.bucketArn);
            this.writeS3BucketActions.forEach(a => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: [
                        bucket.bucketArn,
                        bucket.arnForObjects('*'),
                    ],
                    conditions: {
                        StringEquals: {
                            'aws:SourceAccount': aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = bucket.addToResourcePolicy(statement);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(bucket).addWarning('Cannot change bucket policy of imported S3 bucket. Ensure that your bucket policy contains appropriate permissions');
                }
            }
            this.grantKeyEncrypt(bucket.encryptionKey);
        }
    }
    grantBucketRead(bucket) {
        if (bucket) {
            const { actions, isNew } = this.getActionsFor(bucket.bucketArn);
            this.readS3BucketActions.forEach(a => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: [
                        bucket.bucketArn,
                        bucket.arnForObjects('*'),
                    ],
                    conditions: {
                        StringEquals: {
                            'aws:SourceAccount': aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = bucket.addToResourcePolicy(statement);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(bucket).addWarning('Cannot change bucket policy of imported S3 bucket. Ensure that your bucket policy contains appropriate permissions');
                }
            }
            this.grantKeyDecrypt(bucket.encryptionKey);
        }
    }
    grantBucketReadWrite(bucket) {
        if (bucket) {
            this.grantBucketRead(bucket);
            this.grantBucketWrite(bucket);
        }
    }
    grantKeyEncryptDecrypt(key) {
        this.grantKeyEncrypt(key);
        this.grantKeyDecrypt(key);
    }
    grantKeyEncrypt(key) {
        if (key) {
            const { actions, isNew } = this.getActionsFor(key.keyArn);
            this.encryptKMSKeyActions.forEach(a => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:SourceAccount': aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = key.addToResourcePolicy(statement, /* allowNoOp */ true);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(key).addWarning('Cannot change key policy of imported kms key. Ensure that your key policy contains appropriate permissions');
                }
            }
        }
    }
    grantKeyDecrypt(key) {
        if (key) {
            const { actions, isNew } = this.getActionsFor(key.keyArn);
            this.decryptKMSKeyActions.forEach(a => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: ['*'],
                    conditions: {
                        StringEquals: {
                            'aws:SourceAccount': aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = key.addToResourcePolicy(statement, /* allowNoOp */ true);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(key).addWarning('Cannot change key policy of imported kms key. Ensure that your key policy contains appropriate permissions');
                }
            }
        }
    }
    grantSecretRead(secret) {
        if (secret) {
            const { actions, isNew } = this.getActionsFor(secret.secretArn);
            this.readSMSecretActions.forEach(a => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: [secret.secretFullArn ? secret.secretFullArn : `${secret.secretArn}-??????`],
                    conditions: {
                        StringEquals: {
                            'aws:SourceAccount': aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = secret.addToResourcePolicy(statement);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(secret).addWarning('Cannot change secret policy of imported sm secret. Ensure that your secret policy contains appropriate permissions');
                }
                this.grantKeyEncryptDecrypt(secret.encryptionKey);
            }
        }
    }
}
exports.AppFlowPermissionsManager = AppFlowPermissionsManager;
//# sourceMappingURL=data:application/json;base64,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