"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatefulSetUpdateStrategy = exports.StatefulSet = exports.PodManagementPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const workload = require("./workload");
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes,
 * or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 */
var PodManagementPolicy;
(function (PodManagementPolicy) {
    PodManagementPolicy["ORDERED_READY"] = "OrderedReady";
    PodManagementPolicy["PARALLEL"] = "Parallel";
})(PodManagementPolicy = exports.PodManagementPolicy || (exports.PodManagementPolicy = {}));
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 */
class StatefulSet extends workload.Workload {
    constructor(scope, id, props) {
        var _c, _d, _e, _f;
        super(scope, id, props);
        this.resourceType = 'statefulsets';
        this.apiObject = new k8s.KubeStatefulSet(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this._service = props.service;
        this.apiObject.addDependency(this._service);
        this.replicas = (_c = props.replicas) !== null && _c !== void 0 ? _c : 1;
        this.strategy = (_d = props.strategy) !== null && _d !== void 0 ? _d : StatefulSetUpdateStrategy.rollingUpdate(),
            this.podManagementPolicy = (_e = props.podManagementPolicy) !== null && _e !== void 0 ? _e : PodManagementPolicy.ORDERED_READY;
        this.minReady = (_f = props.minReady) !== null && _f !== void 0 ? _f : cdk8s_1.Duration.seconds(0);
        this._service.select(this);
    }
    /**
      * @internal
      */
    _toKube() {
        return {
            serviceName: this._service.name,
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            podManagementPolicy: this.podManagementPolicy,
            updateStrategy: this.strategy._toKube(),
        };
    }
}
exports.StatefulSet = StatefulSet;
_a = JSII_RTTI_SYMBOL_1;
StatefulSet[_a] = { fqn: "cdk8s-plus-22.StatefulSet", version: "2.0.0-rc.2" };
/**
 * StatefulSet update strategies.
 */
class StatefulSetUpdateStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * The controller will not automatically update the Pods in a StatefulSet.
     * Users must manually delete Pods to cause the controller to create new Pods
     * that reflect modifications.
     */
    static onDelete() {
        return new StatefulSetUpdateStrategy({
            type: 'OnDelete',
        });
    }
    /**
     * The controller will delete and recreate each Pod in the StatefulSet.
     * It will proceed in the same order as Pod termination (from the largest ordinal to the smallest),
     * updating each Pod one at a time. The Kubernetes control plane waits until an updated
     * Pod is Running and Ready prior to updating its predecessor.
     */
    static rollingUpdate(options = {}) {
        var _c;
        return new StatefulSetUpdateStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { partition: (_c = options.partition) !== null && _c !== void 0 ? _c : 0 },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.StatefulSetUpdateStrategy = StatefulSetUpdateStrategy;
_b = JSII_RTTI_SYMBOL_1;
StatefulSetUpdateStrategy[_b] = { fqn: "cdk8s-plus-22.StatefulSetUpdateStrategy", version: "2.0.0-rc.2" };
//# sourceMappingURL=data:application/json;base64,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