"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.LabelSelectorOptions = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        var _t;
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = (_t = props.automountServiceAccountToken) !== null && _t !== void 0 ? _t : true;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        var _t;
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: (_t = cont.name) !== null && _t !== void 0 ? _t : `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        var _t, _u;
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = (_u = (_t = this.serviceAccount) === null || _t === void 0 ? void 0 : _t.name) !== null && _u !== void 0 ? _u : 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        var _t;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_t = this.serviceAccount) === null || _t === void 0 ? void 0 : _t.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-22.AbstractPod", version: "2.0.0-rc.2" };
/**
 * Options for `LabelSelector.of`.
 */
class LabelSelectorOptions {
}
exports.LabelSelectorOptions = LabelSelectorOptions;
_b = JSII_RTTI_SYMBOL_1;
LabelSelectorOptions[_b] = { fqn: "cdk8s-plus-22.LabelSelectorOptions", version: "2.0.0-rc.2" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        var _t, _u;
        return new LabelSelector((_t = options.expressions) !== null && _t !== void 0 ? _t : [], (_u = options.labels) !== null && _u !== void 0 ? _u : {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_c = JSII_RTTI_SYMBOL_1;
LabelSelector[_c] = { fqn: "cdk8s-plus-22.LabelSelector", version: "2.0.0-rc.2" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_d = JSII_RTTI_SYMBOL_1;
Pod[_d] = { fqn: "cdk8s-plus-22.Pod", version: "2.0.0-rc.2" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        var _t, _u, _v, _w, _x;
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = (_t = props.policy) !== null && _t !== void 0 ? _t : DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = (_u = props.hostnameAsFQDN) !== null && _u !== void 0 ? _u : false;
        this._nameservers = (_v = props.nameservers) !== null && _v !== void 0 ? _v : [];
        this._searches = (_w = props.searches) !== null && _w !== void 0 ? _w : [];
        this._options = (_x = props.options) !== null && _x !== void 0 ? _x : [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_e = JSII_RTTI_SYMBOL_1;
PodDns[_e] = { fqn: "cdk8s-plus-22.PodDns", version: "2.0.0-rc.2" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        var _t, _u, _v;
        this._sysctls = [];
        this.ensureNonRoot = (_t = props.ensureNonRoot) !== null && _t !== void 0 ? _t : false;
        this.fsGroupChangePolicy = (_u = props.fsGroupChangePolicy) !== null && _u !== void 0 ? _u : FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of (_v = props.sysctls) !== null && _v !== void 0 ? _v : []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_f = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_f] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "2.0.0-rc.2" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, 'DoesNotExist', undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, 'Gt', values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, 'Lt', values);
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_g = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_g] = { fqn: "cdk8s-plus-22.NodeLabelQuery", version: "2.0.0-rc.2" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_h = JSII_RTTI_SYMBOL_1;
LabelExpression[_h] = { fqn: "cdk8s-plus-22.LabelExpression", version: "2.0.0-rc.2" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery('Equal', key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery('Exists', key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery('Exists');
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_j = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_j] = { fqn: "cdk8s-plus-22.NodeTaintQuery", version: "2.0.0-rc.2" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        var _t;
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: (_t = this.namespaces) === null || _t === void 0 ? void 0 : _t.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_k = JSII_RTTI_SYMBOL_1;
Pods[_k] = { fqn: "cdk8s-plus-22.Pods", version: "2.0.0-rc.2" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_l = JSII_RTTI_SYMBOL_1;
LabeledNode[_l] = { fqn: "cdk8s-plus-22.LabeledNode", version: "2.0.0-rc.2" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_m = JSII_RTTI_SYMBOL_1;
TaintedNode[_m] = { fqn: "cdk8s-plus-22.TaintedNode", version: "2.0.0-rc.2" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_o = JSII_RTTI_SYMBOL_1;
NamedNode[_o] = { fqn: "cdk8s-plus-22.NamedNode", version: "2.0.0-rc.2" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_p = JSII_RTTI_SYMBOL_1;
Node[_p] = { fqn: "cdk8s-plus-22.Node", version: "2.0.0-rc.2" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_q = JSII_RTTI_SYMBOL_1;
Topology[_q] = { fqn: "cdk8s-plus-22.Topology", version: "2.0.0-rc.2" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        var _t;
        for (const query of node.taintSelector) {
            this._tolerations.push({
                key: query.key,
                value: query.value,
                effect: query.effect,
                operator: query.operator,
                tolerationSeconds: (_t = query.evictAfter) === null || _t === void 0 ? void 0 : _t.toSeconds(),
            });
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        var _t;
        const topology = (_t = options.topology) !== null && _t !== void 0 ? _t : Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        var _t;
        const topology = (_t = options.topology) !== null && _t !== void 0 ? _t : Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        var _t, _u, _v;
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: (_u = (_t = config.namespaces) === null || _t === void 0 ? void 0 : _t.labelSelector) === null || _u === void 0 ? void 0 : _u._toKube(),
            namespaces: (_v = config.namespaces) === null || _v === void 0 ? void 0 : _v.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => ({ key: s.key, operator: s.operator, values: s.values })) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_r = JSII_RTTI_SYMBOL_1;
PodScheduling[_r] = { fqn: "cdk8s-plus-22.PodScheduling", version: "2.0.0-rc.2" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n));
    }
}
exports.PodConnections = PodConnections;
_s = JSII_RTTI_SYMBOL_1;
PodConnections[_s] = { fqn: "cdk8s-plus-22.PodConnections", version: "2.0.0-rc.2" };
//# sourceMappingURL=data:application/json;base64,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