"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.validatePeerConfig = exports.NetworkPolicy = exports.NetworkPolicyTrafficDefault = exports.NetworkProtocol = exports.NetworkPolicyIpBlock = exports.NetworkPolicyPort = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const namespace = require("./namespace");
const pod = require("./pod");
const utils_1 = require("./utils");
/**
 * Describes a port to allow traffic on.
 */
class NetworkPolicyPort {
    constructor(port, endPort, protocol) {
        this.port = port;
        this.endPort = endPort;
        this.protocol = protocol;
    }
    /**
     * Distinct TCP ports
     */
    static tcp(port) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(port), undefined, NetworkProtocol.TCP);
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(startPort), endPort, NetworkProtocol.TCP);
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(0), 65535, NetworkProtocol.TCP);
    }
    /**
     * Distinct UDP ports
     */
    static udp(port) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(port), undefined, NetworkProtocol.UDP);
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(startPort), endPort, NetworkProtocol.UDP);
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(0), 65535, NetworkProtocol.UDP);
    }
    /**
     * Custom port configuration.
     */
    static of(props) {
        return new NetworkPolicyPort(props.port ? k8s.IntOrString.fromNumber(props.port) : undefined, props.endPort, props.protocol);
    }
    /**
     * @internal
     */
    _toKube() {
        return { port: this.port, endPort: this.endPort, protocol: this.protocol };
    }
}
exports.NetworkPolicyPort = NetworkPolicyPort;
_a = JSII_RTTI_SYMBOL_1;
NetworkPolicyPort[_a] = { fqn: "cdk8s-plus-22.NetworkPolicyPort", version: "2.0.0-rc.2" };
/**
 * Describes a particular CIDR (Ex. "192.168.1.1/24","2001:db9::/64") that is
 * allowed to the pods matched by a network policy selector.
 * The except entry describes CIDRs that should not be included within this rule.
 */
class NetworkPolicyIpBlock extends constructs_1.Construct {
    constructor(scope, id, 
    /**
     * A string representing the IP Block Valid examples are "192.168.1.1/24" or "2001:db9::/64".
     */
    cidr, 
    /**
     * A slice of CIDRs that should not be included within an IP Block Valid examples are "192.168.1.1/24" or "2001:db9::/64".
     * Except values will be rejected if they are outside the CIDR range.
     */
    except) {
        super(scope, id);
        this.cidr = cidr;
        this.except = except;
    }
    /**
     * Create an IPv4 peer from a CIDR
     */
    static ipv4(scope, id, cidrIp, except) {
        const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
        if (!cidrMatch) {
            throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
        }
        if (!cidrMatch[2]) {
            throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
        }
        return new NetworkPolicyIpBlock(scope, id, cidrIp, except);
    }
    /**
     * Any IPv4 address
     */
    static anyIpv4(scope, id) {
        return new NetworkPolicyIpBlock(scope, id, '0.0.0.0/0');
    }
    /**
     * Create an IPv6 peer from a CIDR
     */
    static ipv6(scope, id, cidrIp, except) {
        const cidrMatch = cidrIp.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
        if (!cidrMatch) {
            throw new Error(`Invalid IPv6 CIDR: "${cidrIp}"`);
        }
        if (!cidrMatch[3]) {
            throw new Error(`CIDR mask is missing in IPv6: "${cidrIp}". Did you mean "${cidrIp}/128"?`);
        }
        return new NetworkPolicyIpBlock(scope, id, cidrIp, except);
    }
    /**
     * Any IPv6 address
     */
    static anyIpv6(scope, id) {
        return new NetworkPolicyIpBlock(scope, id, '::/0');
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { ipBlock: this };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return undefined;
    }
    /**
     * @internal
     */
    _toKube() {
        return { cidr: this.cidr, except: this.except };
    }
}
exports.NetworkPolicyIpBlock = NetworkPolicyIpBlock;
_b = JSII_RTTI_SYMBOL_1;
NetworkPolicyIpBlock[_b] = { fqn: "cdk8s-plus-22.NetworkPolicyIpBlock", version: "2.0.0-rc.2" };
/**
 * Network protocols.
 */
var NetworkProtocol;
(function (NetworkProtocol) {
    /**
     * TCP.
     */
    NetworkProtocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    NetworkProtocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    NetworkProtocol["SCTP"] = "SCTP";
})(NetworkProtocol = exports.NetworkProtocol || (exports.NetworkProtocol = {}));
/**
 * Default behaviors of network traffic in policies.
 */
var NetworkPolicyTrafficDefault;
(function (NetworkPolicyTrafficDefault) {
    /**
     * The policy denies all traffic.
     * Since rules are additive, additional rules or policies can allow
     * specific traffic.
     */
    NetworkPolicyTrafficDefault["DENY"] = "DENY";
    /**
     * The policy allows all traffic (either ingress or egress).
     * Since rules are additive, no additional rule or policies can
     * subsequently deny the traffic.
     */
    NetworkPolicyTrafficDefault["ALLOW"] = "ALLOW";
})(NetworkPolicyTrafficDefault = exports.NetworkPolicyTrafficDefault || (exports.NetworkPolicyTrafficDefault = {}));
/**
 * Control traffic flow at the IP address or port level (OSI layer 3 or 4),
 * network policies are an application-centric construct which allow you
 * to specify how a pod is allowed to communicate with various network peers.
 *
 * - Outgoing traffic is allowed if there are no network policies selecting
 *   the pod (and cluster policy otherwise allows the traffic),
 *   OR if the traffic matches at least one egress rule across all of the
 *   network policies that select the pod.
 *
 * - Incoming traffic is allowed to a pod if there are no network policies
 *   selecting the pod (and cluster policy otherwise allows the traffic),
 *   OR if the traffic source is the pod's local node,
 *   OR if the traffic matches at least one ingress rule across all of
 *   the network policies that select the pod.
 *
 * Network policies do not conflict; they are additive.
 * If any policy or policies apply to a given pod for a given
 * direction, the connections allowed in that direction from
 * that pod is the union of what the applicable policies allow.
 * Thus, order of evaluation does not affect the policy result.
 *
 * For a connection from a source pod to a destination pod to be allowed,
 * both the egress policy on the source pod and the ingress policy on the
 * destination pod need to allow the connection.
 * If either side does not allow the connection, it will not happen.
 *
 * @see https://kubernetes.io/docs/concepts/services-networking/network-policies/#networkpolicy-resource
 */
class NetworkPolicy extends base.Resource {
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        super(scope, id);
        this.resourceType = 'networkpolicies';
        this._egressRules = [];
        this._ingressRules = [];
        this._policyTypes = new Set();
        const podSelector = (_d = props.selector) !== null && _d !== void 0 ? _d : pod.Pods.all(this, 'AllPods');
        this._podSelectorConfig = podSelector.toPodSelectorConfig();
        let ns;
        if (!((_e = props.metadata) === null || _e === void 0 ? void 0 : _e.namespace)) {
            if (((_f = this._podSelectorConfig.namespaces) === null || _f === void 0 ? void 0 : _f.labelSelector) && !((_g = this._podSelectorConfig.namespaces) === null || _g === void 0 ? void 0 : _g.labelSelector.isEmpty())) {
                throw new Error(`Unable to create a network policy for a selector (${podSelector.node.path}) that selects pods in namespaces based on labels`);
            }
            if (((_h = this._podSelectorConfig.namespaces) === null || _h === void 0 ? void 0 : _h.names) && this._podSelectorConfig.namespaces.names.length > 1) {
                throw new Error(`Unable to create a network policy for a selector (${podSelector.node.path}) that selects pods in multiple namespaces`);
            }
            ns = ((_j = this._podSelectorConfig.namespaces) === null || _j === void 0 ? void 0 : _j.names) ? (_k = this._podSelectorConfig.namespaces) === null || _k === void 0 ? void 0 : _k.names[0] : undefined;
        }
        else {
            ns = props.metadata.namespace;
        }
        this.apiObject = new k8s.KubeNetworkPolicy(this, 'Resource', {
            metadata: { ...props.metadata, namespace: ns },
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.configureDefaultBehavior('Egress', (_l = props.egress) === null || _l === void 0 ? void 0 : _l.default);
        this.configureDefaultBehavior('Ingress', (_m = props.ingress) === null || _m === void 0 ? void 0 : _m.default);
        for (const rule of (_p = (_o = props.egress) === null || _o === void 0 ? void 0 : _o.rules) !== null && _p !== void 0 ? _p : []) {
            this.addEgressRule(rule.peer, rule.ports);
        }
        for (const rule of (_r = (_q = props.ingress) === null || _q === void 0 ? void 0 : _q.rules) !== null && _r !== void 0 ? _r : []) {
            this.addIngressRule(rule.peer, rule.ports);
        }
    }
    /**
     * Allow outgoing traffic to the peer.
     *
     * If ports are not passed, traffic will be allowed on all ports.
     */
    addEgressRule(peer, ports) {
        this._policyTypes.add('Egress');
        this._egressRules.push({ ports: (ports !== null && ports !== void 0 ? ports : []).map(p => p._toKube()), to: this.createNetworkPolicyPeers(peer) });
    }
    /**
     * Allow incoming traffic from the peer.
     *
     * If ports are not passed, traffic will be allowed on all ports.
     */
    addIngressRule(peer, ports) {
        this._policyTypes.add('Ingress');
        this._ingressRules.push({ ports: (ports !== null && ports !== void 0 ? ports : []).map(p => p._toKube()), from: this.createNetworkPolicyPeers(peer) });
    }
    createNetworkPolicyPeers(peer) {
        var _d, _e, _f, _g, _h, _j, _k;
        const config = peer.toNetworkPolicyPeerConfig();
        validatePeerConfig(config);
        if (config.ipBlock) {
            // ip block is a single peer.
            return [{ ipBlock: config.ipBlock._toKube() }];
        }
        if (!((_d = config.podSelector.namespaces) === null || _d === void 0 ? void 0 : _d.names)) {
            // when no explicit namespaces are defined we can just use
            // the selector as is
            return [{
                    namespaceSelector: (_f = (_e = config.podSelector.namespaces) === null || _e === void 0 ? void 0 : _e.labelSelector) === null || _f === void 0 ? void 0 : _f._toKube(),
                    podSelector: config.podSelector.labelSelector._toKube(),
                }];
        }
        // when explicit namespaces are defined, we need to create a separate
        // peer for each, since a label selector cannot have multiple name labels. (they will conflict)
        const namespaceSelector = (_k = (_j = (_h = (_g = config.podSelector) === null || _g === void 0 ? void 0 : _g.namespaces) === null || _h === void 0 ? void 0 : _h.labelSelector) === null || _j === void 0 ? void 0 : _j._toKube()) !== null && _k !== void 0 ? _k : {};
        return config.podSelector.namespaces.names.map(n => ({
            podSelector: config.podSelector.labelSelector._toKube(),
            namespaceSelector: {
                matchExpressions: namespaceSelector.matchExpressions,
                matchLabels: {
                    ...namespaceSelector.matchLabels,
                    [namespace.Namespace.NAME_LABEL]: n,
                },
            },
        }));
    }
    configureDefaultBehavior(direction, _default) {
        if (!_default) {
            return;
        }
        if (_default === NetworkPolicyTrafficDefault.DENY) {
            // https://kubernetes.io/docs/concepts/services-networking/network-policies/#default-deny-all-egress-traffic
            this._policyTypes.add(direction);
        }
        if (_default === NetworkPolicyTrafficDefault.ALLOW) {
            // https://kubernetes.io/docs/concepts/services-networking/network-policies/#allow-all-egress-traffic
            this._policyTypes.add(direction);
            if (direction === 'Egress') {
                this._egressRules.push({});
            }
            else {
                this._ingressRules.push({});
            }
        }
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            podSelector: this._podSelectorConfig.labelSelector._toKube(),
            egress: utils_1.undefinedIfEmpty(this._egressRules),
            ingress: utils_1.undefinedIfEmpty(this._ingressRules),
            policyTypes: utils_1.undefinedIfEmpty(Array.from(this._policyTypes)),
        };
    }
}
exports.NetworkPolicy = NetworkPolicy;
_c = JSII_RTTI_SYMBOL_1;
NetworkPolicy[_c] = { fqn: "cdk8s-plus-22.NetworkPolicy", version: "2.0.0-rc.2" };
function validatePeerConfig(peerConfig) {
    if (!peerConfig.ipBlock && !peerConfig.podSelector) {
        throw new Error('Inavlid peer: either \'ipBlock\' or \'podSelector\' must be defined');
    }
    if (peerConfig.ipBlock && peerConfig.podSelector) {
        throw new Error('Inavlid peer: only one of \'ipBlock\' and \'podSelector\' must be defined');
    }
}
exports.validatePeerConfig = validatePeerConfig;
//# sourceMappingURL=data:application/json;base64,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