import * as notifications from '@aws-cdk/aws-codestarnotifications';
import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import * as s3 from '@aws-cdk/aws-s3';
import { IResource } from '@aws-cdk/core';
import { Artifact } from './artifact';
import { Construct } from '@aws-cdk/core';
/**
 * @stability stable
 */
export declare enum ActionCategory {
    /**
     * @stability stable
     */
    SOURCE = "Source",
    /**
     * @stability stable
     */
    BUILD = "Build",
    /**
     * @stability stable
     */
    TEST = "Test",
    /**
     * @stability stable
     */
    APPROVAL = "Approval",
    /**
     * @stability stable
     */
    DEPLOY = "Deploy",
    /**
     * @stability stable
     */
    INVOKE = "Invoke"
}
/**
 * Specifies the constraints on the number of input and output artifacts an action can have.
 *
 * The constraints for each action type are documented on the
 * {@link https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html Pipeline Structure Reference} page.
 *
 * @stability stable
 */
export interface ActionArtifactBounds {
    /**
     * @stability stable
     */
    readonly minInputs: number;
    /**
     * @stability stable
     */
    readonly maxInputs: number;
    /**
     * @stability stable
     */
    readonly minOutputs: number;
    /**
     * @stability stable
     */
    readonly maxOutputs: number;
}
/**
 * The CodePipeline variables that are global, not bound to a specific action.
 *
 * This class defines a bunch of static fields that represent the different variables.
 * These can be used can be used in any action configuration.
 *
 * @stability stable
 */
export declare class GlobalVariables {
    /**
     * The identifier of the current pipeline execution.
     *
     * @stability stable
     */
    static readonly executionId = "#{codepipeline.PipelineExecutionId}";
}
/**
 * @stability stable
 */
export interface ActionProperties {
    /**
     * @stability stable
     */
    readonly actionName: string;
    /**
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The AWS region the given Action resides in.
     *
     * Note that a cross-region Pipeline requires replication buckets to function correctly.
     * You can provide their names with the {@link PipelineProps#crossRegionReplicationBuckets} property.
     * If you don't, the CodePipeline Construct will create new Stacks in your CDK app containing those buckets,
     * that you will need to `cdk deploy` before deploying the main, Pipeline-containing Stack.
     *
     * @default the Action resides in the same region as the Pipeline
     * @stability stable
     */
    readonly region?: string;
    /**
     * The account the Action is supposed to live in.
     *
     * For Actions backed by resources,
     * this is inferred from the Stack {@link resource} is part of.
     * However, some Actions, like the CloudFormation ones,
     * are not backed by any resource, and they still might want to be cross-account.
     * In general, a concrete Action class should specify either {@link resource},
     * or {@link account} - but not both.
     *
     * @stability stable
     */
    readonly account?: string;
    /**
     * The optional resource that is backing this Action.
     *
     * This is used for automatically handling Actions backed by
     * resources from a different account and/or region.
     *
     * @stability stable
     */
    readonly resource?: IResource;
    /**
     * The category of the action.
     *
     * The category defines which action type the owner
     * (the entity that performs the action) performs.
     *
     * @stability stable
     */
    readonly category: ActionCategory;
    /**
     * The service provider that the action calls.
     *
     * @stability stable
     */
    readonly provider: string;
    /**
     * @stability stable
     */
    readonly owner?: string;
    /**
     * @stability stable
     */
    readonly version?: string;
    /**
     * The order in which AWS CodePipeline runs this action. For more information, see the AWS CodePipeline User Guide.
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#action-requirements
     *
     * @stability stable
     */
    readonly runOrder?: number;
    /**
     * @stability stable
     */
    readonly artifactBounds: ActionArtifactBounds;
    /**
     * @stability stable
     */
    readonly inputs?: Artifact[];
    /**
     * @stability stable
     */
    readonly outputs?: Artifact[];
    /**
     * The name of the namespace to use for variables emitted by this action.
     *
     * @default - a name will be generated, based on the stage and action names
     * @stability stable
     */
    readonly variablesNamespace?: string;
}
/**
 * @stability stable
 */
export interface ActionBindOptions {
    /**
     * @stability stable
     */
    readonly role: iam.IRole;
    /**
     * @stability stable
     */
    readonly bucket: s3.IBucket;
}
/**
 * @stability stable
 */
export interface ActionConfig {
    /**
     * @stability stable
     */
    readonly configuration?: any;
}
/**
 * Additional options to pass to the notification rule.
 *
 * @stability stable
 */
export interface PipelineNotifyOnOptions extends notifications.NotificationRuleOptions {
    /**
     * A list of event types associated with this notification rule for CodePipeline Pipeline.
     *
     * For a complete list of event types and IDs, see Notification concepts in the Developer Tools Console User Guide.
     *
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#concepts-api
     * @stability stable
     */
    readonly events: PipelineNotificationEvents[];
}
/**
 * A Pipeline Action.
 *
 * If you want to implement this interface,
 * consider extending the {@link Action} class,
 * which contains some common logic.
 *
 * @stability stable
 */
export interface IAction {
    /**
     * The simple properties of the Action, like its Owner, name, etc.
     *
     * Note that this accessor will be called before the {@link bind} callback.
     *
     * @stability stable
     */
    readonly actionProperties: ActionProperties;
    /**
     * The callback invoked when this Action is added to a Pipeline.
     *
     * @param scope the Construct tree scope the Action can use if it needs to create any resources.
     * @param stage the {@link IStage} this Action is being added to.
     * @param options additional options the Action can use, like the artifact Bucket of the pipeline it's being added to.
     * @stability stable
     */
    bind(scope: Construct, stage: IStage, options: ActionBindOptions): ActionConfig;
    /**
     * Creates an Event that will be triggered whenever the state of this Action changes.
     *
     * @param name the name to use for the new Event.
     * @param target the optional target for the Event.
     * @param options additional options that can be used to customize the created Event.
     * @stability stable
     */
    onStateChange(name: string, target?: events.IRuleTarget, options?: events.RuleProps): events.Rule;
}
/**
 * The abstract view of an AWS CodePipeline as required and used by Actions.
 *
 * It extends {@link events.IRuleTarget},
 * so this interface can be used as a Target for CloudWatch Events.
 *
 * @stability stable
 */
export interface IPipeline extends IResource, notifications.INotificationRuleSource {
    /**
     * The name of the Pipeline.
     *
     * @stability stable
     * @attribute true
     */
    readonly pipelineName: string;
    /**
     * The ARN of the Pipeline.
     *
     * @stability stable
     * @attribute true
     */
    readonly pipelineArn: string;
    /**
     * Define an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Define an event rule triggered by the "CodePipeline Pipeline Execution State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     * @stability stable
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CodeStar notification rule triggered when the pipeline events emitted by you specified, it very similar to `onEvent` API.
     *
     * You can also use the methods `notifyOnExecutionStateChange`, `notifyOnAnyStageStateChange`,
     * `notifyOnAnyActionStateChange` and `notifyOnAnyManualApprovalStateChange`
     * to define rules for these specific event emitted.
     *
     * @param id The id of the CodeStar notification rule.
     * @param target The target to register for the CodeStar Notifications destination.
     * @param options Customization options for CodeStar notification rule.
     * @returns CodeStar notification rule associated with this build project.
     * @stability stable
     */
    notifyOn(id: string, target: notifications.INotificationRuleTarget, options: PipelineNotifyOnOptions): notifications.INotificationRule;
    /**
     * Define an notification rule triggered by the set of the "Pipeline execution" events emitted from this pipeline.
     *
     * @param id Identifier for this notification handler.
     * @param target The target to register for the CodeStar Notifications destination.
     * @param options Additional options to pass to the notification rule.
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-pipeline
     * @stability stable
     */
    notifyOnExecutionStateChange(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Define an notification rule triggered by the set of the "Stage execution" events emitted from this pipeline.
     *
     * @param id Identifier for this notification handler.
     * @param target The target to register for the CodeStar Notifications destination.
     * @param options Additional options to pass to the notification rule.
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-pipeline
     * @stability stable
     */
    notifyOnAnyStageStateChange(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Define an notification rule triggered by the set of the "Action execution" events emitted from this pipeline.
     *
     * @param id Identifier for this notification handler.
     * @param target The target to register for the CodeStar Notifications destination.
     * @param options Additional options to pass to the notification rule.
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-pipeline
     * @stability stable
     */
    notifyOnAnyActionStateChange(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Define an notification rule triggered by the set of the "Manual approval" events emitted from this pipeline.
     *
     * @param id Identifier for this notification handler.
     * @param target The target to register for the CodeStar Notifications destination.
     * @param options Additional options to pass to the notification rule.
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-pipeline
     * @stability stable
     */
    notifyOnAnyManualApprovalStateChange(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
}
/**
 * The abstract interface of a Pipeline Stage that is used by Actions.
 *
 * @stability stable
 */
export interface IStage {
    /**
     * The physical, human-readable name of this Pipeline Stage.
     *
     * @stability stable
     */
    readonly stageName: string;
    /**
     * @stability stable
     */
    readonly pipeline: IPipeline;
    /**
     * The actions belonging to this stage.
     *
     * @stability stable
     */
    readonly actions: IAction[];
    /**
     * @stability stable
     */
    addAction(action: IAction): void;
    /**
     * @stability stable
     */
    onStateChange(name: string, target?: events.IRuleTarget, options?: events.RuleProps): events.Rule;
}
/**
 * Common properties shared by all Actions.
 *
 * @stability stable
 */
export interface CommonActionProps {
    /**
     * The physical, human-readable name of the Action.
     *
     * Note that Action names must be unique within a single Stage.
     *
     * @stability stable
     */
    readonly actionName: string;
    /**
     * The runOrder property for this Action.
     *
     * RunOrder determines the relative order in which multiple Actions in the same Stage execute.
     *
     * @default 1
     * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html
     * @stability stable
     */
    readonly runOrder?: number;
    /**
     * The name of the namespace to use for variables emitted by this action.
     *
     * @default - a name will be generated, based on the stage and action names,
     * if any of the action's variables were referenced - otherwise,
     * no namespace will be set
     * @stability stable
     */
    readonly variablesNamespace?: string;
}
/**
 * Common properties shared by all Actions whose {@link ActionProperties.owner} field is 'AWS' (or unset, as 'AWS' is the default).
 *
 * @stability stable
 */
export interface CommonAwsActionProps extends CommonActionProps {
    /**
     * The Role in which context's this Action will be executing in.
     *
     * The Pipeline's Role will assume this Role
     * (the required permissions for that will be granted automatically)
     * right before executing this Action.
     * This Action will be passed into your {@link IAction.bind}
     * method in the {@link ActionBindOptions.role} property.
     *
     * @default a new Role will be generated
     * @stability stable
     */
    readonly role?: iam.IRole;
}
/**
 * Low-level class for generic CodePipeline Actions implementing the {@link IAction} interface.
 *
 * Contains some common logic that can be re-used by all {@link IAction} implementations.
 * If you're writing your own Action class,
 * feel free to extend this class.
 *
 * @stability stable
 */
export declare abstract class Action implements IAction {
    /**
     * This is a renamed version of the {@link IAction.actionProperties} property.
     *
     * @stability stable
     */
    protected abstract readonly providedActionProperties: ActionProperties;
    private __actionProperties?;
    private __pipeline?;
    private __stage?;
    private __scope?;
    private readonly _namespaceToken;
    private _customerProvidedNamespace?;
    private _actualNamespace?;
    private _variableReferenced;
    /**
     * @stability stable
     */
    protected constructor();
    /**
     * The simple properties of the Action, like its Owner, name, etc.
     *
     * Note that this accessor will be called before the {@link bind} callback.
     *
     * @stability stable
     */
    get actionProperties(): ActionProperties;
    /**
     * The callback invoked when this Action is added to a Pipeline.
     *
     * @stability stable
     */
    bind(scope: Construct, stage: IStage, options: ActionBindOptions): ActionConfig;
    /**
     * Creates an Event that will be triggered whenever the state of this Action changes.
     *
     * @stability stable
     */
    onStateChange(name: string, target?: events.IRuleTarget, options?: events.RuleProps): events.Rule;
    /**
     * @stability stable
     */
    protected variableExpression(variableName: string): string;
    /**
     * This is a renamed version of the {@link IAction.bind} method.
     *
     * @stability stable
     */
    protected abstract bound(scope: Construct, stage: IStage, options: ActionBindOptions): ActionConfig;
    private get _pipeline();
    private get _stage();
    /**
     * Retrieves the Construct scope of this Action.
     * Only available after the Action has been added to a Stage,
     * and that Stage to a Pipeline.
     */
    private get _scope();
}
/**
 * The list of event types for AWS Codepipeline Pipeline.
 *
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-pipeline
 * @stability stable
 */
export declare enum PipelineNotificationEvents {
    /**
     * Trigger notification when pipeline execution failed.
     *
     * @stability stable
     */
    PIPELINE_EXECUTION_FAILED = "codepipeline-pipeline-pipeline-execution-failed",
    /**
     * Trigger notification when pipeline execution canceled.
     *
     * @stability stable
     */
    PIPELINE_EXECUTION_CANCELED = "codepipeline-pipeline-pipeline-execution-canceled",
    /**
     * Trigger notification when pipeline execution started.
     *
     * @stability stable
     */
    PIPELINE_EXECUTION_STARTED = "codepipeline-pipeline-pipeline-execution-started",
    /**
     * Trigger notification when pipeline execution resumed.
     *
     * @stability stable
     */
    PIPELINE_EXECUTION_RESUMED = "codepipeline-pipeline-pipeline-execution-resumed",
    /**
     * Trigger notification when pipeline execution succeeded.
     *
     * @stability stable
     */
    PIPELINE_EXECUTION_SUCCEEDED = "codepipeline-pipeline-pipeline-execution-succeeded",
    /**
     * Trigger notification when pipeline execution superseded.
     *
     * @stability stable
     */
    PIPELINE_EXECUTION_SUPERSEDED = "codepipeline-pipeline-pipeline-execution-superseded",
    /**
     * Trigger notification when pipeline stage execution started.
     *
     * @stability stable
     */
    STAGE_EXECUTION_STARTED = "codepipeline-pipeline-stage-execution-started",
    /**
     * Trigger notification when pipeline stage execution succeeded.
     *
     * @stability stable
     */
    STAGE_EXECUTION_SUCCEEDED = "codepipeline-pipeline-stage-execution-succeeded",
    /**
     * Trigger notification when pipeline stage execution resumed.
     *
     * @stability stable
     */
    STAGE_EXECUTION_RESUMED = "codepipeline-pipeline-stage-execution-resumed",
    /**
     * Trigger notification when pipeline stage execution canceled.
     *
     * @stability stable
     */
    STAGE_EXECUTION_CANCELED = "codepipeline-pipeline-stage-execution-canceled",
    /**
     * Trigger notification when pipeline stage execution failed.
     *
     * @stability stable
     */
    STAGE_EXECUTION_FAILED = "codepipeline-pipeline-stage-execution-failed",
    /**
     * Trigger notification when pipeline action execution succeeded.
     *
     * @stability stable
     */
    ACTION_EXECUTION_SUCCEEDED = "codepipeline-pipeline-action-execution-succeeded",
    /**
     * Trigger notification when pipeline action execution failed.
     *
     * @stability stable
     */
    ACTION_EXECUTION_FAILED = "codepipeline-pipeline-action-execution-failed",
    /**
     * Trigger notification when pipeline action execution canceled.
     *
     * @stability stable
     */
    ACTION_EXECUTION_CANCELED = "codepipeline-pipeline-action-execution-canceled",
    /**
     * Trigger notification when pipeline action execution started.
     *
     * @stability stable
     */
    ACTION_EXECUTION_STARTED = "codepipeline-pipeline-action-execution-started",
    /**
     * Trigger notification when pipeline manual approval failed.
     *
     * @stability stable
     */
    MANUAL_APPROVAL_FAILED = "codepipeline-pipeline-manual-approval-failed",
    /**
     * Trigger notification when pipeline manual approval needed.
     *
     * @stability stable
     */
    MANUAL_APPROVAL_NEEDED = "codepipeline-pipeline-manual-approval-needed",
    /**
     * Trigger notification when pipeline manual approval succeeded.
     *
     * @stability stable
     */
    MANUAL_APPROVAL_SUCCEEDED = "codepipeline-pipeline-manual-approval-succeeded"
}
