"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.validateProps = (props) => {
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error(`validateRuntimeConfig only accepts non-null RuntimeOptions.`);
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error(`RuntimeOptions.jsonFileName must be a json file.`);
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (!bucket.encryptionKey) {
                throw new Error("Website buckets must have encryption enabled!");
            }
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateProps(props);
        // S3 Bucket to hold website files
        this.websiteBucket = (_c = props.websiteBucket) !== null && _c !== void 0 ? _c : new aws_s3_1.Bucket(this, "WebsiteBucket", {
            versioned: true,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.KMS,
            encryptionKey: props.defaultWebsiteBucketEncryptionKey,
            serverAccessLogsPrefix: "access-logs",
        });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = (_d = distributionProps === null || distributionProps === void 0 ? void 0 : distributionProps.webAclId) !== null && _d !== void 0 ? _d : new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", {
            managedRules: [{ vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" }],
        }).webAclArn;
        // Cloudfront Distribution
        const defaultRootObject = (_e = distributionProps === null || distributionProps === void 0 ? void 0 : distributionProps.defaultRootObject) !== null && _e !== void 0 ? _e : "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            defaultBehavior: {
                ...distributionProps === null || distributionProps === void 0 ? void 0 : distributionProps.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: (_f = distributionProps === null || distributionProps === void 0 ? void 0 : distributionProps.errorResponses) !== null && _f !== void 0 ? _f : [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(((_g = props.runtimeOptions) === null || _g === void 0 ? void 0 : _g.jsonFileName) ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, (_h = props.runtimeOptions) === null || _h === void 0 ? void 0 : _h.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.0.84" };
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.0.84" };
//# sourceMappingURL=data:application/json;base64,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