# TimeXus

TimeXus is a Python library that provides enhanced time-related utilities, building upon the basic functionalities found in the standard library. It offers precise sleep functions, system information retrieval, and various date, time, and utility functions. It's designed to be easy to use and extend, making it suitable for a wide range of applications. **Note:** TimeXus does not use the `time` module for its core sleep functionality, but it does utilize `datetime` for some date and time-related features and `time` for `get_timezone`.

## Features

### Precise Sleep Functions:

These functions provide ways to pause the execution of your program for specific durations or random intervals.

*   **`xsleep(seconds)`**

    -   **Description:** A basic sleep function similar to `time.sleep()` but using a more precise, cross-platform mechanism. Pauses the program execution for the specified number of seconds.
    -   **How to use:**

        ```python
        from timexus import xsleep

        xsleep(2.5)  # Pauses execution for 2.5 seconds
        ```

    -   **Auto-print:** Enabled by default, printing a message indicating the sleep duration. Can be globally disabled using `XSleep.set_print("off")`.
    -   **Returns:** The number of seconds slept.

*   **`XSleep.milliseconds(ms)`**

    -   **Description:** Sleeps for the specified number of milliseconds.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.milliseconds(500)  # Pauses execution for 500 milliseconds (0.5 seconds)
        ```

    -   **Auto-print:** Enabled by default. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("milliseconds", "off")`.
    -   **Returns:** The number of seconds slept (ms / 1000).

*   **`XSleep.seconds(secs)`**

    -   **Description:** Sleeps for the specified number of seconds.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.seconds(3)  # Pauses execution for 3 seconds
        ```

    -   **Auto-print:** Enabled by default. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("seconds", "off")`.
    -   **Returns:** The number of seconds slept.

*   **`XSleep.minutes(mins)`**

    -   **Description:** Sleeps for the specified number of minutes.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.minutes(1.5)  # Pauses execution for 1.5 minutes (90 seconds)
        ```

    -   **Auto-print:** Enabled by default. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("minutes", "off")`.
    -   **Returns:** The number of seconds slept (mins * 60).

*   **`XSleep.hours(hrs)`**

    -   **Description:** Sleeps for the specified number of hours.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.hours(0.25)  # Pauses execution for 0.25 hours (15 minutes)
        ```

    -   **Auto-print:** Enabled by default. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("hours", "off")`.
    -   **Returns:** The number of seconds slept (hrs * 3600).

*   **`XSleep.get_random_delay(seconds)`**

    -   **Description:**  Calculates and returns a random delay (in seconds) up to the given `seconds` (integer). It does not sleep automatically, you need to use the returned value with `XSleep.seconds()` or `xsleep()` if you want to sleep for that duration.
    -   **How to use:**

        ```python
        from timexus import XSleep

        delay = XSleep.get_random_delay(5)  # Get a random delay up to 5 seconds
        print(f"Random delay: {delay} seconds") 
        XSleep.seconds(delay)  # Sleep for the random duration
        ```
    - **Auto-print:** Enabled by default, printing the generated random delay. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("get_random_delay", "off")`. You can also use `XSleep.sleep_random()` to sleep directly.
    - **Returns:** The random delay in seconds.

*   **`XSleep.sleep_random(seconds)`**

    -   **Description:** Sleeps for a random time up to the specified number of seconds (integer).
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.sleep_random(3) # Sleeps for a random time up to 3 seconds.
        ```

    -   **Auto-print:** Enabled by default. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("sleep_random", "off")`.
    - **Returns:** The random delay in seconds.

*   **`XSleep.set_print("on" | "off")`**

    -   **Description:** Globally enables or disables the auto-print messages for all sleep functions in the `XSleep` class.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.set_print("off")  # Disable all auto-print messages
        XSleep.seconds(1)  # No message will be printed
        XSleep.set_print("on")  # Re-enable all auto-print messages
        ```

*   **`XSleep.set_print_for_function(func_name, "on" | "off")`**
    -   **Description:** Enables or disables auto-print messages for a specific function within the `XSleep` class. This overrides the global setting.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.set_print_for_function("capture_screenshot", "off") # Disable auto-print for capture_screenshot
        XSleep.capture_screenshot() # No message will be printed

        XSleep.set_print_for_function("seconds", "off")
        XSleep.seconds(1) # No message will be printed

        XSleep.set_print_for_function("get_random_delay", "on")
        XSleep.get_random_delay(2) # Will print the random delay
        ```

*   **`XSleep.get_auto_print_status()`**
    -   **Description:** Prints the current auto-print status for each function and the global auto-print status.
    -   **How to use:**
        ```python
        from timexus import XSleep
        
        XSleep.get_auto_print_status()
        ```

*   **`XSleep.set_custom_message(message)`**

    -   **Description:** Customizes the message that is printed when you sleep for exactly 0 seconds.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.set_custom_message("I'm awake now!")
        XSleep.seconds(0)  # Prints "I'm awake now!"
        ```

*   **`XSleep.get_total_sleep_time()`**

    -   **Description:** Returns the total accumulated sleep time (in seconds) since the `XSleep` class was loaded or since the last call to `XSleep.reset_total_sleep_time()`.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.seconds(1)
        XSleep.milliseconds(500)
        total_sleep = XSleep.get_total_sleep_time()
        print(f"Total sleep time: {total_sleep} seconds")  # Output: Total sleep time: 1.5 seconds
        ```

*   **`XSleep.reset_total_sleep_time()`**

    -   **Description:** Resets the total sleep time counter to 0.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.seconds(2)
        XSleep.reset_total_sleep_time()
        total_sleep = XSleep.get_total_sleep_time()
        print(f"Total sleep time: {total_sleep} seconds")  # Output: Total sleep time: 0 seconds
        ```

### System Information:

These functions provide information about the system on which the script is running.

*   **`XSleep.get_pid()`**

    -   **Description:** Gets the current process ID.
    -   **How to use:**

        ```python
        from timexus import XSleep

        pid = XSleep.get_pid()
        print(f"Process ID: {pid}")
        ```

*   **`XSleep.get_cpu_count()`**

    -   **Description:** Gets the number of CPU cores (logical processors).
    -   **How to use:**

        ```python
        from timexus import XSleep

        cpu_count = XSleep.get_cpu_count()
        print(f"CPU Cores: {cpu_count}")
        ```

*   **`XSleep.is_interactive()`**

    -   **Description:** Checks if the current Python session is interactive (e.g., running in a terminal or an interactive interpreter).
    -   **How to use:**

        ```python
        from timexus import XSleep

        if XSleep.is_interactive():
            print("Running in an interactive session")
        else:
            print("Running in a non-interactive session (e.g., a script)")
        ```

*   **`XSleep.get_platform()`**

    -   **Description:** Returns a string identifying the operating system platform (e.g., "win32", "linux", "darwin").
    -   **How to use:**

        ```python
        from timexus import XSleep

        platform_name = XSleep.get_platform()
        print(f"Platform: {platform_name}")
        ```

*   **`XSleep.get_uptime()`**

    -   **Description:** Returns the system uptime in seconds (the time elapsed since the last boot).
    -   **How to use:**

        ```python
        from timexus import XSleep

        uptime = XSleep.get_uptime()
        print(f"Uptime: {uptime} seconds")
        ```

*   **`XSleep.get_free_memory()`**

    -   **Description:** Returns the amount of free memory available in bytes.
    -   **How to use:**

        ```python
        from timexus import XSleep

        free_memory = XSleep.get_free_memory()
        print(f"Free memory: {free_memory} bytes")
        ```

*   **`XSleep.get_username()`**

    -   **Description:** Returns the username of the current user.
    -   **How to use:**

        ```python
        from timexus import XSleep

        username = XSleep.get_username()
        print(f"Username: {username}")
        ```

*   **`XSleep.get_current_time_ms()`**

    -   **Description:** Returns the current time in milliseconds since the epoch (January 1, 1970, 00:00:00 UTC).
    -   **How to use:**

        ```python
        from timexus import XSleep

        current_time_ms = XSleep.get_current_time_ms()
        print(f"Current time (ms since epoch): {current_time_ms}")
        ```

*   **`XSleep.get_screen_resolution()`**

    -   **Description:** Returns the screen resolution as a string in the format "widthxheight" (e.g., "1920x1080").
    -   **How to use:**

        ```python
        from timexus import XSleep

        resolution = XSleep.get_screen_resolution()
        if resolution:
            print(f"Screen resolution: {resolution}")
        else:
            print("Could not determine screen resolution")
        ```

*   **`XSleep.get_system_info()`**

    -   **Description:** Returns a dictionary containing various system information, including:
        -   `platform`: Operating system platform (e.g., "Windows", "Linux", "Darwin").
        -   `release`: OS release (e.g., "10", "5.15.0-88-generic").
        -   `version`: OS version.
        -   `machine`: Machine architecture (e.g., "AMD64", "x86_64").
        -   `processor`: Processor information.
        -   `username`: Current username.
    -   **How to use:**

        ```python
        from timexus import XSleep

        system_info = XSleep.get_system_info()
        for key, value in system_info.items():
            print(f"{key}: {value}")
        ```

### Date and Time:

These functions provide utilities for working with dates and times.

*   **`XSleep.get_local_time()`**

    -   **Description:** Returns the current local time based on the system's timezone settings. The time is formatted as an ISO 8601 string (YYYY-MM-DDTHH:MM:SS).
    -   **How to use:**

        ```python
        from timexus import XSleep

        local_time = XSleep.get_local_time()
        print(f"Local time: {local_time}")
        ```

*   **`XSleep.time_since_or_until(date_str)`**

    -   **Description:** Calculates the time elapsed since a given date or the time until a future date. The input `date_str` should be in the format "m/d/y" or "m/d/yyyy". The output is a human-readable string that includes years, months, and days.
    -   **How to use:**

        ```python
        from timexus import XSleep

        # Time since a past date
        past_date = "1/1/2023"
        time_since = XSleep.time_since_or_until(past_date)
        print(f"Time since {past_date}: {time_since}")

        # Time until a future date
        future_date = "12/25/2024"
        time_until = XSleep.time_since_or_until(future_date)
        print(f"Time until {future_date}: {time_until}")
        ```

### Utility Functions:

These functions offer miscellaneous utilities that can be helpful in various scripts.

*   **`XSleep.execute_command(command)`**

    -   **Description:** Executes a shell command and returns the output as a string.
    -   **How to use:**

        ```python
        from timexus import XSleep

        output = XSleep.execute_command("ls -l")  # Example: List files in the current directory (Linux/macOS)
        # output = XSleep.execute_command("dir") # Example: List files in the current directory (Windows)
        print(output)
        ```

*   **`XSleep.print_message_with_delay(message, delay_seconds)`**

    -   **Description:** Prints a message to the console, with a delay (in seconds) between each character.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.print_message_with_delay("Hello, world!", 0.1)  # Prints "Hello, world!" with a 0.1-second delay between characters
        ```

    -   **Auto-print:** Enabled by default. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("print_message_with_delay", "off")`.

*   **`XSleep.generate_random_string(length)`**

    -   **Description:** Generates a random string of the specified length. The string contains uppercase and lowercase letters, digits, and punctuation.
    -   **How to use:**

        ```python
        from timexus import XSleep

        random_string = XSleep.generate_random_string(16)
        print(f"Random string: {random_string}")
        ```

*   **`XSleep.capture_screenshot(filename="screenshot.png")`**

    -   **Description:** Captures a screenshot and saves it to a file with the given filename. The default filename is "screenshot.png".
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.capture_screenshot("my_screenshot.png")  # Captures a screenshot and saves it as "my_screenshot.png"
        ```

    -   **Auto-print:** Enabled by default, printing a message indicating the file path where the screenshot was saved. You can disable it globally with `XSleep.set_print("off")` or for this specific function using `XSleep.set_auto_print_for_function("capture_screenshot", "off")`.

*   **`XSleep.get_clipboard_content()`**

    -   **Description:** Gets the current content of the system clipboard.
    -   **How to use:**

        ```python
        from timexus import XSleep

        clipboard_content = XSleep.get_clipboard_content()
        if clipboard_content:
            print(f"Clipboard content: {clipboard_content}")
        else:
            print("Clipboard is empty or could not be accessed")
        ```

*   **`XSleep.set_clipboard_content(data)`**

    -   **Description:** Sets the content of the system clipboard to the given string `data`.
    -   **How to use:**

        ```python
        from timexus import XSleep

        XSleep.set_clipboard_content("This text is now on the clipboard")
        ```

*   **`XSleep.schedule_task(func, delay_seconds)`**

    -   **Description:** Schedules a function to be executed after a specified delay.
    -   **How to use:**
        ```python
        from timexus import XSleep

        def greet():
            print("Hello, this was scheduled!")

        XSleep.schedule_task(greet, 5)  # Executes greet after 5 seconds.
        ```

*   **`XSleep.format_uptime(format_str="%H:%M:%S")`**

    -   **Description:** Returns the system uptime in a custom format.
    -   **How to use:**
        ```python
        from timexus import XSleep

        uptime_formatted = XSleep.format_uptime("%d days, %H:%M:%S")
        print(f"Uptime: {uptime_formatted}")
        ```

*   **`XSleep.get_timezone()`**

    -   **Description:** Gets information about the current timezone, including its UTC offset.
    -   **How to use:**

        ```python
        from timexus import XSleep
        
        timezone_info = XSleep.get_timezone()
        print(f"Timezone: {timezone_info['name']} (UTC{timezone_info['offset']})")
        ```

## Installation

**From PyPI:**

```bash
pip install TimeXus
```


# Bug Fixes (Version 0.3.9)
* get_random_delay() now returns the delay value: Previously, get_random_delay() only printed the random delay value but did not return it. It now correctly returns the generated random delay (as an integer), making it usable in calculations or further logic.
---
* set_print() provides full control over print messages: The set_print() function now reliably and globally disables or enables all print statements within the XSleep library, ensuring consistent behavior.
---
* "Awake!" message logic fixed: The "Awake!" message (or the custom message set by set_custom_message()) is now printed only when you sleep for exactly 0 seconds using any of the sleep functions (e.g., XSleep.seconds(0)).
---
* set_custom_message() always requires an argument: You must now always provide a message argument to set_custom_message(). To effectively disable the custom message, pass an empty string ("").
---
* Methods now return values: Functions like milliseconds(), seconds(), minutes(), hours(), and get_random_delay() now return relevant values (e.g., the number of seconds/milliseconds/minutes/hours slept or the random delay generated), making them significantly more useful.
---
* get_random_delay() and sleep_random() now only accept integers: To simplify usage and avoid potential floating-point inaccuracies, get_random_delay() and sleep_random() now only accept an integer as input, representing the maximum number of seconds for the random delay.
---
* More consistent and informative print messages: Improved the formatting and content of print messages to ensure they accurately reflect the units being used (seconds, milliseconds, microseconds) and are only displayed when both global and function-specific auto-print flags are enabled.
---
* Various internal bug fixes: Addressed potential issues on macOS (fallback path for libc.dylib), improved get_uptime() on macOS/Linux, corrected _get_caller_name() to work correctly when called indirectly, and enhanced error messages in various functions.
---
* XSleep.set_print("on" | "off"): Changed the mode argument from boolean (True / False) to string ("on" / "off").
---
* "on" globally enables all print messages within the XSleep class.
---
* "off" globally disables all print messages within the XSleep class.
---
* XSleep.set_print_for_function(func_name, "on" | "off"): Changed the mode argument from boolean (True / False) to string ("on" / "off").
---


# Resume

## Here's a quick summary of all the functions provided by the `XSleep` class in the `TimeXus` library:

| Function                                    | Description                                                                                                                                      |
| ------------------------------------------- | ------------------------------------------------------------------------------------------------------------------------------------------------ |
| `xsleep(seconds)`                           | Pauses execution for the specified number of seconds (more precise than `time.sleep()`).                                                           |
| `XSleep.milliseconds(ms)`                   | Sleeps for the specified number of milliseconds.                                                                                                 |
| `XSleep.seconds(secs)`                      | Sleeps for the specified number of seconds.                                                                                                      |
| `XSleep.minutes(mins)`                      | Sleeps for the specified number of minutes.                                                                                                      |
| `XSleep.hours(hrs)`                         | Sleeps for the specified number of hours.                                                                                                        |
| `XSleep.get_random_delay(seconds)`          | Returns a random delay (in seconds) up to the given `seconds` (integer).                                                                        |
| `XSleep.sleep_random(seconds)`              | Sleeps for a random time up to the specified number of seconds (integer).                                                                       |
| `XSleep.set_print("on" \| "off")`            | Globally enables or disables auto-print messages for all sleep functions.                                                                        |
| `XSleep.set_print_for_function(func_name, "on" \| "off")` | Enables or disables auto-print messages for a specific function.                                                                                   |
| `XSleep.get_auto_print_status()`             | Prints the current auto-print status for each function and the global status.                                                                     |
| `XSleep.set_custom_message(message)`         | Customizes the message printed when sleeping for exactly 0 seconds.                                                                              |
| `XSleep.get_total_sleep_time()`              | Returns the total accumulated sleep time (in seconds).                                                                                            |
| `XSleep.reset_total_sleep_time()`           | Resets the total sleep time counter to 0.                                                                                                        |
| `XSleep.get_pid()`                           | Gets the current process ID.                                                                                                                    |
| `XSleep.get_cpu_count()`                     | Gets the number of CPU cores.                                                                                                                   |
| `XSleep.is_interactive()`                    | Checks if the current session is interactive.                                                                                                    |
| `XSleep.get_platform()`                      | Returns a string identifying the operating system platform.                                                                                    |
| `XSleep.get_uptime()`                        | Returns the system uptime in seconds.                                                                                                           |
| `XSleep.get_free_memory()`                   | Returns the amount of free memory available in bytes.                                                                                             |
| `XSleep.get_username()`                      | Returns the username of the current user.                                                                                                       |
| `XSleep.get_current_time_ms()`               | Returns the current time in milliseconds since the epoch.                                                                                       |
| `XSleep.get_screen_resolution()`             | Returns the screen resolution (e.g., "1920x1080").                                                                                              |
| `XSleep.get_system_info()`                   | Returns a dictionary containing various system information.                                                                                      |
| `XSleep.get_local_time()`                    | Returns the current local time (ISO 8601 format).                                                                                                |
| `XSleep.time_since_or_until(date_str)`       | Calculates the time elapsed since or until a given date.                                                                                        |
| `XSleep.execute_command(command)`           | Executes a shell command and returns the output.                                                                                                |
| `XSleep.print_message_with_delay(message, delay_seconds)` | Prints a message with a delay between each character.                                                                                             |
| `XSleep.generate_random_string(length)`      | Generates a random string of the specified length.                                                                                              |
| `XSleep.capture_screenshot(filename="screenshot.png")` | Captures a screenshot and saves it to a file.                                                                                                 |
| `XSleep.get_clipboard_content()`             | Gets the current content of the system clipboard.                                                                                               |
| `XSleep.set_clipboard_content(data)`         | Sets the content of the system clipboard.                                                                                                       |
| `XSleep.schedule_task(func, delay_seconds)`  | Schedules a function to be executed after a specified delay in seconds.                                                                         |
| `XSleep.format_uptime(format_str="%H:%M:%S")` | Returns the system uptime formatted according to `format_str`.                                                                                  |
| `XSleep.get_timezone()`                      | Returns information about the current timezone, including its name and UTC offset.                                                                 |