"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
function validate(construct) {
    cdk.ConstructNode.prepare(construct.node);
    return cdk.ConstructNode.validate(construct.node);
}
module.exports = {
    'Artifacts in CodePipeline': {
        'cannot be created with an empty name'(test) {
            test.throws(() => new codepipeline.Artifact(''), /Artifact name must match regular expression/);
            test.done();
        },
        'without a name, when used as an input without being used as an output first - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: new codepipeline.Artifact(),
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Action 'Build' is using an unnamed input Artifact, which is not being produced in this pipeline");
            test.done();
        },
        'with a name, when used as an input without being used as an output first - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: new codepipeline.Artifact('named'),
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Action 'Build' is using input Artifact 'named', which is not being produced in this pipeline");
            test.done();
        },
        'without a name, when used as an output multiple times - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                output: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Both Actions 'Source' and 'Build' are producting Artifact 'Artifact_Source_Source'. Every artifact can only be produced once.");
            test.done();
        },
        "an Action's output can be used as input for an Action in the same Stage with a higher runOrder"(test) {
            const stack = new cdk.Stack();
            const sourceOutput1 = new codepipeline.Artifact('sourceOutput1');
            const buildOutput1 = new codepipeline.Artifact('buildOutput1');
            const sourceOutput2 = new codepipeline.Artifact('sourceOutput2');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput1,
                            }),
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source2',
                                output: sourceOutput2,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput1,
                                output: buildOutput1,
                            }),
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build2',
                                input: sourceOutput2,
                                extraInputs: [buildOutput1],
                                output: new codepipeline.Artifact('buildOutput2'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            //
            }));
            test.done();
        },
        'violation of runOrder constraints is detected and reported'(test) {
            const stack = new cdk.Stack();
            const sourceOutput1 = new codepipeline.Artifact('sourceOutput1');
            const buildOutput1 = new codepipeline.Artifact('buildOutput1');
            const sourceOutput2 = new codepipeline.Artifact('sourceOutput2');
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput1,
                            }),
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source2',
                                output: sourceOutput2,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput1,
                                output: buildOutput1,
                                runOrder: 3,
                            }),
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build2',
                                input: sourceOutput2,
                                extraInputs: [buildOutput1],
                                output: new codepipeline.Artifact('buildOutput2'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Stage 2 Action 2 ('Build'/'build2') is consuming input Artifact 'buildOutput1' before it is being produced at Stage 2 Action 3 ('Build'/'build1')");
            test.done();
        },
        'without a name, sanitize the auto stage-action derived name'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source.@',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source.@',
                        'Actions': [
                            {
                                'Name': 'source1',
                                'OutputArtifacts': [
                                    { 'Name': 'Artifact_Source_source1' },
                                ],
                            },
                        ],
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'build1',
                                'InputArtifacts': [
                                    { 'Name': 'Artifact_Source_source1' },
                                ],
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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