"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error("validateRuntimeConfig only accepts non-null RuntimeOptions.");
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error("RuntimeOptions.jsonFileName must be a json file.");
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.suppressCDKNagViolations = (props) => {
            const stack = aws_cdk_lib_1.Stack.of(this);
            !props.distributionProps?.certificate &&
                [
                    "AwsSolutions-CFR4",
                    "AwsPrototyping-CloudFrontDistributionHttpsViewerNoOutdatedSSL",
                ].forEach((RuleId) => {
                    cdk_nag_1.NagSuppressions.addResourceSuppressions(this.cloudFrontDistribution, [
                        {
                            id: RuleId,
                            reason: "Certificate is not mandatory therefore the Cloudfront certificate will be used.",
                        },
                    ]);
                });
            ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Latest runtime cannot be configured. CDK will need to upgrade the BucketDeployment construct accordingly.",
                    },
                ]);
            });
            ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "All Policies have been scoped to a Bucket. Given Buckets can contain arbitrary content, wildcard resources with bucket scope are required.",
                        appliesTo: [
                            {
                                regex: "/^Action::s3:.*$/g",
                            },
                            {
                                regex: `/^Resource::.*$/g`,
                            },
                        ],
                    },
                ]);
            });
            ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
                cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                    {
                        id: RuleId,
                        reason: "Buckets can contain arbitrary content, therefore wildcard resources under a bucket are required.",
                        appliesTo: [
                            {
                                regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                            },
                        ],
                    },
                ]);
            });
        };
        this.validateProps(props);
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    publicReadAccess: false,
                    blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                    serverAccessLogsPrefix: "access-logs",
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ??
            new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps).webAclArn;
        // Cloudfront Distribution
        const logBucket = props.distributionProps?.logBucket ||
            new aws_s3_1.Bucket(this, "DistributionLogBucket", {
                enforceSSL: true,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                publicReadAccess: false,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                serverAccessLogsPrefix: "access-logs",
            });
        const originAccessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, "OriginAccessIdentity");
        this.websiteBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [this.websiteBucket.bucketArn],
            actions: ["s3:ListBucket"],
            principals: [originAccessIdentity.grantPrincipal],
        }));
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            enableLogging: true,
            logBucket: logBucket,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket, {
                    originAccessIdentity,
                }),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
        this.suppressCDKNagViolations(props);
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.12.21" };
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.12.21" };
//# sourceMappingURL=data:application/json;base64,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