"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const handler_1 = require("../handler");
describe('WaitForStableServiceResource', () => {
    describe('doCreate', () => {
        let consoleLogMock;
        beforeEach(() => {
            aws_sdk_mock_1.setSDKInstance(AWS);
            AWS.config.region = 'us-east-1';
            consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
        });
        afterEach(() => {
            jest.clearAllMocks();
            aws_sdk_mock_1.restore('ECS');
        });
        test('success', async () => {
            // GIVEN
            const props = {
                cluster: 'clusterArn',
                services: ['serviceArn'],
            };
            aws_sdk_mock_1.mock('ECS', 'waitFor', (_state, _params, callback) => {
                callback(null, { status: 'ready' });
            });
            const handler = new handler_1.WaitForStableServiceResource(new AWS.ECS());
            // WHEN
            const result = await handler.doCreate('physicalId', props);
            // THEN
            expect(result).toBeUndefined();
            expect(consoleLogMock.mock.calls.length).toBe(2);
            expect(consoleLogMock.mock.calls[0][0]).toStrictEqual(`Waiting for ECS services to stabilize. Cluster: ${props.cluster}. Services: ${props.services[0]}`);
            expect(consoleLogMock.mock.calls[1][0]).toStrictEqual('Finished waiting. ECS services are stable.');
        });
        test('failure', async () => {
            // GIVEN
            const props = {
                cluster: 'clusterArn',
                services: ['serviceArn'],
            };
            aws_sdk_mock_1.mock('ECS', 'waitFor', (_state, _params, callback) => {
                callback({ code: 'errorcode', message: 'not stable' }, null);
            });
            const handler = new handler_1.WaitForStableServiceResource(new AWS.ECS());
            // WHEN
            const promise = handler.doCreate('physicalId', props);
            // THEN
            await expect(promise).rejects.toThrowError(/ECS services failed to stabilize in expected time:/);
        });
    });
    test('doDelete does not do anything', async () => {
        // GIVEN
        const props = {
            cluster: 'clusterArn',
            services: ['serviceArn'],
        };
        const handler = new handler_1.WaitForStableServiceResource(new AWS.ECS());
        // WHEN
        const promise = await handler.doDelete('physicalId', props);
        // THEN
        await expect(promise).toBeUndefined();
    });
    describe('.validateInput()', () => {
        test('returns true with valid input', async () => {
            // GIVEN
            const validInput = {
                cluster: 'clusterArn',
                services: ['serviceArn'],
                forceRun: '',
            };
            // WHEN
            const handler = new handler_1.WaitForStableServiceResource(new AWS.ECS());
            const returnValue = handler.validateInput(validInput);
            // THEN
            expect(returnValue).toBeTruthy();
        });
        const noCluster = {
            services: [''],
        };
        const clusterNotString = {
            services: [''],
            cluster: 10,
        };
        const noServices = {
            cluster: '',
        };
        const servicesNotArray = {
            cluster: '',
            services: '',
        };
        const servicesNotArrayOfStrings = {
            cluster: '',
            services: [10],
        };
        const forceRunNotString = {
            cluster: '',
            services: [''],
            forceRun: true,
        };
        test.each([
            [],
            '',
            noCluster,
            clusterNotString,
            noServices,
            servicesNotArray,
            servicesNotArrayOfStrings,
            forceRunNotString,
        ])('returns false with invalid input %p', async (invalidInput) => {
            // WHEN
            const handler = new handler_1.WaitForStableServiceResource(new AWS.ECS());
            const returnValue = handler.validateInput(invalidInput);
            // THEN
            expect(returnValue).toBeFalsy();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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