"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const util_1 = require("util");
const handlers_1 = require("../handlers");
// Prevent log from printing during the test
console.log = jest.fn();
jest.setTimeout(20000); // 20s timeout, to give time to files.
async function recursiveDeleteDirectory(location) {
    if (!location)
        return;
    const contents = await fs_1.promises.readdir(location);
    const stats = await Promise.all(contents.map(async (loc) => {
        return await fs_1.promises.stat(path_1.join(location, loc));
    }));
    const files = contents.filter((_, i) => stats[i].isFile());
    const directories = contents.filter((_, i) => stats[i].isDirectory());
    await Promise.all(files.map(async (loc) => fs_1.promises.unlink(path_1.join(location, loc))));
    await Promise.all(directories.map(async (loc) => recursiveDeleteDirectory(path_1.join(location, loc))));
    await fs_1.promises.rmdir(location);
}
describe('Testing filesystem modifications', () => {
    var tempDirectory;
    beforeEach(async () => {
        // Create a temp directory for putting files.
        tempDirectory = await fs_1.promises.mkdtemp(path_1.join(os_1.tmpdir(), 'tmp.'));
    });
    afterEach(async () => {
        await recursiveDeleteDirectory(tempDirectory);
        tempDirectory = '';
    });
    test('Add to empty directory', async () => {
        // WHEN
        // Add 5 64 MiB files to the temp directory.
        await handlers_1.growFilesystem(5, 64, tempDirectory);
        // THEN
        const dirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        expect(dirContents).toEqual(['00000', '00001', '00002', '00003', '00004']);
        for (var file of dirContents) {
            const stat = await fs_1.promises.stat(path_1.join(tempDirectory, file));
            expect(stat.size).toBe(67108864);
        }
    });
    test('Append to directory', async () => {
        // GIVEN
        for (var i = 4; i < 8; i++) {
            const filename = path_1.join(tempDirectory, i.toString());
            await fs_1.promises.writeFile(filename, 'Some data');
        }
        // WHEN
        // Add 2 64 MiB files to the temp directory.
        await handlers_1.growFilesystem(2, 64, tempDirectory);
        // THEN
        // Make sure that the files that we added started numbering at 8
        const dirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        expect(dirContents).toEqual(['00008', '00009', '4', '5', '6', '7']);
    });
    test('Delete from directory', async () => {
        // GIVEN
        for (var i = 1; i < 8; i++) {
            const filename = path_1.join(tempDirectory, i.toString().padStart(3, '0'));
            await fs_1.promises.writeFile(filename, 'Some data');
        }
        const preDirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        // WHEN
        // Remove two files from the filesystem
        await handlers_1.shrinkFilesystem(2, tempDirectory);
        // THEN
        const dirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        expect(preDirContents).toEqual(['001', '002', '003', '004', '005', '006', '007']);
        expect(dirContents).toEqual(['001', '002', '003', '004', '005']);
    });
});
describe('Testing getDiskUsage behavior', () => {
    var tempDirectory;
    beforeEach(async () => {
        // Create a temp directory for putting files.
        tempDirectory = await fs_1.promises.mkdtemp(path_1.join(os_1.tmpdir(), 'tmp.'));
    });
    afterEach(async () => {
        await recursiveDeleteDirectory(tempDirectory);
        tempDirectory = '';
    });
    test('Throws when no mountPoint', async () => {
        await expect(() => handlers_1.getDiskUsage({}, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Throws mountPoint does not exist', async () => {
        await expect(() => handlers_1.getDiskUsage({
            mountPoint: path_1.join(tempDirectory, 'does_not_exist'),
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Throws when mountPoint not a directory', async () => {
        // WHEN
        const filename = path_1.join(tempDirectory, '001');
        await fs_1.promises.writeFile(filename, 'Some data');
        // THEN
        await expect(() => handlers_1.getDiskUsage({
            mountPoint: filename,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Correctly calculates disk usage', async () => {
        // GIVEN
        // This overrides the default padding file size to 64 MiB from 1024 MiB. Keep this in mind when interpreting the test.
        // All of the interface points are phrased in terms of 1 GiB files, but this little hack changes the semantics of those
        // to be phrased in terms of 64 MiB files.
        handlers_1.setDefaultFilesize(64);
        const execPromise = util_1.promisify(child_process_1.exec);
        await execPromise(`/usr/bin/dd if=/dev/zero of=${path_1.join(tempDirectory, 'file1.tmp')} bs=32M count=2`);
        await fs_1.promises.mkdir(path_1.join(tempDirectory, 'subdir'));
        await execPromise(`/usr/bin/dd if=/dev/zero of=${path_1.join(tempDirectory, 'subdir', 'file2.tmp')} bs=32M count=2`);
        // WHEN
        const usage = await handlers_1.getDiskUsage({
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        });
        // THEN
        expect(usage).toBe(2);
    });
});
describe('Testing padFilesystem macro behavior', () => {
    var tempDirectory;
    beforeEach(async () => {
        // Create a temp directory for putting files.
        tempDirectory = await fs_1.promises.mkdtemp(path_1.join(os_1.tmpdir(), 'tmp.'));
    });
    afterEach(async () => {
        await recursiveDeleteDirectory(tempDirectory);
        tempDirectory = '';
    });
    test('Throws when no desiredPadding', async () => {
        await expect(() => handlers_1.padFilesystem({
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Throws desiredPadding not number', async () => {
        await expect(() => handlers_1.padFilesystem({
            desiredPadding: 'one hundred',
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Throws when no mountPoint', async () => {
        await expect(() => handlers_1.padFilesystem({
            desiredPadding: '2',
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Throws mountPoint does not exist', async () => {
        await expect(() => handlers_1.padFilesystem({
            desiredPadding: '2',
            mountPoint: path_1.join(tempDirectory, 'does_not_exist'),
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Throws when mountPoint not a directory', async () => {
        // WHEN
        const filename = path_1.join(tempDirectory, '001');
        await fs_1.promises.writeFile(filename, 'Some data');
        // THEN
        await expect(() => handlers_1.padFilesystem({
            desiredPadding: '2',
            mountPoint: filename,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        })).rejects.toThrow();
    });
    test('Adds file if needed', async () => {
        // GIVEN
        // Empty directory: tempDirectory
        // This overrides the default padding file size to 64 MiB from 1024 MiB. Keep this in mind when interpreting the test.
        // All of the interface points are phrased in terms of 1 GiB files, but this little hack changes the semantics of those
        // to be phrased in terms of 64 MiB files.
        handlers_1.setDefaultFilesize(64);
        // WHEN
        await handlers_1.padFilesystem({
            desiredPadding: '1',
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        });
        // THEN
        const dirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        expect(dirContents).toEqual(['00000']);
        for (var file of dirContents) {
            const stat = await fs_1.promises.stat(path_1.join(tempDirectory, file));
            expect(stat.size).toBe(67108864);
        }
    });
    test('Removes file if needed', async () => {
        // GIVEN
        // This overrides the default padding file size to 64 MiB from 1024 MiB. Keep this in mind when interpreting the test.
        // All of the interface points are phrased in terms of 1 GiB files, but this little hack changes the semantics of those
        // to be phrased in terms of 64 MiB files.
        handlers_1.setDefaultFilesize(64);
        // tempDirectory with 2 64 MiB files in it
        await handlers_1.padFilesystem({
            desiredPadding: '2',
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        });
        // WHEN
        const preDirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        // Desire to shrink down to 1 file
        await handlers_1.padFilesystem({
            desiredPadding: '1',
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        });
        // THEN
        const dirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        expect(preDirContents).toEqual(['00000', '00001']);
        expect(dirContents).toEqual(['00000']);
        for (var file of dirContents) {
            const stat = await fs_1.promises.stat(path_1.join(tempDirectory, file));
            expect(stat.size).toBe(67108864);
        }
    });
    test('No change to filesystem', async () => {
        // GIVEN
        // This overrides the default padding file size to 64 MiB from 1024 MiB. Keep this in mind when interpreting the test.
        // All of the interface points are phrased in terms of 1 GiB files, but this little hack changes the semantics of those
        // to be phrased in terms of 64 MiB files.
        handlers_1.setDefaultFilesize(64);
        // tempDirectory with a 64 MiB file in it
        await handlers_1.padFilesystem({
            desiredPadding: '1',
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        });
        // WHEN
        const preDirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        // Desire for 64 MiB of files
        await handlers_1.padFilesystem({
            desiredPadding: '1',
            mountPoint: tempDirectory,
        }, {
            logGroupName: '',
            logStreamName: '',
            getRemainingTimeInMillis: () => 1000,
        });
        // THEN
        const dirContents = (await fs_1.promises.readdir(tempDirectory)).sort();
        expect(preDirContents).toEqual(['00000']);
        expect(dirContents).toEqual(preDirContents);
        for (var file of dirContents) {
            const stat = await fs_1.promises.stat(path_1.join(tempDirectory, file));
            expect(stat.size).toBe(67108864);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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