/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * This module contains an engine for simple CfnCustomResources.
 * Simple, in this context, is a CfnCustomResource for which:
 * 1. The 'Update' is only performed if the ResourceProperties of the 'Update'
 * differs from the OldResourceProperties;
 * 2. The PhysicalResourceId is always the sha256 hash of the ResourceProperties;
 * 3. The 'Update' is performed by doing a 'Create' on the ResourceProperties,
 * and letting a subsequent 'Delete' on the OldResourceProperties clean up its
 * resources.
 */
import { LambdaContext } from '../aws-lambda';
import { CfnRequestEvent } from './types';
export declare abstract class SimpleCustomResource {
    protected readonly debugMode: boolean;
    constructor();
    /**
     * Called by the handler on the given ResourceProperties to ensure that
     * we set up the CfnCustomResource input correctly.
     * @param data
     * @returns True if the given data is correctly formed.
     */
    abstract validateInput(data: object): boolean;
    /**
     * Called to perform the 'Create' action. Either in response to a 'Create'
     * request, or a 'Update' request wherein the ResourceProperties & OldResourceProperties
     * differ.
     * @param physicalId A stable hash value derived from the value of ResourceProperties
     * @param resourceProperties The ResourceProperties given to the handler.
     * @returns The Data to send back to CloudFormation as attributes of this CfnCustomResource
     */
    abstract doCreate(physicalId: string, resourceProperties: object): Promise<object | undefined>;
    /**
     * Called to perform the 'Delete' action. There are three locations in the state-diagram
     * of CloudFormation where we will recieve a 'Delete' request.
     * 1. Normally, when the stack is being deleted.
     * 2. On the replaced OldResourceProperties if an 'Update' request changed the PhysicalResourceId
     *    of the resource, and the stack update was successful.
     * 3. On the new ResourceProperties if an 'Update' request changed the PhysicalResourceId
     *    of the resource, and the stack update was NOT successful. i.e. rollback
     * @param physicalId A stable hash value derived from the value of ResourceProperties
     * @param resourceProperties The ResourceProperties given to the handler.
     */
    abstract doDelete(physicalId: string, resourceProperties: object): Promise<void>;
    /**
     * Handler/engine for the CustomResource state machine. Users of this class should
     * instantiate the class, and then immediately call this function.
     * @param event The event passed to the lambda handler.
     * @param context The lambda context passed to the lambda handler.
     */
    handler(event: CfnRequestEvent, context: LambdaContext): Promise<string>;
    private handleEvent;
}
